# データベース設計書

## 概要

本ドキュメントは、Apache Spark プロジェクトにおけるカタログメタデータのデータモデル設計を記述するものである。Apache Spark は分散データ処理フレームワークであり、従来のアプリケーションのようなリレーショナルデータベースを直接保持しない。ただし、テーブル・データベース・関数・パーティションなどのメタデータを管理するための「カタログシステム」を内部に持ち、これが事実上のデータベーススキーマとして機能する。

カタログシステムには以下の2つの実装が存在する:

1. **InMemoryCatalog** -- テスト・探索用のインメモリ実装（`sql/catalyst`モジュール）
2. **HiveExternalCatalog** -- Hive Metastore を利用した永続的な実装（`sql/hive`モジュール）

これらは共通の `ExternalCatalog` トレイトを実装し、同一のデータモデル（`CatalogDatabase`、`CatalogTable`、`CatalogFunction`、`CatalogTablePartition`等）を使用する。

本設計書では、このカタログメタデータのエンティティ構造をテーブル定義として記述する。

## テーブル一覧

| テーブル名 | 対応エンティティ | 説明 |
| --- | --- | --- |
| CatalogDatabase | CatalogDatabase | データベース（名前空間）のメタデータ |
| CatalogTable | CatalogTable | テーブルのメタデータ定義 |
| CatalogTablePartition | CatalogTablePartition | テーブルパーティション（Hiveスタイル）のメタデータ |
| CatalogStorageFormat | CatalogStorageFormat | テーブル・パーティションのストレージ形式情報 |
| CatalogFunction | CatalogFunction | ユーザー定義関数（UDF）のメタデータ |
| CatalogStatistics | CatalogStatistics | テーブル統計情報 |
| CatalogColumnStat | CatalogColumnStat | カラム単位の統計情報 |
| BucketSpec | BucketSpec | バケッティング情報 |
| ClusterBySpec | ClusterBySpec | クラスタリング情報 |
| CatalogTableType | CatalogTableType | テーブル種別（EXTERNAL/MANAGED/VIEW） |
| FunctionResource | FunctionResource | 関数のリソース情報 |

## 各テーブル定義

### 1. CatalogDatabase

データベース（名前空間）のメタデータを表現するエンティティ。Sparkにおけるデータベースは、テーブルと関数をグループ化する名前空間として機能する。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| name | String | NOT NULL | データベース名（一意キー） |
| description | String | NOT NULL | データベースの説明 |
| locationUri | URI | NOT NULL | データベースのデフォルトロケーション（ファイルシステムパス） |
| properties | Map[String, String] | NOT NULL | データベースプロパティ（キーバリューペア） |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L1061-1065)

### 2. CatalogTable

テーブルのメタデータを表現する中心的なエンティティ。EXTERNAL/MANAGED/VIEWの3種別のテーブルを管理する。Hive Metastore とのやりとりにも使用される。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| identifier | TableIdentifier | NOT NULL | テーブル識別子（catalog, database, table名を含む） |
| tableType | CatalogTableType | NOT NULL | テーブル種別（EXTERNAL/MANAGED/VIEW） |
| storage | CatalogStorageFormat | NOT NULL | ストレージ形式情報 |
| schema | StructType | NOT NULL | テーブルスキーマ（カラム定義） |
| provider | Option[String] | NULL許可 | データソースプロバイダー名（例: parquet, json, hive） |
| partitionColumnNames | Seq[String] | NOT NULL | パーティションカラム名リスト |
| bucketSpec | Option[BucketSpec] | NULL許可 | バケッティング情報 |
| owner | String | NOT NULL | テーブル所有者（デフォルト: 現在のユーザー） |
| createTime | Long | NOT NULL | テーブル作成時刻（ミリ秒） |
| lastAccessTime | Long | NOT NULL | 最終アクセス時刻（ミリ秒、-1は未知） |
| createVersion | String | NOT NULL | テーブルを作成したSparkバージョン |
| properties | Map[String, String] | NOT NULL | テーブルプロパティ |
| stats | Option[CatalogStatistics] | NULL許可 | テーブル統計情報 |
| viewText | Option[String] | NULL許可 | ビュー定義のSQLテキスト |
| comment | Option[String] | NULL許可 | テーブルコメント |
| collation | Option[String] | NULL許可 | テーブルの照合順序 |
| unsupportedFeatures | Seq[String] | NOT NULL | Spark SQLで未サポートの機能リスト |
| tracksPartitionsInCatalog | Boolean | NOT NULL | パーティション情報をカタログで追跡するか |
| schemaPreservesCase | Boolean | NOT NULL | スキーマが大文字小文字を保持するか |
| ignoredProperties | Map[String, String] | NOT NULL | Sparkが無視するテーブルプロパティ |
| viewOriginalText | Option[String] | NULL許可 | ビューの元のSQLテキスト |
| streamingSourceIdentifyingName | Option[StreamingSourceIdentifyingName] | NULL許可 | ストリーミングソース識別名 |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L425-448)

### 3. CatalogTablePartition

Hiveスタイルのパーティションメタデータを表現するエンティティ。テーブルのパーティションごとのストレージ情報やパラメータを管理する。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| spec | Map[String, String] | NOT NULL | パーティションスペック（カラム名と値のマッピング） |
| storage | CatalogStorageFormat | NOT NULL | パーティションのストレージ形式 |
| parameters | Map[String, String] | NOT NULL | パーティションパラメータ（デフォルト: 空Map） |
| createTime | Long | NOT NULL | パーティション作成時刻（ミリ秒） |
| lastAccessTime | Long | NOT NULL | 最終アクセス時刻（ミリ秒、-1は未知） |
| stats | Option[CatalogStatistics] | NULL許可 | パーティション統計情報 |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L196-202)

### 4. CatalogStorageFormat

テーブルまたはパーティションのストレージ形式を表現するエンティティ。入出力フォーマットやSerde（シリアライゼーション/デシリアライゼーション）情報を含む。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| locationUri | Option[URI] | NULL許可 | データの物理的な格納場所 |
| inputFormat | Option[String] | NULL許可 | 入力フォーマットクラス名 |
| outputFormat | Option[String] | NULL許可 | 出力フォーマットクラス名 |
| serde | Option[String] | NULL許可 | Serdeクラス名 |
| compressed | Boolean | NOT NULL | 圧縮の有無 |
| properties | Map[String, String] | NOT NULL | ストレージプロパティ |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L144-150)

### 5. CatalogFunction

カタログに定義された関数のメタデータを表現するエンティティ。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| identifier | FunctionIdentifier | NOT NULL | 関数識別子（database + funcName） |
| className | String | NOT NULL | 完全修飾クラス名（例: "org.apache.spark.util.MyFunc"） |
| resources | Seq[FunctionResource] | NOT NULL | 関数が使用するリソースのリスト |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L133-138)

### 6. CatalogStatistics

テーブルの統計情報を表現するエンティティ。メタストアとのやりとりに使用される。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| sizeInBytes | BigInt | NOT NULL | テーブルの合計サイズ（バイト） |
| rowCount | Option[BigInt] | NULL許可 | 行数 |
| colStats | Map[String, CatalogColumnStat] | NOT NULL | カラム名をキーとするカラム統計情報マップ |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L841-844)

### 7. CatalogColumnStat

個々のカラムの統計情報を表現するエンティティ。クエリ最適化のためのCBO（Cost-Based Optimization）に活用される。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| distinctCount | Option[BigInt] | NULL許可 | ユニーク値の数 |
| min | Option[String] | NULL許可 | 最小値（文字列表現） |
| max | Option[String] | NULL許可 | 最大値（文字列表現） |
| nullCount | Option[BigInt] | NULL許可 | NULL値の数 |
| avgLen | Option[Long] | NULL許可 | 平均長 |
| maxLen | Option[Long] | NULL許可 | 最大長 |
| histogram | Option[Histogram] | NULL許可 | ヒストグラム情報 |
| version | Int | NOT NULL | 統計情報のバージョン（現在: 2） |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L882-890)

### 8. BucketSpec

バケッティング情報を表現するエンティティ。データセットをより管理しやすい部分に分解するための技術で、バケット数は固定である。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| numBuckets | Int | NOT NULL | バケット数（1以上、bucketingMaxBuckets以下） |
| bucketColumnNames | Seq[String] | NOT NULL | バケットIDの生成に使用するカラム名リスト |
| sortColumnNames | Seq[String] | NOT NULL | 各バケット内のソートに使用するカラム名リスト |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L372-375)

### 9. ClusterBySpec

テーブルのクラスタリング情報を表現するエンティティ。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| columnNames | Seq[NamedReference] | NOT NULL | クラスタリングに使用するカラム名リスト |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L273)

### 10. CatalogTableType

テーブルの種別を表現する列挙型エンティティ。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| name | String | NOT NULL | テーブル種別名（"EXTERNAL", "MANAGED", "VIEW"のいずれか） |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/interface.scala` (L1048-1055)

### 11. FunctionResource

関数が必要とするリソース情報を表現するエンティティ。

| カラム名 | データ型 | NULL | 説明 |
| --- | --- | --- | --- |
| resourceType | FunctionResourceType | NOT NULL | リソース種別（jar, file, archive） |
| uri | String | NOT NULL | リソースのURI |

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/functionResources.scala` (L49)

## Hive Metastore プロパティプレフィックス

HiveExternalCatalog では、テーブルプロパティにSparkのメタデータを格納するために以下のプレフィックスが使用される:

| プレフィックス | 説明 |
| --- | --- |
| `spark.sql.` | Spark SQLプロパティの基本プレフィックス |
| `spark.sql.sources.` | データソース関連プロパティ |
| `spark.sql.sources.provider` | データソースプロバイダー名 |
| `spark.sql.sources.schema` | スキーマ情報 |
| `spark.sql.sources.schema.numPartCols` | パーティションカラム数 |
| `spark.sql.sources.schema.numSortCols` | ソートカラム数 |
| `spark.sql.sources.schema.numBuckets` | バケット数 |
| `spark.sql.sources.schema.numBucketCols` | バケットカラム数 |
| `spark.sql.statistics.` | 統計情報プレフィックス |
| `spark.sql.statistics.totalSize` | テーブルの合計サイズ |
| `spark.sql.statistics.numRows` | 行数 |
| `spark.sql.statistics.colStats.` | カラム統計プレフィックス |
| `spark.sql.partitionProvider` | パーティション管理方式（catalog/filesystem） |
| `spark.sql.create.version` | テーブル作成Sparkバージョン |
| `view.` | ビュー関連プロパティのプレフィックス |

定義元: `sql/hive/src/main/scala/org/apache/spark/sql/hive/HiveExternalCatalog.scala` (L1446-1471)

## InMemoryCatalog 内部構造

InMemoryCatalog はテスト・探索目的のインメモリ実装で、以下のネストされたHashMap構造でメタデータを保持する:

```
catalog: HashMap[String, DatabaseDesc]
  └── DatabaseDesc
        ├── db: CatalogDatabase
        ├── tables: HashMap[String, TableDesc]
        │     └── TableDesc
        │           ├── table: CatalogTable
        │           └── partitions: HashMap[TablePartitionSpec, CatalogTablePartition]
        └── functions: HashMap[String, CatalogFunction]
```

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/InMemoryCatalog.scala` (L47-64)

## ExternalCatalog インターフェース

ExternalCatalog トレイトは、カタログシステムの統一インターフェースを提供する。以下の操作カテゴリを定義する:

- **Databases**: createDatabase, dropDatabase, alterDatabase, getDatabase, databaseExists, listDatabases
- **Tables**: createTable, dropTable, renameTable, alterTable, alterTableDataSchema, alterTableSchema, alterTableStats, getTable, tableExists, listTables, listViews, loadTable, loadPartition, loadDynamicPartitions
- **Partitions**: createPartitions, dropPartitions, renamePartitions, alterPartitions, getPartition, listPartitionNames, listPartitions, listPartitionsByFilter
- **Functions**: createFunction, dropFunction, alterFunction, renameFunction, getFunction, functionExists, listFunctions

定義元: `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalog.scala`

## 備考

- Apache Sparkは分散データ処理フレームワークであり、従来のRDBMSのようなテーブル定義を直接持たない。本設計書で記述するエンティティは、Sparkのカタログシステムが内部的に管理するメタデータ構造である。
- HiveExternalCatalog 実装では、これらのメタデータは Hive Metastore（通常はDerby/MySQL/PostgreSQLなどのRDBMS）に永続化される。
- InMemoryCatalog 実装では、すべてのメタデータはJVMヒープ上のHashMapに保持され、セッション終了時に失われる。
- ビュー（VIEW）はCatalogTableの特殊ケースとして管理され、`view.`プレフィックスのプロパティにビュー固有の情報を保持する。
- テーブル統計情報（CatalogStatistics, CatalogColumnStat）は、Sparkのコストベースオプティマイザ（CBO）によるクエリ最適化に使用される。
