# 機能設計書 48-ナイーブベイズ

## 概要

本ドキュメントは、Apache Spark MLlibにおけるナイーブベイズ（Naive Bayes）分類機能の設計を記述する。ナイーブベイズアルゴリズムによる分類モデルの訓練と予測を提供する。

### 本機能の処理概要

ナイーブベイズは、ベイズの定理と特徴量の条件付き独立性仮定に基づく確率的分類アルゴリズムであり、4種類のモデル（Multinomial、Complement、Bernoulli、Gaussian）を提供する。

**業務上の目的・背景**：ナイーブベイズは計算効率が非常に高く、大規模データに対しても高速に学習できる。特にテキスト分類（スパムフィルタリング、文書カテゴリ分類）で広く使用されるベースラインモデルである。少量のトレーニングデータでも比較的良好な性能を発揮する。

**機能の利用シーン**：テキスト分類（TF-IDFベクトルに対するMultinomial NB）、文書分類（二値特徴量に対するBernoulli NB）、連続特徴量に対するGaussian NB等で使用される。

**主要な処理内容**：
1. モデルタイプの選択（Multinomial / Complement / Bernoulli / Gaussian）
2. クラスごとの事前確率と特徴量の条件付き確率の推定
3. スムージングパラメータによる零頻度問題の対処
4. ベイズの定理に基づく事後確率の計算と分類

**関連システム・外部連携**：Pipeline API、特徴量変換器（Tokenizer、TF-IDF等）。

**権限による制御**：特段のロール制御はない。

## 関連画面

本機能に直接関連する画面はない。

## 機能種別

計算処理 / 機械学習分類

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| featuresCol | String | Yes | 特徴量カラム名 | Vector型カラム |
| labelCol | String | Yes | ラベルカラム名 | 非負整数 |
| weightCol | String | No | 重みカラム名 | 非負の数値 |
| smoothing | Double | No | スムージングパラメータ、デフォルト1.0 | 0以上 |
| modelType | String | No | モデルタイプ、デフォルト"multinomial" | "multinomial"/"complement"/"bernoulli"/"gaussian" |

### 入力データソース

MLlibのDataFrame/Datasetとして提供されるラベル付きトレーニングデータ。Multinomial/Bernoulli/Complementモデルでは特徴量値は非負である必要がある。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| NaiveBayesModel | NaiveBayesModel | 学習済みモデル（pi: 事前確率対数、theta: 条件付き確率対数） |
| prediction | Double | 予測クラスラベル |
| rawPrediction | Vector | 各クラスの対数事後確率 |
| probability | Vector | 各クラスの予測確率 |

### 出力先

メモリ内のNaiveBayesModelオブジェクト、またはMLWriter経由で永続化。

## 処理フロー

### 処理シーケンス

```
1. train()メソッド
   └─ trainWithLabelCheck(dataset, nonNegativeLabel=true)
   └─ Summarizerによるクラスごとの統計量集約
   └─ モデルタイプに応じたパラメータ推定:
       ├─ Multinomial: log(count_k + smoothing) - log(total + smoothing * numFeatures)
       ├─ Complement: 補集合の統計量を使用
       ├─ Bernoulli: 特徴量の出現/非出現確率
       └─ Gaussian: 平均と分散の推定
   └─ NaiveBayesModel構築

2. predictRaw(features)
   └─ モデルタイプに応じた対数尤度計算
   └─ pi（事前確率対数）+ theta * features（条件付き確率対数の合計）
```

### フローチャート

```mermaid
flowchart TD
    A[train開始] --> B[ラベルチェック]
    B --> C[Summarizerで統計量集約]
    C --> D{modelType}
    D -->|multinomial| E[Multinomialパラメータ推定]
    D -->|complement| F[Complementパラメータ推定]
    D -->|bernoulli| G[Bernoulliパラメータ推定]
    D -->|gaussian| H[Gaussianパラメータ推定]
    E --> I[NaiveBayesModel生成]
    F --> I
    G --> I
    H --> I
    I --> J[predictRaw: 対数事後確率計算]
    J --> K[probability: ソフトマックス正規化]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-48-01 | 非負特徴量要件 | Multinomial/Bernoulli/Complement NBでは特徴量値は非負 | modelType指定時 |
| BR-48-02 | スムージング | ラプラススムージング（デフォルト1.0）で零頻度問題を回避 | 常時 |
| BR-48-03 | Complementモデル | 各クラスの補集合統計量を使用して係数推定 | modelType="complement" |
| BR-48-04 | Gaussianモデル | 連続特徴量に対してガウス分布を仮定 | modelType="gaussian" |

### 計算ロジック

- Multinomial NB: `log P(class=k|x) = log pi_k + sum_i(x_i * log theta_{k,i})`
- Gaussian NB: `log P(class=k|x) = log pi_k + sum_i(-0.5 * log(2*pi*sigma_{k,i}^2) - (x_i - mu_{k,i})^2 / (2*sigma_{k,i}^2))`

## データベース操作仕様

本機能はデータベースへの直接操作は行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| IllegalArgumentException | バリデーションエラー | 無効なmodelType | サポートされたモデルタイプを設定 |

### リトライ仕様

特になし（閉形式解のため反復最適化は不要）。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- ナイーブベイズは閉形式解を持つため、反復最適化が不要で高速
- Summarizerによる効率的な統計量集約

## セキュリティ考慮事項

モデル永続化時のアクセス制御を適用する必要がある。

## 備考

- Spark 3.0.0以降、Complement NBとGaussian NBが追加された
- mllib旧APIとの互換性のため`trainWithLabelCheck`メソッドが存在する

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | NaiveBayes.scala | `mllib/src/main/scala/org/apache/spark/ml/classification/NaiveBayes.scala` | NaiveBayesParamsのパラメータ定義（44-72行目）、特にsmoothingとmodelTypeを理解する |

**読解のコツ**: modelTypeはString型パラメータで、4つの値（"multinomial", "complement", "bernoulli", "gaussian"）を受け付ける。各モデルタイプにより内部の計算ロジックが大きく異なる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | NaiveBayes.scala | `mllib/src/main/scala/org/apache/spark/ml/classification/NaiveBayes.scala` | train()/trainWithLabelCheck()メソッド（135-150行目）を理解する |

**主要処理フロー**:
1. **51-52行目**: smoothingパラメータ定義（デフォルト1.0）
2. **63-66行目**: modelTypeパラメータ定義
3. **71行目**: デフォルト設定（smoothing=1.0, modelType="multinomial"）
4. **98-101行目**: ProbabilisticClassifierを継承
5. **135-137行目**: train() → trainWithLabelCheck(dataset, nonNegativeLabel=true)
6. **147-150行目**: trainWithLabelCheck() - パラメータログ出力

### プログラム呼び出し階層図

```
NaiveBayes.train(dataset)
    |
    +-- trainWithLabelCheck(dataset, nonNegativeLabel=true)
            |
            +-- Summarizer（クラスごとの統計量集約）
            |
            +-- モデルタイプ分岐
            |       +-- Multinomial: カウントベースのパラメータ推定
            |       +-- Complement: 補集合統計量ベースの推定
            |       +-- Bernoulli: 出現確率ベースの推定
            |       +-- Gaussian: 平均・分散の推定
            |
            +-- NaiveBayesModel(pi, theta, sigma)

NaiveBayesModel.predictRaw(features)
    |
    +-- モデルタイプに応じた対数尤度計算
    +-- pi + theta * features (Multinomial/Complement/Bernoulli)
    +-- pi + Gaussian対数尤度 (Gaussian)
```

### データフロー図

```
[入力]                  [処理]                          [出力]

Dataset ──────> ラベルチェック
(label,          |
 features,       v
 weight)    Summarizer統計量集約
                 |
                 v
            モデルタイプ別パラメータ推定
                 |
                 v
            NaiveBayesModel ──────> pi(事前確率対数)
            (pi, theta, sigma)      theta(条件付き確率対数)
                 |
                 v
            predictRaw: 対数事後確率
                 |
                 v
            (prediction, rawPrediction, probability)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| NaiveBayes.scala | `mllib/src/main/scala/org/apache/spark/ml/classification/NaiveBayes.scala` | ソース | ナイーブベイズの全実装 |
| ProbabilisticClassifier.scala | `mllib/src/main/scala/org/apache/spark/ml/classification/ProbabilisticClassifier.scala` | ソース | 確率的分類器基底クラス |
| Summarizer.scala | `mllib/src/main/scala/org/apache/spark/ml/stat/Summarizer.scala` | ソース | 統計量集約 |
