# 機能設計書 56-一般化線形回帰

## 概要

本ドキュメントは、Apache Spark MLlibが提供する一般化線形回帰（GeneralizedLinearRegression）機能の設計を記述する。本機能は、一般化線形モデル（GLM）のフレームワークに基づき、様々な誤差分布とリンク関数の組み合わせをサポートする回帰分析機能を提供する。

### 本機能の処理概要

一般化線形回帰は、正規分布以外の誤差分布（ポアソン、ガンマ、二項分布、Tweedie等）を扱える回帰分析の一般化フレームワークである。リンク関数を通じて線形予測子と応答変数の平均を関連付ける。

**業務上の目的・背景**：実務データは正規分布に従わないことが多い。例えばカウントデータ（ポアソン）、比率データ（二項分布）、正値連続データ（ガンマ）など、応答変数の性質に応じた適切な分布仮定が必要。GLMは統計学の標準的フレームワークであり、R/SAS等との互換性も高い。

**機能の利用シーン**：保険料率の算出（Tweedie分布）、クリック数予測（ポアソン）、故障率分析（ガンマ）、生存時間分析等で利用される。

**主要な処理内容**：
1. ファミリー（誤差分布）とリンク関数の組み合わせ検証
2. IRLS（Iteratively Reweighted Least Squares）による最適化
3. オフセット項と重み付きサンプルのサポート
4. AIC（赤池情報量基準）、偏差、残差等の統計的サマリー

**関連システム・外部連携**：WeightedLeastSquaresソルバー、IRLS反復ソルバー。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能に直接関連する画面はなし |

## 機能種別

計算処理（機械学習 - 一般化線形回帰）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| family | String | No | 誤差分布族（デフォルト: "gaussian"） | "gaussian","binomial","poisson","gamma","tweedie" |
| link | String | No | リンク関数（デフォルト: ファミリーのデフォルト） | "identity","log","inverse","logit","probit","cloglog","sqrt" |
| variancePower | Double | No | Tweedieの分散べき（デフォルト: 0.0） | 0 or [1, Inf) |
| linkPower | Double | No | Tweedieのリンクべき | - |
| fitIntercept | Boolean | No | 切片推定（デフォルト: true） | - |
| maxIter | Int | No | 最大反復回数（デフォルト: 25） | >= 0 |
| tol | Double | No | 収束判定閾値（デフォルト: 1E-6） | >= 0 |
| regParam | Double | No | L2正則化パラメータ（デフォルト: 0.0） | >= 0 |
| solver | String | No | ソルバー（デフォルト: "irls"） | "irls" |
| weightCol | String | No | 重み列名 | - |
| offsetCol | String | No | オフセット列名 | - |
| linkPredictionCol | String | No | リンク予測出力列名 | - |
| aggregationDepth | Int | No | treeAggregate深度（デフォルト: 2） | >= 2 |

### 入力データソース

DataFrame形式。ラベル列はDouble型、特徴量列はVector型。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| prediction | Double | 予測値（リンク関数の逆関数適用後） |
| linkPrediction | Double | 線形予測値（linkPredictionCol設定時） |

### 出力先

入力DataFrameに予測列を追加。

## 処理フロー

### 処理シーケンス

```
1. ファミリーとリンク関数の組み合わせ検証
2. データの準備（オフセット、重み含む）
3. 特徴量数の制約チェック（MAX_NUM_FEATURES以下）
4. IRLS反復最適化
   └─ 各反復で重み付き最小二乗問題を解く
5. 係数・切片の抽出
6. 統計的サマリーの生成（偏差、AIC、残差等）
```

### フローチャート

```mermaid
flowchart TD
    A[開始: train] --> B[ファミリー/リンク検証]
    B --> C[特徴量数チェック]
    C --> D{特徴量数 <= MAX_NUM_FEATURES?}
    D -->|No| E[SparkException]
    D -->|Yes| F[FamilyAndLink構築]
    F --> G[データ準備: OffsetInstance]
    G --> H[IRLS最適化]
    H --> I[係数・切片抽出]
    I --> J[サマリー生成]
    J --> K[GeneralizedLinearRegressionModel]
    K --> L[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-56-01 | ファミリー・リンク組み合わせ | サポートされる組み合わせのみ許可 | 訓練時 |
| BR-56-02 | Tweedie特有パラメータ | tweedieファミリー時はlinkPowerを使用（linkパラメータは無視） | family=="tweedie" |
| BR-56-03 | 特徴量数制限 | MAX_NUM_FEATURES以下であること | 訓練時 |
| BR-56-04 | リンク関数適用 | 予測値 = linkFunction.inverse(dot(features, coefficients) + intercept + offset) | 予測時 |

### サポートされるファミリーとリンク関数の組み合わせ

| ファミリー | デフォルトリンク | サポートリンク |
|----------|--------------|--------------|
| gaussian | identity | identity, log, inverse |
| binomial | logit | logit, probit, cloglog |
| poisson | log | log, identity, sqrt |
| gamma | inverse | inverse, identity, log |
| tweedie | linkPower | linkPower（パラメータ指定） |

### 計算ロジック

- 線形予測: `dot(features, coefficients) + intercept + offset`
- 予測値: `link.inverse(eta)` （etaは線形予測値）
- IRLS: 各反復で `z = eta + (y - mu) * link.deriv(mu)` を計算し、重み付き最小二乗を解く

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | SparkException | 特徴量数がMAX_NUM_FEATURESを超過 | 特徴量削減 |
| - | RequireError | 未サポートのファミリー・リンク組み合わせ | サポート組合せを使用 |
| - | RequireError | fitIntercept=falseかつ特徴量数0 | 特徴量追加またはfitIntercept=true |

### リトライ仕様

特になし。IRLSの収束はmaxIterとtolで制御。

## トランザクション仕様

Sparkの遅延評価に基づく。

## パフォーマンス要件

- IRLSの各反復でWeightedLeastSquaresを解くため、大規模特徴量では計算負荷が高い
- aggregationDepthで集約の深さを制御可能

## セキュリティ考慮事項

Sparkの標準セキュリティ機構に従う。

## 備考

- Spark 2.0.0で導入
- Rのglm()関数と同等の統計的サマリーを提供
- Tweedieファミリーの追加はSpark 2.2.0
- offsetCol対応はSpark 2.3.0

---

## コードリーディングガイド

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | GeneralizedLinearRegression.scala | `mllib/src/main/scala/org/apache/spark/ml/regression/GeneralizedLinearRegression.scala` | GeneralizedLinearRegressionBase（49-229行目）でファミリー、リンク、各パラメータを理解 |
| 1-2 | GeneralizedLinearRegression.scala | 同上 | Family/Link内部クラス群でGaussian, Binomial, Poisson, Gamma, Tweedieの各分布を確認 |

**読解のコツ**: Familyオブジェクトは分散関数とリンク関数のデフォルトを定義し、Linkオブジェクトはリンク関数とその逆関数・微分を定義する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | GeneralizedLinearRegression.scala | 同上 | train()（380-399行目）でFamilyAndLink構築とIRLS呼出し |

**主要処理フロー**:
1. **382行目**: FamilyAndLink構築
2. **389行目**: 特徴量数取得
3. **392-396行目**: MAX_NUM_FEATURES制約チェック
4. IRLS最適化実行
5. モデル生成とサマリー作成

#### Step 3: IRLS最適化を理解する

IRLS（Iteratively Reweighted Least Squares）は、GLMの標準的な推定手法であり、各反復で作業応答変数と作業重みを計算し、重み付き最小二乗問題を解く。

#### Step 4: サマリーを理解する

GLMSummaryは偏差、AIC、Pearson残差、deviance残差等の統計量を提供する。

### プログラム呼び出し階層図

```
GeneralizedLinearRegression.train(dataset)
    |
    +-- FamilyAndLink(this)
    +-- getNumFeatures(dataset)
    +-- [IRLS最適化]
    |       +-- WeightedLeastSquares.fit() [各反復]
    |
    +-- new GeneralizedLinearRegressionModel(uid, coefficients, intercept)
    +-- model.createSummary()
```

### データフロー図

```
[入力]                    [処理]                              [出力]

DataFrame          train()
  (label,          +-- Family/Link検証              GeneralizedLinearRegressionModel
   features,       +-- OffsetInstance構築               (coefficients,
   weight,         +-- IRLS反復最適化                     intercept)
   offset)
                                                    GeneralizedLinearRegressionSummary
                                                      (deviance, AIC, residuals)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| GeneralizedLinearRegression.scala | `mllib/src/main/scala/org/apache/spark/ml/regression/GeneralizedLinearRegression.scala` | ソース | Estimator/Model/Summary/Family/Link全体の実装 |
| WeightedLeastSquares.scala | `mllib/src/main/scala/org/apache/spark/ml/optim/WeightedLeastSquares.scala` | ソース | IRLS各反復の重み付き最小二乗ソルバー |
| Instance.scala | `mllib/src/main/scala/org/apache/spark/ml/feature/Instance.scala` | ソース | OffsetInstance等のデータ構造 |
