# 機能設計書 62-ガウス混合モデル

## 概要

本ドキュメントは、Apache Spark MLlibにおけるガウス混合モデル（Gaussian Mixture Model, GMM）機能の設計を記述する。期待値最大化（EM）アルゴリズムにより、データを複数のガウス分布の混合として表現するクラスタリング手法の訓練・予測処理を対象とする。

### 本機能の処理概要

**業務上の目的・背景**：K-Meansのようなハードクラスタリングとは異なり、ガウス混合モデルはソフトクラスタリングを提供する。各データ点が各クラスタに属する確率を算出できるため、不確実性を伴うクラスタリングが必要な場面で有用である。多変量ガウス分布の混合としてデータの分布をモデル化し、EMアルゴリズムにより最尤推定を行う。

**機能の利用シーン**：異常検知（正常データの分布からの逸脱検出）、データの密度推定、オーバーラップするクラスタを持つデータのセグメンテーション、生成モデルとしての利用など。各クラスタへの所属確率が必要な分析シナリオで利用される。

**主要な処理内容**：
1. EMアルゴリズムのE-step：各データ点の各コンポーネントへの帰属度（責任度）を計算
2. EMアルゴリズムのM-step：帰属度に基づき各コンポーネントのパラメータ（平均、共分散、混合重み）を更新
3. 収束判定：対数尤度の変化量が閾値以下または最大反復回数到達で収束
4. 学習済みモデルによる確率的クラスタ割り当てと新規データの予測

**関連システム・外部連携**：Spark DataFrame APIおよびPipeline APIと統合されている。内部的にBREEZEライブラリの多変量ガウス分布を利用する。

**権限による制御**：特になし。Sparkアプリケーション実行権限に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 該当なし | - | MLlib機能はWeb UIに専用画面を持たない |

## 機能種別

計算処理（機械学習アルゴリズム - クラスタリング）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| k | Int | No | ガウスコンポーネント数（デフォルト: 2） | >= 1 |
| maxIter | Int | No | 最大反復回数（デフォルト: 100） | > 0 |
| tol | Double | No | 収束閾値（デフォルト: 0.01） | >= 0 |
| seed | Long | No | 乱数シード | - |
| featuresCol | String | No | 特徴量カラム名（デフォルト: "features"） | Vector型カラム |
| predictionCol | String | No | 予測結果カラム名（デフォルト: "prediction"） | - |
| probabilityCol | String | No | 確率カラム名（デフォルト: "probability"） | - |
| weightCol | String | No | 重みカラム名 | 非負の数値型 |
| aggregationDepth | Int | No | 集約の深さ（デフォルト: 2） | >= 2 |

### 入力データソース

Spark DataFrame。featuresColで指定されたカラムにVector型の特徴量データを含むこと。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| prediction | Int | 最も確率の高いクラスタのインデックス |
| probability | Vector | 各クラスタへの所属確率ベクトル |
| weights | Array[Double] | 各ガウスコンポーネントの混合重み |
| gaussians | Array[MultivariateGaussian] | 各コンポーネントの多変量ガウス分布（平均・共分散） |

### 出力先

入力DataFrameにprediction/probabilityカラムを追加したDataFrame。モデルはMLWriterを通じて永続化可能。

## 処理フロー

### 処理シーケンス

```
1. スキーマバリデーション
   └─ featuresColがVector互換カラムか検証
2. 入力データの準備
   └─ 特徴量と重みをRDDに変換し、キャッシュ
3. 次元数・データ数の検証
   └─ k * 特徴量次元が最大値を超えないか確認
4. EMアルゴリズム実行
   └─ 初期化 → E-step/M-step反復 → 収束判定
5. GaussianMixtureModelの生成
   └─ 学習済みパラメータ（weights, means, covariances）をモデルに格納
6. サマリーの生成
   └─ 対数尤度・クラスタサイズ等を算出
```

### フローチャート

```mermaid
flowchart TD
    A[開始: fit呼び出し] --> B[スキーマ検証]
    B --> C[RDD変換・キャッシュ]
    C --> D[次元数・データ数検証]
    D --> E[EM初期化]
    E --> F[E-step: 帰属度計算]
    F --> G[M-step: パラメータ更新]
    G --> H{収束判定}
    H -->|未収束| F
    H -->|収束| I[GaussianMixtureModel生成]
    I --> J[Summary生成]
    J --> K[モデル返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-62-01 | コンポーネント数制約 | kは1以上でなければならない | 常時 |
| BR-62-02 | 収束判定 | 対数尤度の変化がtol以下で収束とみなす | EM反復中 |
| BR-62-03 | 予測ルール | predictionは各コンポーネントの帰属確率が最大のインデックス | 予測時 |
| BR-62-04 | 確率出力 | probabilityは各コンポーネントへの帰属確率ベクトル | 予測時 |

### 計算ロジック

EMアルゴリズム：
- E-step: 各データ点xに対し、コンポーネントkの帰属度 r(k) = w_k * N(x|mu_k, Sigma_k) / sum(w_j * N(x|mu_j, Sigma_j))
- M-step: w_k = sum(r_k)/N, mu_k = sum(r_k * x)/sum(r_k), Sigma_k = 加重共分散

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし（インメモリ処理） |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| IllegalArgumentException | パラメータ検証エラー | k < 1 | kを1以上に設定 |
| SparkException | 数値エラー | 共分散行列が特異（正則でない） | 正則化またはデータ前処理 |
| SchemaValidationError | スキーマエラー | featuresColがVector型でない | 正しいカラム名を指定 |

### リトライ仕様

リトライ機構は組み込まれていない。EMアルゴリズムの収束はmaxIterとtolで制御。

## トランザクション仕様

トランザクション管理なし。分散RDD処理によるインメモリ計算。

## パフォーマンス要件

aggregationDepthパラメータにより集約処理の並列度を制御可能。大規模データでは適切な値を設定することでパフォーマンスを改善できる。

## セキュリティ考慮事項

特段のセキュリティ制御なし。Sparkアプリケーションのアクセス制御に依存する。

## 備考

- Spark 2.0.0でml API版が導入
- 内部でBREEZEの多変量正規分布を利用
- 高次元データでは共分散行列の推定に注意が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | GaussianMixture.scala | `mllib/src/main/scala/org/apache/spark/ml/clustering/GaussianMixture.scala` | GaussianMixtureParamsトレイトでパラメータ定義を確認 |

**読解のコツ**: GaussianMixtureはHasMaxIter, HasFeaturesCol, HasSeed, HasPredictionCol, HasProbabilityCol, HasWeightCol等の共通パラメータトレイトを継承している。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | GaussianMixture.scala | `mllib/src/main/scala/org/apache/spark/ml/clustering/GaussianMixture.scala` | GaussianMixtureクラスのfit()メソッド |

**主要処理フロー**:
1. スキーマ検証とパラメータロギング
2. DataFrameからRDDへの変換
3. EMアルゴリズムの実行（反復処理）
4. GaussianMixtureModelの生成

#### Step 3: モデルの予測処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GaussianMixture.scala | `mllib/src/main/scala/org/apache/spark/ml/clustering/GaussianMixture.scala` | GaussianMixtureModelのtransform()メソッド |

### プログラム呼び出し階層図

```
GaussianMixture.fit(dataset)
    |
    +-- transformSchema(schema)
    |
    +-- dataset → RDD変換
    |
    +-- EMアルゴリズム実行
    |       +-- E-step: 帰属度計算
    |       +-- M-step: パラメータ更新
    |       +-- 収束判定
    |
    +-- GaussianMixtureModel(uid, weights, gaussians)
    |
    +-- GaussianMixtureSummary(...)
```

### データフロー図

```
[入力]                   [処理]                          [出力]

DataFrame           GaussianMixture.fit()          GaussianMixtureModel
(features,      --> EMアルゴリズム実行          --> (weights, gaussians,
 weight)                                             summary)

DataFrame           GaussianMixtureModel           DataFrame
(features)     --> .transform()               --> (features + prediction
                    確率計算・argmax                    + probability)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| GaussianMixture.scala | `mllib/src/main/scala/org/apache/spark/ml/clustering/GaussianMixture.scala` | ソース | ml API層のEstimator/Model定義 |
| MultivariateGaussian.scala | `mllib/src/main/scala/org/apache/spark/ml/stat/distribution/MultivariateGaussian.scala` | ソース | 多変量ガウス分布の実装 |
| ClusteringSummary.scala | `mllib/src/main/scala/org/apache/spark/ml/clustering/ClusteringSummary.scala` | ソース | クラスタリングサマリー基底クラス |
