# 通知設計書 12-SparkListenerUnpersistRDD

## 概要

本ドキュメントは、Apache SparkのSparkListenerUnpersistRDDイベント通知の設計仕様を記載する。このイベントはアプリケーションが手動でRDDをアンパーシスト（永続化解除）した際に発火し、LiveListenerBus経由で非同期に配信される。

### 本通知の処理概要

SparkListenerUnpersistRDDは、SparkContextのunpersistRDDメソッドを通じてRDDの永続化が解除された際にリスナーバスへ通知を行うイベントである。

**業務上の目的・背景**：Sparkアプリケーションでは、パフォーマンス向上のためにRDDをメモリやディスクに永続化（persist/cache）することが一般的である。しかし、不要になったRDDのリソースを解放するためにunpersist操作が行われる。この通知により、Spark UIでのストレージ情報の更新、イベントログへの記録、動的リソース割り当て（ExecutorMonitor）でのキャッシュブロック追跡が適切に行われる。

**通知の送信タイミング**：SparkContext.unpersistRDDメソッド内で、BlockManagerMasterへのRDD削除要求の後に`listenerBus.post(SparkListenerUnpersistRDD(rddId))`が呼び出される（SparkContext.scala 2162行目）。

**通知の受信者**：LiveListenerBusに登録された全てのSparkListenerInterface実装クラス。主要な受信者はAppStatusListener（UI更新）、EventLoggingListener（イベントログ）、ExecutorMonitor（動的リソース割り当て用のキャッシュブロック追跡）である。

**通知内容の概要**：アンパーシストされたRDDのID（整数値）が含まれる。

**期待されるアクション**：受信者はRDDのストレージ情報をUIから削除し、ExecutorMonitorはキャッシュブロックの参照を更新してExecutorのアイドル判定に反映する。

## 通知種別

アプリ内イベント通知（SparkListenerEvent）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（LiveListenerBus経由、AsyncEventQueue使用） |
| 優先度 | 中（通常のイベントキュー処理） |
| リトライ | なし（イベントキューが満杯の場合はドロップされる可能性あり） |

### 送信先決定ロジック

LiveListenerBusに登録された全てのAsyncEventQueueに対してブロードキャスト配信される。SparkListenerBus.doPostEventメソッドでイベントタイプに基づくパターンマッチングが行われ、`listener.onUnpersistRDD(unpersistRDD)`が呼び出される。

## 通知テンプレート

### メール通知の場合

該当なし。本通知はSpark内部のイベントバスを通じたプログラム間通知である。

### 本文テンプレート

```
イベント種別: SparkListenerUnpersistRDD
JSON形式:
{
  "Event": "SparkListenerUnpersistRDD",
  "RDD ID": <rddId>
}
```

### 添付ファイル

該当なし。

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| rddId | アンパーシストされたRDDの識別子 | SparkContext.unpersistRDD引数 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | RDD.unpersist() | RDDの永続化解除が要求された場合 | ユーザコードからの明示的なunpersist呼び出し |
| API呼び出し | SparkContext.unpersistRDD() | SparkContext内部のunpersistRDDが呼ばれた場合 | RDD.unpersistからの間接呼び出し |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| LiveListenerBus停止済み | LiveListenerBusのstop()が呼ばれた後はイベントがドロップされる |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[RDD.unpersist呼び出し] --> B[SparkContext.unpersistRDD]
    B --> C[BlockManagerMaster.removeRdd]
    C --> D[各Executorのブロック削除]
    B --> E[persistentRdds.remove]
    E --> F[listenerBus.post SparkListenerUnpersistRDD]
    F --> G[LiveListenerBus.post]
    G --> H[各AsyncEventQueueにイベント配信]
    H --> I[SparkListenerBus.doPostEvent]
    I --> J{受信リスナー}
    J -->|AppStatusListener| K[RDDストレージ情報更新]
    J -->|EventLoggingListener| L[イベントログ書き込み]
    J -->|ExecutorMonitor| M[キャッシュブロック追跡更新]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| persistentRdds（インメモリMap） | 永続化済みRDDの管理 | SparkContext内部のConcurrentMap |

### テーブル別参照項目詳細

#### persistentRdds

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| rddId | アンパーシスト対象RDDの特定 | キーとして指定されたrddIdで検索 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| persistentRdds（インメモリMap） | DELETE | 対象RDDのエントリ削除 |
| KVStore（AppStatusListener経由） | UPDATE | RDDストレージ情報の更新 |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| INSERT | Event | SparkListenerUnpersistRDD | EventLoggingListenerによりイベントログに書き込み |
| INSERT | RDD ID | rddId | アンパーシストされたRDD ID |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| キュー満杯 | AsyncEventQueueのキャパシティ超過 | イベントがドロップされ、ドロップカウンターが増加 |
| リスナー例外 | リスナーのonUnpersistRDDで例外発生 | ListenerBusが例外をキャッチしログに記録。他リスナーへの配信は継続 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0回（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。アプリケーション実行中にunpersistが呼ばれた任意のタイミングで配信される。

## セキュリティ考慮事項

本イベントにはRDD IDのみが含まれ、機密情報や個人情報は含まれない。イベントログファイルのファイルシステムパーミッションによるアクセス制御で十分である。

## 備考

- RDD.unpersist()はblocking引数を受け取り、trueの場合はBlockManagerMasterでの削除完了を待機する。ただし、イベントの投稿自体は常に非同期で行われる。
- ExecutorMonitor（動的リソース割り当て）はonUnpersistRDDで当該RDDのキャッシュブロック情報をクリアし、Executorのアイドル判定に反映する。
- History ServerでのイベントログフィルタリングではBasicEventFilterBuilderがRDD IDに基づいてフィルタリングを行う。

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | 116行目: case class SparkListenerUnpersistRDD(rddId: Int)の定義。rddIdの1フィールドのみ |

**読解のコツ**: 非常にシンプルなcase classで、RDD IDのみを持つ。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | SparkContext.scala | `core/src/main/scala/org/apache/spark/SparkContext.scala` | 2159-2163行目: unpersistRDDメソッド。BlockManagerMaster.removeRddの後にイベント投稿 |

**主要処理フロー**:
1. **2160行目**: `env.blockManager.master.removeRdd(rddId, blocking)` -- 実データの削除
2. **2161行目**: `persistentRdds.remove(rddId)` -- 内部マップからの削除
3. **2162行目**: `listenerBus.post(SparkListenerUnpersistRDD(rddId))` -- イベント発火

#### Step 3: イベントバスの配信メカニズムを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | LiveListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/LiveListenerBus.scala` | 126行目: postメソッド |
| 3-2 | SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | 52行目: doPostEventでlistener.onUnpersistRDDを呼び出す |

#### Step 4: 受信リスナーの処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | 903行目: onUnpersistRDDでRDDストレージ情報を更新 |
| 4-2 | ExecutorMonitor.scala | `core/src/main/scala/org/apache/spark/scheduler/dynalloc/ExecutorMonitor.scala` | 438行目: onUnpersistRDDでキャッシュブロック情報を更新 |
| 4-3 | EventLoggingListener.scala | `core/src/main/scala/org/apache/spark/scheduler/EventLoggingListener.scala` | 174行目: イベントログに記録 |

### プログラム呼び出し階層図

```
RDD.unpersist(blocking)
    |
    +-- SparkContext.unpersistRDD(rddId, blocking)
            |
            +-- env.blockManager.master.removeRdd(rddId, blocking)
            +-- persistentRdds.remove(rddId)
            +-- listenerBus.post(SparkListenerUnpersistRDD(rddId))
                    |
                    +-- LiveListenerBus.post(event)
                            |
                            +-- postToQueues(event)
                                    |
                                    +-- SparkListenerBus.doPostEvent(listener, event)
                                            |
                                            +-- listener.onUnpersistRDD(event)
                                                    |
                                                    +-- AppStatusListener.onUnpersistRDD()
                                                    +-- ExecutorMonitor.onUnpersistRDD()
                                                    +-- EventLoggingListener.onUnpersistRDD()
```

### データフロー図

```
[入力]                    [処理]                           [出力]

RDD.unpersist() ------> SparkContext.unpersistRDD() ----> SparkListenerUnpersistRDD
(rddId, blocking)            |                               イベント
                             v
                       BlockManagerMaster
                       .removeRdd()
                             |
                             v
                       LiveListenerBus.post() -----------> AsyncEventQueue
                                                               |
                                                               v
                                                         AppStatusListener --> KVStore (UI更新)
                                                         ExecutorMonitor --> キャッシュ追跡更新
                                                         EventLoggingListener --> イベントログ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | ソース | イベントcase classの定義 |
| SparkContext.scala | `core/src/main/scala/org/apache/spark/SparkContext.scala` | ソース | イベントの発火元（unpersistRDD） |
| LiveListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/LiveListenerBus.scala` | ソース | 非同期イベントバスの実装 |
| SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | ソース | イベントのディスパッチ |
| AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | ソース | UI用データの更新 |
| ExecutorMonitor.scala | `core/src/main/scala/org/apache/spark/scheduler/dynalloc/ExecutorMonitor.scala` | ソース | 動的リソース割り当て用キャッシュ追跡 |
| EventLoggingListener.scala | `core/src/main/scala/org/apache/spark/scheduler/EventLoggingListener.scala` | ソース | イベントログへの記録 |
| JsonProtocol.scala | `core/src/main/scala/org/apache/spark/util/JsonProtocol.scala` | ソース | JSON形式のシリアライズ・デシリアライズ |
| BasicEventFilterBuilder.scala | `core/src/main/scala/org/apache/spark/deploy/history/BasicEventFilterBuilder.scala` | ソース | History Serverでのイベントフィルタリング |
