# 通知設計書 20-SparkListenerNodeUnexcluded

## 概要

本ドキュメントは、Apache SparkのSparkListenerNodeUnexcludedイベント通知の設計仕様を記載する。このイベントは以前除外されたノード（ホスト）が再有効化された際に発火し、LiveListenerBus経由で非同期に配信される。Spark 3.1.0で導入された。

### 本通知の処理概要

SparkListenerNodeUnexcludedは、HealthTrackerがタイムアウトに基づいて除外ノードを再有効化した際にリスナーバスへ通知を行うイベントである。

**業務上の目的・背景**：Sparkの除外メカニズムでは、問題のあるノードを一時的にタスクスケジューリングおよびExecutor割り当てから除外するが、EXCLUDE_ON_FAILURE_TIMEOUT_MILLIS（デフォルト1時間）の経過後に自動的に再有効化される。これは一時的なノード障害（ネットワーク一時断、ディスク一時障害等）からの回復を想定した設計である。この通知により、Spark UIでのノード除外ステータスの解除表示、イベントログへの記録が可能となる。

**通知の送信タイミング**：HealthTracker.applyExcludeOnFailureTimeoutメソッド内で、nodeIdToExcludedExpiryTimeのexpiryTimeが現在時刻を過ぎたノードに対して`listenerBus.post(SparkListenerNodeUnexcluded(now, node))`が呼び出される（137行目）。

**通知の受信者**：LiveListenerBusに登録された全てのSparkListenerInterface実装クラス。主要な受信者はAppStatusListener（UI更新）とEventLoggingListener（イベントログ）である。

**通知内容の概要**：再有効化された時刻とノードのホストIDが含まれる。

**期待されるアクション**：受信者はノードの除外ステータスを解除し、UIに反映する。クラスタマネージャは当該ノードへの新規Executor割り当てを再開する。

## 通知種別

アプリ内イベント通知（SparkListenerEvent）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（LiveListenerBus経由、AsyncEventQueue使用） |
| 優先度 | 中 |
| リトライ | なし |

### 送信先決定ロジック

LiveListenerBusに登録された全てのAsyncEventQueueに対してブロードキャスト配信される。SparkListenerBus.doPostEventメソッド（88行目）で`listener.onNodeUnexcluded(nodeUnexcluded)`が呼び出される。

## 通知テンプレート

### メール通知の場合

該当なし。

### 本文テンプレート

```
イベント種別: SparkListenerNodeUnexcluded
JSON形式:
{
  "Event": "SparkListenerNodeUnexcluded",
  "Timestamp": <timestamp>,
  "Host ID": "<hostId>"
}
```

### 添付ファイル

該当なし。

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| time | 再有効化時刻（ミリ秒） | clock.getTimeMillis() | Yes |
| hostId | 再有効化されたノードのホスト名 | nodeIdToExcludedExpiryTimeのキー | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 内部処理 | タイムアウト到達 | ノードの除外expiryTimeが現在時刻を過ぎた場合 | HealthTracker.applyExcludeOnFailureTimeout 137行目 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| タイムアウト未到達 | expiryTimeが現在時刻を超えていない場合はイベント発火しない |
| nextExpiryTime未到達 | 最短のexpiryTimeが現在時刻を超えていない場合、applyExcludeOnFailureTimeout自体がスキップされる |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[TaskSchedulerImpl.resourceOffers等のタイミング] --> B[HealthTracker.applyExcludeOnFailureTimeout]
    B --> C{now > nextExpiryTime?}
    C -->|Yes| D[Executor再有効化処理 先に実行]
    D --> E[nodeIdToExcludedExpiryTime.filter expiryTime < now]
    C -->|No| F[処理スキップ]
    E --> G{対象ノード存在?}
    G -->|Yes| H[nodeIdToExcludedExpiryTimeから削除]
    H --> I[listenerBus.post SparkListenerNodeUnblacklisted 後方互換]
    I --> J[listenerBus.post SparkListenerNodeUnexcluded]
    J --> K[_excludedNodeList.set更新]
    G -->|No| L[処理なし]
    J --> M[LiveListenerBus.post]
    M --> N{受信リスナー}
    N -->|AppStatusListener| O[ノード除外ステータス解除]
    N -->|EventLoggingListener| P[イベントログ書き込み]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| nodeIdToExcludedExpiryTime（インメモリMap） | タイムアウト判定 | HealthTracker内部のHashMap |

### テーブル別参照項目詳細

#### nodeIdToExcludedExpiryTime

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| node（キー） | 再有効化対象の特定 | expiryTime < now でフィルタ |
| expiryTime（値） | タイムアウト判定 | 現在時刻との比較 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| nodeIdToExcludedExpiryTime（インメモリMap） | DELETE | タイムアウト到達ノードのエントリ削除 |
| _excludedNodeList（AtomicReference） | UPDATE | 除外ノードリストのスナップショット更新 |
| KVStore（AppStatusListener経由） | UPDATE | ノード除外ステータスの解除 |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| INSERT | Event | SparkListenerNodeUnexcluded | イベントログに書き込み |
| INSERT | Timestamp | clock.getTimeMillis() | 再有効化時刻 |
| INSERT | Host ID | hostId | 再有効化されたノードのホスト名 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| キュー満杯 | AsyncEventQueueのキャパシティ超過 | イベントがドロップされる |
| リスナー例外 | リスナーで例外発生 | ListenerBusが例外をキャッチしログに記録 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0回 |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。

## セキュリティ考慮事項

イベントにはホストIDのみが含まれ、機密情報は含まれない。

## 備考

- 本イベントはSpark 3.1.0で導入された（@Since("3.1.0")）。
- SparkListenerNodeExcludedと対になるイベントである。
- 後方互換のためSparkListenerNodeUnblacklisted（@deprecated）も同時に発火される（136行目）。
- applyExcludeOnFailureTimeoutでは、Executorの再有効化（SparkListenerExecutorUnexcluded）が先に処理され、その後にノードの再有効化が処理される。
- ノード再有効化時には_excludedNodeListのスナップショットも更新される（139行目）。
- updateNextExpiryTimeが呼び出され、次回のタイムアウトチェック時刻が再計算される。

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | 218行目: case class SparkListenerNodeUnexcluded(time, hostId)の定義 |

**読解のコツ**: time（Long）とhostId（String）の2フィールドのみのシンプルなcase class。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | HealthTracker.scala | `core/src/main/scala/org/apache/spark/scheduler/HealthTracker.scala` | 128-143行目: applyExcludeOnFailureTimeout内のノード再有効化処理 |

**主要処理フロー**:
1. **128行目**: nodeIdToExcludedExpiryTime.filter(_._2 < now) -- タイムアウト到達ノードの抽出
2. **134行目**: nodeIdToExcludedExpiryTime.remove(node) -- 除外リストからの削除
3. **136行目**: SparkListenerNodeUnblacklisted発火（後方互換）
4. **137行目**: SparkListenerNodeUnexcluded発火
5. **139行目**: _excludedNodeList.set(nodeIdToExcludedExpiryTime.keySet.toSet) -- スナップショット更新
6. **141行目**: updateNextExpiryTime() -- 次回タイムアウト時刻の再計算

#### Step 3: 受信リスナーの処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | 354行目: onNodeUnexcludedでupdateNodeExcluded(hostId, false)を呼び出し |
| 3-2 | EventLoggingListener.scala | `core/src/main/scala/org/apache/spark/scheduler/EventLoggingListener.scala` | 240行目: イベントログに記録 |

### プログラム呼び出し階層図

```
TaskSchedulerImpl.resourceOffers() (定期呼び出し)
    |
    +-- HealthTracker.applyExcludeOnFailureTimeout()
            |
            +-- [Executor再有効化処理（先に実行）]
            |
            +-- nodeIdToExcludedExpiryTime.filter(expiryTime < now)
            +-- nodeIdToExcludedExpiryTime.remove(node)
            +-- listenerBus.post(SparkListenerNodeUnblacklisted) [後方互換]
            +-- listenerBus.post(SparkListenerNodeUnexcluded)
                    |
                    +-- LiveListenerBus.post(event)
                            |
                            +-- AppStatusListener.onNodeUnexcluded()
                            +-- EventLoggingListener.onNodeUnexcluded()
            +-- _excludedNodeList.set()
            +-- updateNextExpiryTime()
```

### データフロー図

```
[入力]                       [処理]                              [出力]

タイムアウト到達 ----------> HealthTracker -----------------> SparkListenerNodeUnexcluded
(now > expiryTime)           .applyExcludeOnFailureTimeout()    イベント
                                  |
                                  v
                            nodeIdToExcludedExpiryTime削除
                            _excludedNodeList更新
                                  |
                                  v
                            LiveListenerBus.post() ----------> AsyncEventQueue
                                                                    |
                                                                    v
                                                              AppStatusListener --> ノード除外解除
                                                              EventLoggingListener --> ログ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | ソース | イベントcase classの定義 |
| HealthTracker.scala | `core/src/main/scala/org/apache/spark/scheduler/HealthTracker.scala` | ソース | イベントの発火元（タイムアウト処理） |
| LiveListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/LiveListenerBus.scala` | ソース | 非同期イベントバス |
| SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | ソース | イベントディスパッチ |
| AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | ソース | ノード除外ステータス解除 |
| EventLoggingListener.scala | `core/src/main/scala/org/apache/spark/scheduler/EventLoggingListener.scala` | ソース | イベントログ記録 |
