# 通知設計書 27-SparkListenerApplicationEnd

## 概要

本ドキュメントは、Apache SparkにおけるSparkListenerApplicationEndイベント通知の設計仕様を定義する。このイベントは、Sparkアプリケーションが終了する際に発火し、終了時刻と終了コードをリスナーに通知する。

### 本通知の処理概要

本通知は、SparkContext.stop()が呼ばれアプリケーションが終了する際に発火するイベントであり、Sparkアプリケーションのライフサイクルにおける最終イベントである。

**業務上の目的・背景**：Sparkアプリケーションの終了を正式に記録し、Spark UIおよび履歴サーバでアプリケーションの完了状態を表示するための基盤データを提供する。終了時刻と終了コードにより、アプリケーションが正常に完了したか異常終了したかを判断できる。

**通知の送信タイミング**：SparkContext.postApplicationEnd(exitCode)メソッドが呼ばれた際に発火する。SparkContext.stop()の処理フロー内で、ハートビートメトリクスの最終報告後に呼ばれる。

**通知の受信者**：LiveListenerBusに登録されたすべてのSparkListenerInterface実装クラス。主要な受信者はAppStatusListener（アプリケーション終了情報のKVStore更新）、EventLoggingListener（イベントログへの永続化）である。

**通知内容の概要**：time（終了時刻、ミリ秒）およびexitCode（終了コード、Option[Int]）の2フィールドを含む。

**期待されるアクション**：AppStatusListenerがアプリケーションの終了時刻と完了フラグをKVStoreに更新する。EventLoggingListenerがイベントログに記録し、履歴サーバがアプリケーションの完了状態を更新する。

## 通知種別

アプリ内イベント通知（Sparkイベントリスナーバス経由の非同期配信）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（LiveListenerBus経由） |
| 優先度 | 高（アプリケーションライフサイクルイベント） |
| リトライ | 無 |

### 送信先決定ロジック

LiveListenerBusに登録されたすべてのSparkListenerInterface実装リスナーに対してブロードキャスト配信される。SparkListenerBus.doPostEventメソッドにおいて、SparkListenerApplicationEndイベントとパターンマッチし、listener.onApplicationEnd()が呼び出される。

## 通知テンプレート

### メール通知の場合

該当なし。本通知はSparkの内部イベントバスを介したプログラマティック通知である。

### 本文テンプレート

```
イベント: SparkListenerApplicationEnd
終了時刻: {time}
終了コード: {exitCode}
```

### 添付ファイル

該当なし。

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| time | アプリケーション終了時刻（ミリ秒） | System.currentTimeMillis | Yes |
| exitCode | アプリケーション終了コード | SparkContext.stop(exitCode)パラメータ（Option[Int]） | No（デフォルトNone） |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| アプリケーション終了 | SparkContext.stop() | 常時（SparkContext停止時に必ず発火） | postApplicationEnd(exitCode)メソッド内で発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 特になし | SparkContextが停止される限り、必ず発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[SparkContext.stop] --> B[reportDriverHeartBeatMetrics 最終報告]
    B --> C[SparkContext.postApplicationEnd]
    C --> D[listenerBus.post SparkListenerApplicationEnd]
    D --> E[SparkListenerBus.doPostEvent]
    E --> F[listener.onApplicationEnd]
    F --> G[AppStatusListener: 終了情報KVStore更新]
    F --> H[EventLoggingListener: イベントログ記録]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| AppStatusStore (KVStore) | UPDATE | AppStatusListenerがApplicationAttemptInfoの終了時刻・完了フラグを更新 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonApplicationEnd実装で例外発生 | ListenerBusが例外をキャッチしログ出力 |
| ハートビート報告失敗 | reportDriverHeartBeatMetricsの失敗 | try-catchで捕捉しログ出力後、postApplicationEndは続行される |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | N/A（アプリケーション終了時に1回のみ発火） |
| 1日あたり上限 | N/A |

### 配信時間帯

制限なし。アプリケーション終了時に1回のみ発火する。

## セキュリティ考慮事項

本通知には終了時刻と終了コードのみが含まれ、個人情報や機密データは含まれない。

## 備考

- SparkListenerApplicationStartの対となるイベントである。
- exitCodeフィールドはOption[Int]型であり、デフォルト値はNone。SparkContext.stop()ではSome(exitCode)が渡される（SparkContext.scala行2953）。
- EventLoggingListenerではflushLogger=trueで即座にフラッシュされる。
- JsonProtocolでのJSON直列化では"Timestamp"および"ExitCode"キーが使用される。
- FsHistoryProviderではアプリケーション終了イベントのJSONプレフィックス（APPL_END_EVENT_PREFIX）で高速フィルタリングを行う。

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | 行283-285: SparkListenerApplicationEndケースクラスの定義。time、exitCodeの2フィールド |

**読解のコツ**: exitCodeはOption[Int]型でデフォルトNone。正常終了時は0、異常終了時は非0の値が設定される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | SparkContext.scala | `core/src/main/scala/org/apache/spark/SparkContext.scala` | 行2945-2953: postApplicationEndメソッド。ハートビートメトリクスの最終報告後にイベントを発火 |

**主要処理フロー**:
1. **行2948-2951**: reportDriverHeartBeatMetricsの最終報告（try-catch内）
2. **行2953**: listenerBus.post(SparkListenerApplicationEnd(System.currentTimeMillis, Some(exitCode)))

#### Step 3: リスナー処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | 行192-210: onApplicationEndの実装。ApplicationAttemptInfoの終了時刻・完了フラグを更新 |
| 3-2 | EventLoggingListener.scala | `core/src/main/scala/org/apache/spark/scheduler/EventLoggingListener.scala` | 行182-184: onApplicationEndの実装 |

### プログラム呼び出し階層図

```
SparkContext.stop(exitCode)
    |
    +-- postApplicationEnd(exitCode)
           |
           +-- reportDriverHeartBeatMetrics (最終)
           |
           +-- listenerBus.post(SparkListenerApplicationEnd)
                  |
                  +-- SparkListenerBus.doPostEvent
                         |
                         +-- AppStatusListener.onApplicationEnd
                         |      |
                         |      +-- KVStore更新 (終了時刻, 完了フラグ)
                         |
                         +-- EventLoggingListener.onApplicationEnd
                                |
                                +-- logEvent (flushLogger=true)
```

### データフロー図

```
[入力]                          [処理]                              [出力]

SparkContext.stop() -------> postApplicationEnd(exitCode) ----> SparkListenerApplicationEnd
(exitCode)                        |                                   |
                                  v                                   v
                           LiveListenerBus                     AppStatusStore更新
                                  |                              イベントログ (JSON)
                                  v
                            リスナーへ配信
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | ソース | イベントケースクラス定義 |
| SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | ソース | イベント配信ディスパッチ |
| SparkContext.scala | `core/src/main/scala/org/apache/spark/SparkContext.scala` | ソース | イベント発火元（postApplicationEnd） |
| AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | ソース | アプリケーション終了情報のKVStore更新 |
| EventLoggingListener.scala | `core/src/main/scala/org/apache/spark/scheduler/EventLoggingListener.scala` | ソース | イベントログ永続化 |
| FsHistoryProvider.scala | `core/src/main/scala/org/apache/spark/deploy/history/FsHistoryProvider.scala` | ソース | 履歴サーバ（APPL_END_EVENT_PREFIXで高速フィルタリング） |
| JsonProtocol.scala | `core/src/main/scala/org/apache/spark/util/JsonProtocol.scala` | ソース | JSON直列化・逆直列化 |
