# 通知設計書 3-SparkListenerTaskStart

## 概要

本ドキュメントは、Apache Sparkのコアイベント通知である`SparkListenerTaskStart`の設計について記述する。このイベントは、タスクの実行が開始された際にDAGSchedulerから発火するイベント通知である。

### 本通知の処理概要

SparkListenerTaskStartは、Sparkジョブ内の個別タスクが実行を開始したことをリスナーに通知するイベントである。

**業務上の目的・背景**：Sparkアプリケーションのタスクレベルの進捗追跡に必要である。Spark UIのステージ詳細画面でタスク一覧の更新、実行中タスク数のカウント、タスクのExecutor割り当て情報の表示などに利用される。動的リソース割り当て（ExecutorAllocationManager）もこのイベントを利用して実行中タスク数を追跡し、Executorのスケーリング判断を行う。

**通知の送信タイミング**：DAGScheduler内の`handleBeginEvent`メソッドで、TaskSetManagerからタスクの実行開始が通知された際に発火する（DAGScheduler.scala L1293）。TaskSetManagerがExecutorにタスクを割り当てた後、Executorがタスクの実行を開始した際にドライバ側で受信される。

**通知の受信者**：SparkListenerInterfaceを実装し、LiveListenerBusに登録されたすべてのリスナーが受信する。

**通知内容の概要**：stageId（ステージID）、stageAttemptId（ステージアテンプト番号）、taskInfo（TaskInfo: タスクID、インデックス、Executor情報、開始時刻など）を含む。

**期待されるアクション**：受信したリスナーは、タスクの実行状態をUIに反映する、イベントログに記録する、実行中タスク数をカウントする、動的リソース割り当ての判断材料とする。

## 通知種別

アプリ内通知（Sparkイベントバス経由の非同期インプロセス通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（LiveListenerBus経由、AsyncEventQueueで別スレッド配信） |
| 優先度 | 中 |
| リトライ | 無し |

### 送信先決定ロジック

LiveListenerBusに登録された全リスナーに対してブロードキャスト配信される。

## 通知テンプレート

### メール通知の場合

本イベントはメール通知ではなく、Sparkイベントバス経由のインプロセス通知である。

### 本文テンプレート

```
イベントクラス: SparkListenerTaskStart
シリアライズ形式: JSON（Jackson, @JsonTypeInfo）

{
  "Event": "org.apache.spark.scheduler.SparkListenerTaskStart",
  "Stage ID": <int>,
  "Stage Attempt ID": <int>,
  "Task Info": { ... }
}
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| 該当なし | - | - | インプロセス通知のため添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| stageId | ステージID | task.stageId | Yes |
| stageAttemptId | ステージアテンプト番号 | task.stageAttemptId | Yes |
| taskInfo | タスク詳細情報（taskId, index, executorId, host, launchTime等） | TaskInfo | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| スケジューラ内部処理 | DAGScheduler.handleBeginEvent | タスクの実行開始時 | DAGScheduler.scala L1292-1293 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| LiveListenerBus停止後 | イベントは配信されない |
| キュー容量超過 | イベントはドロップされる |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[Executorがタスク実行開始] --> B[TaskSetManager.handleTaskStart]
    B --> C[DAGScheduler.handleBeginEvent]
    C --> D[SparkListenerTaskStart生成]
    D --> E[listenerBus.post]
    E --> F[AsyncEventQueue経由で非同期配信]
    F --> G[各リスナーのonTaskStart呼び出し]
    G --> H[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| KVStore | AppStatusListenerがタスク情報を格納 | Spark UIバックエンドストア |

### テーブル別参照項目詳細

#### KVStore

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| TaskDataWrapper | タスクの実行状態情報 | taskIdで検索 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| KVStore | INSERT | AppStatusListenerがTaskDataWrapperを新規作成 |
| イベントログファイル | APPEND | EventLoggingListenerがJSON形式でイベントを追記 |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| INSERT | TaskDataWrapper | タスク状態=RUNNING | AppStatusListenerによる新規作成 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| キューオーバーフロー | イベント生成速度がリスナー処理速度を上回った場合 | イベントドロップ |
| リスナー例外 | onTaskStart内で例外発生 | 例外キャッチ、ログ出力、他リスナーへの配信継続 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0 |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| キュー容量 | デフォルト10,000イベント |

### 配信時間帯

時間帯制限なし。

## セキュリティ考慮事項

- TaskInfoにはexecutorId、host等のインフラ情報が含まれる
- イベントログファイルのアクセス制御が必要

## 備考

- 大規模ジョブでは大量のTaskStartイベントが短時間に発火されるため、リスナーの処理性能がボトルネックになる可能性がある
- SparkListenerTaskStartは3つのフィールド（stageId: Int, stageAttemptId: Int, taskInfo: TaskInfo）を持つcase classである（SparkListener.scala L40-41）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | L40-41: `case class SparkListenerTaskStart(stageId: Int, stageAttemptId: Int, taskInfo: TaskInfo)`。3フィールドの定義 |

**読解のコツ**: TaskInfoクラスの構造（taskId, index, attemptNumber, partitionId, launchTime, executorId, host等）も併せて確認する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | DAGScheduler.scala | `core/src/main/scala/org/apache/spark/scheduler/DAGScheduler.scala` | L1292-1293: handleBeginEvent。task.stageIdとtask.stageAttemptIdとtaskInfoを引数にSparkListenerTaskStartを生成 |

**主要処理フロー**:
1. **L1292**: `handleBeginEvent(task: Task[_], taskInfo: TaskInfo)`メソッド
2. **L1293**: `listenerBus.post(SparkListenerTaskStart(task.stageId, task.stageAttemptId, taskInfo))`

#### Step 3: イベントディスパッチを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | L40-41: SparkListenerTaskStartのパターンマッチ → listener.onTaskStart()呼び出し |

### プログラム呼び出し階層図

```
TaskSetManager (タスク割り当て通知)
    |
    +-- DAGScheduler.handleBeginEvent() [DAGScheduler.scala L1292]
            |
            +-- listenerBus.post(SparkListenerTaskStart(...))
                    |
                    +-- LiveListenerBus.post()
                            |
                            +-- AsyncEventQueue.post() [各キュー]
                                    |
                                    +-- listener.onTaskStart()
```

### データフロー図

```
[入力]                     [処理]                         [出力]

TaskSetManager          LiveListenerBus               各リスナー
(Task, TaskInfo)       (AsyncEventQueue x N)

stageId ------------> SparkListenerTaskStart -------> AppStatusListener
stageAttemptId                                        (KVStore更新)
taskInfo                                              EventLoggingListener
                                                      (イベントログ)
                                                      ExecutorAllocationManager
                                                      (タスク数追跡)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | ソース | SparkListenerTaskStartケースクラス定義 |
| DAGScheduler.scala | `core/src/main/scala/org/apache/spark/scheduler/DAGScheduler.scala` | ソース | handleBeginEventからのイベント発火 |
| SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | ソース | イベントディスパッチ |
| LiveListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/LiveListenerBus.scala` | ソース | 非同期配信制御 |
| AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | ソース | UI用ステータス更新 |
