# 通知設計書 42-CreateDatabasePreEvent

## 概要

本ドキュメントは、Apache SparkのSQL外部カタログにおけるデータベース作成前イベント `CreateDatabasePreEvent` の通知設計を記載する。このイベントは外部カタログでデータベースが作成される直前に発火され、リスナーに事前通知を提供する。

### 本通知の処理概要

`CreateDatabasePreEvent` は、`ExternalCatalogWithListener` を通じてデータベースの作成操作が実行される直前に発火されるイベント通知である。外部カタログへの変更を監視するリスナーに対し、操作が実行される前にインターセプトする機会を提供する。

**業務上の目的・背景**：Sparkの外部カタログ（Hive Metastoreなど）に対するメタデータ変更は、データパイプラインやデータガバナンスに大きな影響を与える。データベース作成前にイベントを発火することで、監査ログの記録、アクセス制御の検証、命名規則の検証など、操作実行前の事前チェックやフック処理を実装可能にする。Pre/Post パターンにより、操作の失敗時（例：既に同名データベースが存在する場合）にはPreEventのみが発火しPostEventは発火しないことで、操作の成否を区別できる。

**通知の送信タイミング**：`ExternalCatalogWithListener.createDatabase()` メソッド内で、委譲先の `delegate.createDatabase()` が呼び出される直前に `postToAll(CreateDatabasePreEvent(db))` により発火される。

**通知の受信者**：`ExternalCatalogEventListener` インタフェースを実装し、`ExternalCatalogWithListener` にリスナーとして登録されたすべてのオブジェクトが受信する。ListenerBusの `postToAll` メソッドにより全登録リスナーにブロードキャスト配信される。

**通知内容の概要**：作成対象のデータベース名（`database: String`）を含む。`DatabaseEvent` トレイトを継承し、`ExternalCatalogEvent` および `SparkListenerEvent` を間接的に継承する。

**期待されるアクション**：受信したリスナーは、データベース作成前の事前検証（命名規則チェック、権限チェックなど）、監査ログの記録、外部システムへの通知などを実行する。操作をブロックする仕組みは提供されていないが、リスナー側で例外をスローすることで操作を中断させることは可能である。

## 通知種別

アプリ内イベント通知（Spark ListenerBus経由のインプロセス同期通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ListenerBus.postToAll経由） |
| 優先度 | 高（操作実行前のため、全リスナーへの配信完了後に操作が実行される） |
| リトライ | 無（同期呼び出しのため） |

### 送信先決定ロジック

`ExternalCatalogWithListener` に `addListener()` メソッドで登録された全ての `ExternalCatalogEventListener` に対して、`postToAll` により同期的にブロードキャスト配信される。リスナーの `onEvent(event: ExternalCatalogEvent)` メソッドが呼び出される。

## 通知テンプレート

### メール通知の場合

本通知はメール通知ではなく、Sparkアプリ内のListenerBusを通じたインプロセス通知であるため、メールテンプレートは該当しない。

### 本文テンプレート

```
CreateDatabasePreEvent(
  database: String  // 作成対象のデータベース名
)
```

### 添付ファイル

該当なし（インプロセスイベント通知のため）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| database | 作成対象のデータベース名 | CatalogDatabase.name | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | ExternalCatalogWithListener.createDatabase() | 常に発火（ignoreIfExistsの値に関係なく） | delegate.createDatabase()呼び出しの直前に発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | createDatabase()が呼び出されれば常にPreEventは発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ExternalCatalogWithListener.createDatabase呼び出し] --> B[dbDefinition.nameからデータベース名取得]
    B --> C[postToAll - CreateDatabasePreEvent発火]
    C --> D[全リスナーのonEventを同期呼び出し]
    D --> E[delegate.createDatabase実行]
    E --> F{作成成功?}
    F -->|成功| G[postToAll - CreateDatabaseEvent発火]
    F -->|失敗 - 例外| H[CreateDatabaseEventは発火しない]
    G --> I[終了]
    H --> I
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| CatalogDatabase | 作成対象データベース定義の参照 | name, description, locationUri, propertiesを含む |

### テーブル別参照項目詳細

#### CatalogDatabase

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| name | イベントのdatabaseフィールドに設定 | createDatabaseメソッドの引数から取得 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | PreEventの発火時点ではデータ更新は行われない |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| 該当なし | - | - | イベント発火のみでログテーブルへの書き込みは行わない |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonEvent内で例外がスローされた場合 | ListenerBusがキャッチしログに記録。後続リスナーへの配信は継続される |
| PreEvent発火後の操作失敗 | delegate.createDatabase()が例外をスロー（既存DB等） | PreEventは発火済みだがPostEvent（CreateDatabaseEvent）は発火しない |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。データベース作成操作時に随時発火される。

## セキュリティ考慮事項

本イベントはインプロセスで配信されるため、外部ネットワーク通信は発生しない。データベース名のみを含むため、直接的な機密情報の漏洩リスクは低い。ただし、データベース名から業務内容を推測できる可能性があるため、カスタムリスナーの実装時は監査ログの保管場所やアクセス制御に注意が必要。`ExternalCatalogEvent` は `SparkListenerEvent` を継承しているため、イベントログへの記録対象となりうる。

## 備考

- `CreateDatabasePreEvent` は `DatabaseEvent` トレイトを継承し、`ExternalCatalogEvent` および `SparkListenerEvent` を間接的に継承する
- Pre/Post イベントのペアにより、操作の成否を追跡可能。操作が失敗した場合はPreEventのみが発火する
- テストスイート `ExternalCatalogEventSuite` で、作成成功時と失敗時の双方のイベント発火パターンが検証されている
- `ignoreIfExists = true` の場合でも、既存DBが存在する場合のPreEventとPostEventの両方が発火する

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、イベントクラスの継承階層とデータ構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `ExternalCatalogEvent` トレイト（26行目）、`DatabaseEvent` トレイト（38-43行目）、`CreateDatabasePreEvent` ケースクラス（48行目）の継承関係を理解する |

**読解のコツ**: Scalaのトレイト継承により、`CreateDatabasePreEvent` は `DatabaseEvent` -> `ExternalCatalogEvent` -> `SparkListenerEvent` の階層を持つ。ケースクラスのパラメータ `database: String` がトレイトの抽象メンバを実装している。

#### Step 2: エントリーポイントを理解する

イベント発火の起点である `ExternalCatalogWithListener.createDatabase` を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `createDatabase` メソッド（44-49行目）。Pre/Postイベントの発火順序とdelegate呼び出しの関係 |

**主要処理フロー**:
1. **45行目**: `val db = dbDefinition.name` でデータベース名を取得
2. **46行目**: `postToAll(CreateDatabasePreEvent(db))` でPreイベントを発火
3. **47行目**: `delegate.createDatabase(dbDefinition, ignoreIfExists)` で実際の作成操作を委譲
4. **48行目**: `postToAll(CreateDatabaseEvent(db))` でPostイベントを発火（成功時のみ到達）

#### Step 3: ListenerBusの配信メカニズムを理解する

ListenerBusによるイベント配信の仕組みを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | クラス定義（27-29行目）。`ListenerBus[ExternalCatalogEventListener, ExternalCatalogEvent]` の継承 |
| 3-2 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `doPostEvent` メソッド（34-38行目）。各リスナーの `onEvent` メソッドを呼び出す |

**主要処理フロー**:
- **34行目**: `doPostEvent` はListenerBusの抽象メソッドの実装
- **37行目**: `listener.onEvent(event)` で各リスナーにイベントを配信

#### Step 4: テストケースで動作を確認する

テストスイートでイベントの発火パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | databaseテストケース（59-90行目）。成功時はPre+Post、失敗時はPreのみの発火パターン |

**主要処理フロー**:
- **63-64行目**: 作成成功時 `CreateDatabasePreEvent("db5") :: CreateDatabaseEvent("db5") :: Nil`
- **71-72行目**: 作成失敗時（既存DB） `CreateDatabasePreEvent("db5") :: Nil`（Preのみ）

### プログラム呼び出し階層図

```
ExternalCatalogWithListener.createDatabase() [ExternalCatalogWithListener.scala:44]
    |
    +-- postToAll(CreateDatabasePreEvent(db)) [ExternalCatalogWithListener.scala:46]
    |       |
    |       +-- doPostEvent() [ExternalCatalogWithListener.scala:34]
    |               |
    |               +-- ExternalCatalogEventListener.onEvent()
    |
    +-- delegate.createDatabase() [ExternalCatalogWithListener.scala:47]
    |
    +-- postToAll(CreateDatabaseEvent(db)) [ExternalCatalogWithListener.scala:48]
```

### データフロー図

```
[入力]                          [処理]                              [出力]

CatalogDatabase -------> ExternalCatalogWithListener -------> CreateDatabasePreEvent
  (dbDefinition)              .createDatabase()                     |
                                    |                               v
                                    |                    ExternalCatalogEventListener
                                    |                          .onEvent()
                                    v
                              delegate.createDatabase()
                                    |
                                    v
                            CreateDatabaseEvent (成功時のみ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | ソース | CreateDatabasePreEventイベントクラスの定義 |
| ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | ソース | イベント発火のラッパークラス |
| ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | テスト | イベント発火パターンのテスト |
