# 通知設計書 43-CreateDatabaseEvent

## 概要

本ドキュメントは、Apache SparkのSQL外部カタログにおけるデータベース作成後イベント `CreateDatabaseEvent` の通知設計を記載する。このイベントは外部カタログでデータベースが正常に作成された後に発火され、リスナーに事後通知を提供する。

### 本通知の処理概要

`CreateDatabaseEvent` は、`ExternalCatalogWithListener` を通じてデータベースの作成操作が正常に完了した直後に発火されるイベント通知である。対応する `CreateDatabasePreEvent` と対になって動作し、操作の成功を確認するためのイベントである。

**業務上の目的・背景**：データベース作成操作の成功をリスナーに通知することで、監査ログの完了記録、メタデータキャッシュの更新、依存サービスへの通知、データカタログツールとの同期などの後処理を実装可能にする。PreEventと組み合わせることで、操作の開始と完了を追跡でき、操作が失敗した場合（PostEventが発火しない）との区別が可能となる。

**通知の送信タイミング**：`ExternalCatalogWithListener.createDatabase()` メソッド内で、委譲先の `delegate.createDatabase()` が正常に完了した直後に `postToAll(CreateDatabaseEvent(db))` により発火される。delegate呼び出しで例外が発生した場合はこのイベントは発火しない。

**通知の受信者**：`ExternalCatalogEventListener` インタフェースを実装し、`ExternalCatalogWithListener` にリスナーとして登録されたすべてのオブジェクトが受信する。ListenerBusの `postToAll` メソッドにより全登録リスナーにブロードキャスト配信される。

**通知内容の概要**：作成されたデータベース名（`database: String`）を含む。`DatabaseEvent` トレイトを継承し、`ExternalCatalogEvent` および `SparkListenerEvent` を間接的に継承する。

**期待されるアクション**：受信したリスナーは、データベース作成完了の監査ログ記録、メタデータキャッシュの無効化・更新、外部データカタログとの同期、依存するETLパイプラインへの通知などを実行する。このイベントの受信は操作が成功したことを意味する。

## 通知種別

アプリ内イベント通知（Spark ListenerBus経由のインプロセス同期通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ListenerBus.postToAll経由） |
| 優先度 | 中（操作完了後の通知） |
| リトライ | 無（同期呼び出しのため） |

### 送信先決定ロジック

`ExternalCatalogWithListener` に `addListener()` メソッドで登録された全ての `ExternalCatalogEventListener` に対して、`postToAll` により同期的にブロードキャスト配信される。リスナーの `onEvent(event: ExternalCatalogEvent)` メソッドが呼び出される。

## 通知テンプレート

### メール通知の場合

本通知はメール通知ではなく、Sparkアプリ内のListenerBusを通じたインプロセス通知であるため、メールテンプレートは該当しない。

### 本文テンプレート

```
CreateDatabaseEvent(
  database: String  // 作成されたデータベース名
)
```

### 添付ファイル

該当なし（インプロセスイベント通知のため）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| database | 作成されたデータベース名 | CatalogDatabase.name | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | ExternalCatalogWithListener.createDatabase() | delegate.createDatabase()が正常完了した場合 | 操作成功時のみ発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| delegate.createDatabase()が例外をスロー | データベース作成が失敗した場合（ignoreIfExists=falseで既存DB存在時など）はPostEventは発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ExternalCatalogWithListener.createDatabase呼び出し] --> B[postToAll - CreateDatabasePreEvent発火]
    B --> C[delegate.createDatabase実行]
    C --> D{作成成功?}
    D -->|成功| E[postToAll - CreateDatabaseEvent発火]
    D -->|失敗 - 例外| F[CreateDatabaseEventは発火しない]
    E --> G[全リスナーのonEventを同期呼び出し]
    G --> H[終了]
    F --> H
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| CatalogDatabase | 作成されたデータベース定義の参照 | name, description, locationUri, propertiesを含む |

### テーブル別参照項目詳細

#### CatalogDatabase

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| name | イベントのdatabaseフィールドに設定 | createDatabaseメソッドの引数から取得 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 外部カタログ（Hive Metastore等） | CREATE | delegate.createDatabase()によるデータベース作成（イベント発火前に完了済み） |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| 該当なし | - | - | イベント発火のみでログテーブルへの書き込みは行わない |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonEvent内で例外がスローされた場合 | ListenerBusがキャッチしログに記録。後続リスナーへの配信は継続される |
| イベント未発火 | delegate.createDatabase()が例外をスロー | CreateDatabaseEventは発火しない。PreEventのみが記録される |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。データベース作成操作の成功時に随時発火される。

## セキュリティ考慮事項

本イベントはインプロセスで配信されるため、外部ネットワーク通信は発生しない。データベース名のみを含むため、直接的な機密情報の漏洩リスクは低い。ただし、イベントの受信はデータベースの存在と作成タイミングを確認できることを意味するため、カスタムリスナーの実装時はアクセス制御に注意が必要。

## 備考

- `CreateDatabaseEvent` は `CreateDatabasePreEvent` と対になるPostイベントである
- 操作成功時のみ発火するため、このイベントの受信はデータベース作成が成功したことの証拠となる
- `ignoreIfExists = true` の場合、既存DBがあっても例外はスローされず、Pre/Post両方のイベントが発火する
- テストスイート `ExternalCatalogEventSuite` で発火パターンが検証されている

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `CreateDatabaseEvent` ケースクラス（53行目）。`DatabaseEvent` トレイト（38-43行目）からの継承関係 |

**読解のコツ**: `CreateDatabasePreEvent`（48行目）と `CreateDatabaseEvent`（53行目）は同じ `DatabaseEvent` トレイトを継承し、同じ `database` フィールドを持つ。Preは操作前、Postは操作後を示す名前規則である。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `createDatabase` メソッド（44-49行目）。48行目のPostイベント発火に注目。delegate成功後にのみ到達する |

**主要処理フロー**:
1. **46行目**: `postToAll(CreateDatabasePreEvent(db))` - Preイベント発火
2. **47行目**: `delegate.createDatabase(dbDefinition, ignoreIfExists)` - 実際の操作実行
3. **48行目**: `postToAll(CreateDatabaseEvent(db))` - Postイベント発火（成功時のみ到達）

#### Step 3: テストケースで動作を確認する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | databaseテストケース（59-90行目）。63-64行目で成功時のPre+Post発火を確認 |

### プログラム呼び出し階層図

```
ExternalCatalogWithListener.createDatabase() [ExternalCatalogWithListener.scala:44]
    |
    +-- postToAll(CreateDatabasePreEvent(db)) [46行目]
    |
    +-- delegate.createDatabase() [47行目]
    |
    +-- postToAll(CreateDatabaseEvent(db)) [48行目] ← 本イベント
            |
            +-- doPostEvent() [34行目]
                    |
                    +-- ExternalCatalogEventListener.onEvent()
```

### データフロー図

```
[入力]                          [処理]                              [出力]

CatalogDatabase -------> ExternalCatalogWithListener -------> CreateDatabasePreEvent (操作前)
  (dbDefinition)              .createDatabase()                     |
                                    |                               v
                                    +-- delegate.createDatabase() --+
                                    |                               |
                                    v                               v
                              (操作成功時のみ)              CreateDatabaseEvent (操作後)
                                                                    |
                                                                    v
                                                         ExternalCatalogEventListener
                                                                .onEvent()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | ソース | CreateDatabaseEventイベントクラスの定義 |
| ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | ソース | イベント発火のラッパークラス |
| ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | テスト | イベント発火パターンのテスト |
