# 通知設計書 44-DropDatabasePreEvent

## 概要

本ドキュメントは、Apache SparkのSQL外部カタログにおけるデータベース削除前イベント `DropDatabasePreEvent` の通知設計を記載する。このイベントは外部カタログでデータベースが削除される直前に発火され、リスナーに事前通知を提供する。

### 本通知の処理概要

`DropDatabasePreEvent` は、`ExternalCatalogWithListener` を通じてデータベースの削除操作が実行される直前に発火されるイベント通知である。データベース削除という破壊的操作に対して、リスナーが事前に検知・対応する機会を提供する。

**業務上の目的・背景**：データベースの削除はメタデータストアに対する重大な変更操作であり、関連するテーブル、ビュー、関数などのすべてのオブジェクトに影響を及ぼす可能性がある。削除前にイベントを発火することで、依存オブジェクトの確認、バックアップの作成、監査ログの記録、削除の承認プロセスの実装など、事前対策を可能にする。Pre/Postパターンにより、削除操作が失敗した場合（存在しないDBの削除等）にはPreEventのみが発火しPostEventは発火しない。

**通知の送信タイミング**：`ExternalCatalogWithListener.dropDatabase()` メソッド内で、委譲先の `delegate.dropDatabase()` が呼び出される直前に `postToAll(DropDatabasePreEvent(db))` により発火される。

**通知の受信者**：`ExternalCatalogEventListener` インタフェースを実装し、`ExternalCatalogWithListener` にリスナーとして登録されたすべてのオブジェクトが受信する。

**通知内容の概要**：削除対象のデータベース名（`database: String`）を含む。`DatabaseEvent` トレイトを継承し、`ExternalCatalogEvent` および `SparkListenerEvent` を間接的に継承する。

**期待されるアクション**：受信したリスナーは、削除対象データベースの依存関係確認、バックアップの作成トリガー、監査ログの記録、外部サービスへの事前通知などを実行する。

## 通知種別

アプリ内イベント通知（Spark ListenerBus経由のインプロセス同期通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ListenerBus.postToAll経由） |
| 優先度 | 高（破壊的操作の事前通知） |
| リトライ | 無（同期呼び出しのため） |

### 送信先決定ロジック

`ExternalCatalogWithListener` に `addListener()` メソッドで登録された全ての `ExternalCatalogEventListener` に対して、`postToAll` により同期的にブロードキャスト配信される。

## 通知テンプレート

### メール通知の場合

本通知はメール通知ではなく、Sparkアプリ内のListenerBusを通じたインプロセス通知であるため、メールテンプレートは該当しない。

### 本文テンプレート

```
DropDatabasePreEvent(
  database: String  // 削除対象のデータベース名
)
```

### 添付ファイル

該当なし（インプロセスイベント通知のため）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| database | 削除対象のデータベース名 | dropDatabaseメソッドのdb引数 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | ExternalCatalogWithListener.dropDatabase() | 常に発火（ignoreIfNotExistsの値に関係なく） | delegate.dropDatabase()呼び出しの直前に発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | dropDatabase()が呼び出されれば常にPreEventは発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ExternalCatalogWithListener.dropDatabase呼び出し] --> B[postToAll - DropDatabasePreEvent発火]
    B --> C[全リスナーのonEventを同期呼び出し]
    C --> D[delegate.dropDatabase実行]
    D --> E{削除成功?}
    E -->|成功| F[postToAll - DropDatabaseEvent発火]
    E -->|失敗 - 例外| G[DropDatabaseEventは発火しない]
    F --> H[終了]
    G --> H
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| 該当なし | - | PreEventの発火には追加のデータ参照は不要 |

### テーブル別参照項目詳細

該当なし。dropDatabaseメソッドの引数から直接データベース名を取得する。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | PreEventの発火時点ではデータ更新は行われない |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| 該当なし | - | - | イベント発火のみでログテーブルへの書き込みは行わない |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonEvent内で例外がスローされた場合 | ListenerBusがキャッチしログに記録 |
| PreEvent発火後の操作失敗 | delegate.dropDatabase()が例外をスロー（存在しないDB等） | PreEventは発火済みだがPostEventは発火しない |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。データベース削除操作時に随時発火される。

## セキュリティ考慮事項

本イベントはインプロセスで配信されるため、外部ネットワーク通信は発生しない。データベース名のみを含むため直接的な機密情報の漏洩リスクは低い。ただし、データベース削除は破壊的操作であるため、カスタムリスナーによる監査ログ記録やアラート通知の実装が推奨される。

## 備考

- `DropDatabasePreEvent` は `DropDatabaseEvent` と対になるPreイベントである
- テストスイート `ExternalCatalogEventSuite` では、存在しないDB削除時（ignoreIfNotExists=false）にPreEventのみ発火するパターンが検証されている（80-83行目）
- `ignoreIfNotExists = true` の場合、存在しないDBでもPre/Post両方のイベントが発火する（88-89行目）
- `cascade` パラメータにより、テーブルを含むDBの削除可否が制御されるが、イベントにはcascade情報は含まれない

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `DropDatabasePreEvent` ケースクラス（58行目）。`DatabaseEvent` トレイト（38-43行目）からの継承 |

**読解のコツ**: DropDatabase系イベントはCreateDatabase系と同じ `DatabaseEvent` トレイトを継承し、`database: String` フィールドのみを含む単純な構造である。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `dropDatabase` メソッド（51-55行目）。引数db, ignoreIfNotExists, cascadeを受け取り、Pre/Postイベントを発火 |

**主要処理フロー**:
1. **52行目**: `postToAll(DropDatabasePreEvent(db))` - Preイベント発火
2. **53行目**: `delegate.dropDatabase(db, ignoreIfNotExists, cascade)` - 実際の削除操作を委譲
3. **54行目**: `postToAll(DropDatabaseEvent(db))` - Postイベント発火（成功時のみ到達）

#### Step 3: テストケースで動作を確認する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | databaseテストケース（80-89行目）。存在しないDB削除時のPreのみ発火、存在するDB削除時のPre+Post発火パターン |

### プログラム呼び出し階層図

```
ExternalCatalogWithListener.dropDatabase() [ExternalCatalogWithListener.scala:51]
    |
    +-- postToAll(DropDatabasePreEvent(db)) [52行目] ← 本イベント
    |       |
    |       +-- doPostEvent() [34行目]
    |               |
    |               +-- ExternalCatalogEventListener.onEvent()
    |
    +-- delegate.dropDatabase(db, ignoreIfNotExists, cascade) [53行目]
    |
    +-- postToAll(DropDatabaseEvent(db)) [54行目]
```

### データフロー図

```
[入力]                          [処理]                              [出力]

db: String ------------> ExternalCatalogWithListener -------> DropDatabasePreEvent
ignoreIfNotExists              .dropDatabase()                      |
cascade                             |                               v
                                    |                    ExternalCatalogEventListener
                                    v                          .onEvent()
                              delegate.dropDatabase()
                                    |
                                    v
                            DropDatabaseEvent (成功時のみ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | ソース | DropDatabasePreEventイベントクラスの定義 |
| ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | ソース | イベント発火のラッパークラス |
| ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | テスト | イベント発火パターンのテスト |
