# 通知設計書 45-DropDatabaseEvent

## 概要

本ドキュメントは、Apache SparkのSQL外部カタログにおけるデータベース削除後イベント `DropDatabaseEvent` の通知設計を記載する。このイベントは外部カタログでデータベースが正常に削除された後に発火され、リスナーに事後通知を提供する。

### 本通知の処理概要

`DropDatabaseEvent` は、`ExternalCatalogWithListener` を通じてデータベースの削除操作が正常に完了した直後に発火されるイベント通知である。対応する `DropDatabasePreEvent` と対になって動作し、削除操作の成功を確認するためのイベントである。

**業務上の目的・背景**：データベースの削除は破壊的操作であり、関連するメタデータの整合性維持が重要である。削除完了後にイベントを発火することで、メタデータキャッシュのクリア、依存サービスへの通知、監査ログの完了記録、データカタログツールとの同期などの後処理を確実に実行可能にする。PreEventと組み合わせることで、操作の成否を追跡でき、ロールバックや補償処理の判断材料となる。

**通知の送信タイミング**：`ExternalCatalogWithListener.dropDatabase()` メソッド内で、委譲先の `delegate.dropDatabase()` が正常に完了した直後に `postToAll(DropDatabaseEvent(db))` により発火される。

**通知の受信者**：`ExternalCatalogEventListener` インタフェースを実装し、`ExternalCatalogWithListener` にリスナーとして登録されたすべてのオブジェクトが受信する。

**通知内容の概要**：削除されたデータベース名（`database: String`）を含む。`DatabaseEvent` トレイトを継承する。

**期待されるアクション**：受信したリスナーは、メタデータキャッシュの無効化、依存するETLパイプラインの停止・更新トリガー、監査ログの完了記録、外部データカタログからの該当データベースの削除などを実行する。

## 通知種別

アプリ内イベント通知（Spark ListenerBus経由のインプロセス同期通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ListenerBus.postToAll経由） |
| 優先度 | 中（操作完了後の通知） |
| リトライ | 無（同期呼び出しのため） |

### 送信先決定ロジック

`ExternalCatalogWithListener` に登録された全ての `ExternalCatalogEventListener` に対して、`postToAll` により同期的にブロードキャスト配信される。

## 通知テンプレート

### メール通知の場合

本通知はインプロセスイベント通知であるため、メールテンプレートは該当しない。

### 本文テンプレート

```
DropDatabaseEvent(
  database: String  // 削除されたデータベース名
)
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| database | 削除されたデータベース名 | dropDatabaseメソッドのdb引数 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | ExternalCatalogWithListener.dropDatabase() | delegate.dropDatabase()が正常完了した場合 | 操作成功時のみ発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| delegate.dropDatabase()が例外をスロー | データベース削除が失敗した場合はPostEventは発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ExternalCatalogWithListener.dropDatabase呼び出し] --> B[postToAll - DropDatabasePreEvent発火]
    B --> C[delegate.dropDatabase実行]
    C --> D{削除成功?}
    D -->|成功| E[postToAll - DropDatabaseEvent発火]
    D -->|失敗 - 例外| F[DropDatabaseEventは発火しない]
    E --> G[全リスナーのonEventを同期呼び出し]
    G --> H[終了]
    F --> H
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| 該当なし | - | PostEvent発火にはデータ参照不要 |

### テーブル別参照項目詳細

該当なし。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 外部カタログ（Hive Metastore等） | DROP | delegate.dropDatabase()によるデータベース削除（イベント発火前に完了済み） |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| 該当なし | - | - | イベント発火のみ |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonEvent内で例外がスローされた場合 | ListenerBusがキャッチしログに記録 |
| イベント未発火 | delegate.dropDatabase()が例外をスロー | PostEventは発火しない。PreEventのみが記録される |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。

## セキュリティ考慮事項

本イベントはインプロセスで配信されるため、外部ネットワーク通信は発生しない。データベース名のみを含むため機密情報漏洩リスクは低い。カスタムリスナーで監査ログを記録する場合は、ログの保管場所やアクセス制御に注意が必要。

## 備考

- `DropDatabaseEvent` は `DropDatabasePreEvent` と対になるPostイベントである
- 操作成功時のみ発火するため、このイベントの受信はデータベース削除が成功したことの証拠となる
- `ignoreIfNotExists = true` の場合、存在しないDBでもPre/Post両方のイベントが発火する
- テストスイート `ExternalCatalogEventSuite`（85-89行目）で発火パターンが検証されている

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `DropDatabaseEvent` ケースクラス（63行目）。`DatabaseEvent` トレイトからの継承 |

**読解のコツ**: `DropDatabasePreEvent`（58行目）と `DropDatabaseEvent`（63行目）は同じ `DatabaseEvent` トレイトを継承する対イベント。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `dropDatabase` メソッド（51-55行目）。54行目のPostイベント発火に注目 |

**主要処理フロー**:
1. **52行目**: `postToAll(DropDatabasePreEvent(db))` - Preイベント発火
2. **53行目**: `delegate.dropDatabase(db, ignoreIfNotExists, cascade)` - 実際の削除
3. **54行目**: `postToAll(DropDatabaseEvent(db))` - Postイベント発火（本イベント）

#### Step 3: テストケースで動作を確認する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | databaseテストケース（85-89行目）。成功時のPre+Post発火パターン |

### プログラム呼び出し階層図

```
ExternalCatalogWithListener.dropDatabase() [ExternalCatalogWithListener.scala:51]
    |
    +-- postToAll(DropDatabasePreEvent(db)) [52行目]
    |
    +-- delegate.dropDatabase() [53行目]
    |
    +-- postToAll(DropDatabaseEvent(db)) [54行目] ← 本イベント
            |
            +-- doPostEvent() [34行目]
                    |
                    +-- ExternalCatalogEventListener.onEvent()
```

### データフロー図

```
[入力]                          [処理]                              [出力]

db: String ------------> ExternalCatalogWithListener -------> DropDatabasePreEvent (操作前)
ignoreIfNotExists              .dropDatabase()                      |
cascade                             |                               v
                                    +-- delegate.dropDatabase() ----+
                                    |                               |
                                    v                               v
                              (操作成功時のみ)              DropDatabaseEvent (操作後)
                                                                    |
                                                                    v
                                                         ExternalCatalogEventListener
                                                                .onEvent()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | ソース | DropDatabaseEventイベントクラスの定義 |
| ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | ソース | イベント発火のラッパークラス |
| ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | テスト | イベント発火パターンのテスト |
