# 通知設計書 47-AlterDatabaseEvent

## 概要

本ドキュメントは、Apache SparkのSQL外部カタログにおけるデータベース変更後イベント `AlterDatabaseEvent` の通知設計を記載する。このイベントは外部カタログでデータベースの定義が正常に変更された後に発火され、リスナーに事後通知を提供する。

### 本通知の処理概要

`AlterDatabaseEvent` は、`ExternalCatalogWithListener` を通じてデータベースの定義変更操作（ALTER DATABASE）が正常に完了した直後に発火されるイベント通知である。対応する `AlterDatabasePreEvent` と対になって動作し、変更操作の成功を確認するためのイベントである。

**業務上の目的・背景**：データベース定義の変更が成功したことをリスナーに通知することで、メタデータキャッシュの更新、データカタログとの同期、監査ログの完了記録、依存サービスへの通知などの後処理を確実に実行可能にする。PreEventと組み合わせることで変更操作の成否を追跡できる。

**通知の送信タイミング**：`ExternalCatalogWithListener.alterDatabase()` メソッド内で、委譲先の `delegate.alterDatabase()` が正常に完了した直後に `postToAll(AlterDatabaseEvent(db))` により発火される。

**通知の受信者**：`ExternalCatalogEventListener` インタフェースを実装し、`ExternalCatalogWithListener` にリスナーとして登録されたすべてのオブジェクトが受信する。

**通知内容の概要**：変更されたデータベース名（`database: String`）を含む。変更内容の詳細はイベントに含まれない。`DatabaseEvent` トレイトを継承する。

**期待されるアクション**：受信したリスナーは、メタデータキャッシュの更新・無効化、外部データカタログとの同期、監査ログの完了記録、変更内容に基づく後続処理のトリガーなどを実行する。

## 通知種別

アプリ内イベント通知（Spark ListenerBus経由のインプロセス同期通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ListenerBus.postToAll経由） |
| 優先度 | 中（操作完了後の通知） |
| リトライ | 無（同期呼び出しのため） |

### 送信先決定ロジック

`ExternalCatalogWithListener` に登録された全ての `ExternalCatalogEventListener` に対して、`postToAll` により同期的にブロードキャスト配信される。

## 通知テンプレート

### メール通知の場合

本通知はインプロセスイベント通知であるため、メールテンプレートは該当しない。

### 本文テンプレート

```
AlterDatabaseEvent(
  database: String  // 変更されたデータベース名
)
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| database | 変更されたデータベース名 | CatalogDatabase.name (dbDefinition.name) | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | ExternalCatalogWithListener.alterDatabase() | delegate.alterDatabase()が正常完了した場合 | 操作成功時のみ発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| delegate.alterDatabase()が例外をスロー | 変更操作が失敗した場合はPostEventは発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ExternalCatalogWithListener.alterDatabase呼び出し] --> B[postToAll - AlterDatabasePreEvent発火]
    B --> C[delegate.alterDatabase実行]
    C --> D{変更成功?}
    D -->|成功| E[postToAll - AlterDatabaseEvent発火]
    D -->|失敗 - 例外| F[AlterDatabaseEventは発火しない]
    E --> G[全リスナーのonEventを同期呼び出し]
    G --> H[終了]
    F --> H
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| CatalogDatabase | 変更対象データベース定義 | PostEvent発火にはname参照のみ必要 |

### テーブル別参照項目詳細

#### CatalogDatabase

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| name | イベントのdatabaseフィールドに設定 | alterDatabaseメソッドの引数から取得 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 外部カタログ（Hive Metastore等） | ALTER | delegate.alterDatabase()によるDB定義変更（イベント発火前に完了済み） |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| 該当なし | - | - | イベント発火のみ |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonEvent内で例外がスローされた場合 | ListenerBusがキャッチしログに記録 |
| イベント未発火 | delegate.alterDatabase()が例外をスロー | PostEventは発火しない |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。

## セキュリティ考慮事項

本イベントはインプロセスで配信されるため、外部ネットワーク通信は発生しない。データベース名のみを含むため機密情報漏洩リスクは低い。変更内容の詳細はイベントに含まれないため、詳細な監査が必要な場合はリスナー側でカタログAPIを呼び出す必要がある。

## 備考

- `AlterDatabaseEvent` は `AlterDatabasePreEvent` と対になるPostイベントである
- 変更内容の詳細はイベントに含まれない。データベース名のみ通知される
- テストスイート `ExternalCatalogEventSuite`（75-77行目）で `AlterDatabasePreEvent("db5") :: AlterDatabaseEvent("db5") :: Nil` の発火パターンが検証されている

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `AlterDatabaseEvent` ケースクラス（73行目）。`DatabaseEvent` トレイトからの継承 |

**読解のコツ**: `AlterDatabasePreEvent`（68行目）と `AlterDatabaseEvent`（73行目）は同じ `DatabaseEvent` トレイトを継承する対イベント。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `alterDatabase` メソッド（57-62行目）。61行目のPostイベント発火に注目 |

**主要処理フロー**:
1. **58行目**: `val db = dbDefinition.name` - DB名取得
2. **59行目**: `postToAll(AlterDatabasePreEvent(db))` - Preイベント
3. **60行目**: `delegate.alterDatabase(dbDefinition)` - 実際の変更
4. **61行目**: `postToAll(AlterDatabaseEvent(db))` - Postイベント（本イベント）

#### Step 3: テストケースで動作を確認する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | databaseテストケース（75-77行目）。ALTER成功時のPre+Post発火パターン |

### プログラム呼び出し階層図

```
ExternalCatalogWithListener.alterDatabase() [ExternalCatalogWithListener.scala:57]
    |
    +-- postToAll(AlterDatabasePreEvent(db)) [59行目]
    |
    +-- delegate.alterDatabase(dbDefinition) [60行目]
    |
    +-- postToAll(AlterDatabaseEvent(db)) [61行目] ← 本イベント
            |
            +-- doPostEvent() [34行目]
                    |
                    +-- ExternalCatalogEventListener.onEvent()
```

### データフロー図

```
[入力]                          [処理]                              [出力]

CatalogDatabase -------> ExternalCatalogWithListener -------> AlterDatabasePreEvent (操作前)
  (dbDefinition)              .alterDatabase()                      |
                                    |                               v
                                    +-- delegate.alterDatabase() ---+
                                    |                               |
                                    v                               v
                              (操作成功時のみ)              AlterDatabaseEvent (操作後)
                                                                    |
                                                                    v
                                                         ExternalCatalogEventListener
                                                                .onEvent()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | ソース | AlterDatabaseEventイベントクラスの定義 |
| ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | ソース | イベント発火のラッパークラス |
| ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | テスト | イベント発火パターンのテスト |
