# 通知設計書 48-CreateTablePreEvent

## 概要

本ドキュメントは、Apache SparkのSQL外部カタログにおけるテーブル作成前イベント `CreateTablePreEvent` の通知設計を記載する。このイベントは外部カタログでテーブルが作成される直前に発火され、リスナーに事前通知を提供する。

### 本通知の処理概要

`CreateTablePreEvent` は、`ExternalCatalogWithListener` を通じてテーブルの作成操作が実行される直前に発火されるイベント通知である。テーブル作成という重要なメタデータ変更に対して、リスナーが事前に検知・対応する機会を提供する。

**業務上の目的・背景**：テーブルはSparkのデータ処理において中心的なメタデータオブジェクトであり、その作成はデータパイプライン、ETLプロセス、データガバナンスに直接影響する。テーブル作成前にイベントを発火することで、命名規則の検証、スキーマポリシーの検証、ストレージ場所の確認、アクセス制御の事前設定、監査ログの記録など、作成前の事前チェックやフック処理を実装可能にする。Pre/Postパターンにより、作成が失敗した場合にはPreEventのみが発火する。

**通知の送信タイミング**：`ExternalCatalogWithListener.createTable()` メソッド内で、委譲先の `delegate.createTable()` が呼び出される直前に `postToAll(CreateTablePreEvent(db, name))` により発火される。

**通知の受信者**：`ExternalCatalogEventListener` インタフェースを実装し、`ExternalCatalogWithListener` にリスナーとして登録されたすべてのオブジェクトが受信する。

**通知内容の概要**：テーブルが属するデータベース名（`database: String`）とテーブル名（`name: String`）を含む。`TableEvent` トレイトを継承し、`DatabaseEvent` -> `ExternalCatalogEvent` -> `SparkListenerEvent` の階層を持つ。

**期待されるアクション**：受信したリスナーは、テーブル命名規則の検証、スキーマポリシーの事前チェック、ストレージ場所の適切性確認、監査ログの記録、外部システムへの事前通知などを実行する。

## 通知種別

アプリ内イベント通知（Spark ListenerBus経由のインプロセス同期通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ListenerBus.postToAll経由） |
| 優先度 | 高（操作実行前のため） |
| リトライ | 無（同期呼び出しのため） |

### 送信先決定ロジック

`ExternalCatalogWithListener` に登録された全ての `ExternalCatalogEventListener` に対して、`postToAll` により同期的にブロードキャスト配信される。

## 通知テンプレート

### メール通知の場合

本通知はインプロセスイベント通知であるため、メールテンプレートは該当しない。

### 本文テンプレート

```
CreateTablePreEvent(
  database: String,  // テーブルが属するデータベース名
  name: String       // 作成対象のテーブル名
)
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| database | テーブルが属するデータベース名 | CatalogTable.database (tableDefinition.database) | Yes |
| name | 作成対象のテーブル名 | CatalogTable.identifier.table (tableDefinition.identifier.table) | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | ExternalCatalogWithListener.createTable() | 常に発火（ignoreIfExistsの値に関係なく） | delegate.createTable()呼び出しの直前に発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | createTable()が呼び出されれば常にPreEventは発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ExternalCatalogWithListener.createTable呼び出し] --> B[tableDefinitionからdb名とテーブル名を取得]
    B --> C[createVersionをSPARK_VERSIONに設定]
    C --> D[postToAll - CreateTablePreEvent発火]
    D --> E[全リスナーのonEventを同期呼び出し]
    E --> F[delegate.createTable実行]
    F --> G{作成成功?}
    G -->|成功| H[postToAll - CreateTableEvent発火]
    G -->|失敗 - 例外| I[CreateTableEventは発火しない]
    H --> J[終了]
    I --> J
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| CatalogTable | 作成対象テーブル定義の参照 | identifier, tableType, storage, schema等を含む |

### テーブル別参照項目詳細

#### CatalogTable

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| database | イベントのdatabaseフィールドに設定 | tableDefinition.databaseから取得 |
| identifier.table | イベントのnameフィールドに設定 | tableDefinition.identifier.tableから取得 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | PreEventの発火時点ではデータ更新は行われない |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| 該当なし | - | - | イベント発火のみ |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonEvent内で例外がスローされた場合 | ListenerBusがキャッチしログに記録 |
| PreEvent発火後の操作失敗 | delegate.createTable()が例外をスロー（既存テーブル等） | PreEventは発火済みだがPostEventは発火しない |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。テーブル作成操作時に随時発火される。

## セキュリティ考慮事項

本イベントはインプロセスで配信されるため、外部ネットワーク通信は発生しない。データベース名とテーブル名のみを含むため、直接的な機密情報の漏洩リスクは低い。ただし、テーブル名から業務データの性質を推測できる可能性があるため、カスタムリスナーの実装時はログの保管場所やアクセス制御に注意が必要。テーブル定義の詳細（スキーマ、ストレージ場所等）はイベントに含まれない。

## 備考

- `CreateTablePreEvent` は `TableEvent` トレイトを継承し、`DatabaseEvent` -> `ExternalCatalogEvent` -> `SparkListenerEvent` の階層を持つ
- `TableEvent` トレイトは `DatabaseEvent` を拡張し、`database` と `name` の2フィールドを持つ
- `ExternalCatalogWithListener.createTable()` メソッドでは、delegate呼び出し前に `createVersion` フィールドが `SPARK_VERSION` に設定される（92行目）
- テストスイート `ExternalCatalogEventSuite`（112-121行目）で、作成成功時と失敗時の発火パターンが検証されている
- `ignoreIfExists = true` で既存テーブルが存在する場合でもPre/Post両方が発火する

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `TableEvent` トレイト（78-83行目）。`DatabaseEvent` を拡張し `name: String` フィールドを追加 |
| 1-2 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `CreateTablePreEvent` ケースクラス（88行目）。database, nameの2フィールドを持つ |

**読解のコツ**: `TableEvent` は `DatabaseEvent` を継承するため、`database` フィールドは `DatabaseEvent` から、`name` フィールドは `TableEvent` から来る。ケースクラスのパラメータがこれらのトレイトの抽象メンバを実装している。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `createTable` メソッド（88-96行目）。tableDefinitionからDB名とテーブル名を抽出し、Pre/Postイベントを発火 |

**主要処理フロー**:
1. **89行目**: `val db = tableDefinition.database` でデータベース名を取得
2. **90行目**: `val name = tableDefinition.identifier.table` でテーブル名を取得
3. **91-92行目**: `createVersion` を `SPARK_VERSION` に設定
4. **93行目**: `postToAll(CreateTablePreEvent(db, name))` - Preイベント発火
5. **94行目**: `delegate.createTable(tableDefinitionWithVersion, ignoreIfExists)` - 実際の作成
6. **95行目**: `postToAll(CreateTableEvent(db, name))` - Postイベント発火（成功時のみ）

#### Step 3: テストケースで動作を確認する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | tableテストケース（92-162行目）。成功時Pre+Post、失敗時Preのみの発火パターン |

**主要処理フロー**:
- **112-113行目**: 成功時 `CreateTablePreEvent("db5", "tbl1") :: CreateTableEvent("db5", "tbl1") :: Nil`
- **120-121行目**: 失敗時（既存テーブル） `CreateTablePreEvent("db5", "tbl1") :: Nil`（Preのみ）

### プログラム呼び出し階層図

```
ExternalCatalogWithListener.createTable() [ExternalCatalogWithListener.scala:88]
    |
    +-- tableDefinition.database, tableDefinition.identifier.table [89-90行目]
    |
    +-- tableDefinition.copy(createVersion = SPARK_VERSION) [91-92行目]
    |
    +-- postToAll(CreateTablePreEvent(db, name)) [93行目] ← 本イベント
    |       |
    |       +-- doPostEvent() [34行目]
    |               |
    |               +-- ExternalCatalogEventListener.onEvent()
    |
    +-- delegate.createTable(tableDefinitionWithVersion, ignoreIfExists) [94行目]
    |
    +-- postToAll(CreateTableEvent(db, name)) [95行目]
```

### データフロー図

```
[入力]                          [処理]                              [出力]

CatalogTable ---------> ExternalCatalogWithListener -------> CreateTablePreEvent
  (tableDefinition)          .createTable()                    (database, name)
  - database                      |                               |
  - identifier.table              |                               v
  - tableType                     |                    ExternalCatalogEventListener
  - storage                       v                          .onEvent()
  - schema              delegate.createTable()
                                  |
                                  v
                         CreateTableEvent (成功時のみ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | ソース | CreateTablePreEventイベントクラスおよびTableEventトレイトの定義 |
| ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | ソース | イベント発火のラッパークラス |
| ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | テスト | イベント発火パターンのテスト |
