# 通知設計書 49-CreateTableEvent

## 概要

本ドキュメントは、Apache SparkのSQL外部カタログにおけるテーブル作成後イベント `CreateTableEvent` の通知設計を記載する。このイベントは外部カタログでテーブルが正常に作成された後に発火され、リスナーに事後通知を提供する。

### 本通知の処理概要

`CreateTableEvent` は、`ExternalCatalogWithListener` を通じてテーブルの作成操作が正常に完了した直後に発火されるイベント通知である。対応する `CreateTablePreEvent` と対になって動作し、テーブル作成操作の成功を確認するためのイベントである。

**業務上の目的・背景**：テーブル作成の成功をリスナーに通知することで、メタデータキャッシュへの新規テーブル追加、データカタログツールとの同期、ETLパイプラインの起動トリガー、依存サービスへの通知、監査ログの完了記録などの後処理を確実に実行可能にする。PreEventと組み合わせることで、操作の成否を追跡でき、失敗した場合（PostEventが発火しない）との区別が可能となる。

**通知の送信タイミング**：`ExternalCatalogWithListener.createTable()` メソッド内で、委譲先の `delegate.createTable()` が正常に完了した直後に `postToAll(CreateTableEvent(db, name))` により発火される。delegate呼び出しで例外が発生した場合はこのイベントは発火しない。

**通知の受信者**：`ExternalCatalogEventListener` インタフェースを実装し、`ExternalCatalogWithListener` にリスナーとして登録されたすべてのオブジェクトが受信する。

**通知内容の概要**：テーブルが属するデータベース名（`database: String`）とテーブル名（`name: String`）を含む。`TableEvent` トレイトを継承する。テーブルの詳細定義（スキーマ、ストレージ形式等）はイベントに含まれない。

**期待されるアクション**：受信したリスナーは、メタデータキャッシュへの新規テーブル登録、外部データカタログとの同期、依存するETLパイプラインの起動、監査ログの完了記録、権限の自動設定などを実行する。

## 通知種別

アプリ内イベント通知（Spark ListenerBus経由のインプロセス同期通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ListenerBus.postToAll経由） |
| 優先度 | 中（操作完了後の通知） |
| リトライ | 無（同期呼び出しのため） |

### 送信先決定ロジック

`ExternalCatalogWithListener` に登録された全ての `ExternalCatalogEventListener` に対して、`postToAll` により同期的にブロードキャスト配信される。

## 通知テンプレート

### メール通知の場合

本通知はインプロセスイベント通知であるため、メールテンプレートは該当しない。

### 本文テンプレート

```
CreateTableEvent(
  database: String,  // テーブルが属するデータベース名
  name: String       // 作成されたテーブル名
)
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| database | テーブルが属するデータベース名 | CatalogTable.database | Yes |
| name | 作成されたテーブル名 | CatalogTable.identifier.table | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| API呼び出し | ExternalCatalogWithListener.createTable() | delegate.createTable()が正常完了した場合 | 操作成功時のみ発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| delegate.createTable()が例外をスロー | テーブル作成が失敗した場合はPostEventは発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ExternalCatalogWithListener.createTable呼び出し] --> B[postToAll - CreateTablePreEvent発火]
    B --> C[delegate.createTable実行]
    C --> D{作成成功?}
    D -->|成功| E[postToAll - CreateTableEvent発火]
    D -->|失敗 - 例外| F[CreateTableEventは発火しない]
    E --> G[全リスナーのonEventを同期呼び出し]
    G --> H[終了]
    F --> H
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| CatalogTable | 作成されたテーブル定義の参照 | identifier, tableType, storage, schema等を含む |

### テーブル別参照項目詳細

#### CatalogTable

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| database | イベントのdatabaseフィールドに設定 | tableDefinition.databaseから取得 |
| identifier.table | イベントのnameフィールドに設定 | tableDefinition.identifier.tableから取得 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 外部カタログ（Hive Metastore等） | CREATE | delegate.createTable()によるテーブル作成（イベント発火前に完了済み） |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| 該当なし | - | - | イベント発火のみ |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonEvent内で例外がスローされた場合 | ListenerBusがキャッチしログに記録 |
| イベント未発火 | delegate.createTable()が例外をスロー | PostEventは発火しない。PreEventのみが記録される |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。

## セキュリティ考慮事項

本イベントはインプロセスで配信されるため、外部ネットワーク通信は発生しない。データベース名とテーブル名のみを含むため、直接的な機密情報の漏洩リスクは低い。テーブル定義の詳細（スキーマ等）はイベントに含まれないため、詳細な監査が必要な場合はリスナー側でカタログAPIを呼び出す必要がある。

## 備考

- `CreateTableEvent` は `CreateTablePreEvent` と対になるPostイベントである
- `TableEvent` トレイトを継承し、`database` と `name` の2フィールドを持つ
- テストスイート `ExternalCatalogEventSuite`（112-121行目）で発火パターンが検証されている
- `ignoreIfExists = true` で既存テーブルが存在する場合でもPre/Post両方が発火する
- テーブル定義の詳細（スキーマ、ストレージ形式、パーティション定義等）はイベントに含まれない

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | `CreateTableEvent` ケースクラス（93行目）。`TableEvent` トレイト（78-83行目）からの継承 |

**読解のコツ**: `CreateTablePreEvent`（88行目）と `CreateTableEvent`（93行目）は同じ `TableEvent` トレイトを継承し、同じ `database` と `name` フィールドを持つ。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | `createTable` メソッド（88-96行目）。95行目のPostイベント発火に注目。delegate成功後にのみ到達する |

**主要処理フロー**:
1. **89-90行目**: DB名とテーブル名の取得
2. **91-92行目**: createVersionの設定
3. **93行目**: `postToAll(CreateTablePreEvent(db, name))` - Preイベント
4. **94行目**: `delegate.createTable(tableDefinitionWithVersion, ignoreIfExists)` - 実際の作成
5. **95行目**: `postToAll(CreateTableEvent(db, name))` - Postイベント（本イベント）

#### Step 3: テストケースで動作を確認する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | tableテストケース（112-121行目）。成功時のPre+Post発火パターン |

### プログラム呼び出し階層図

```
ExternalCatalogWithListener.createTable() [ExternalCatalogWithListener.scala:88]
    |
    +-- tableDefinition.database, tableDefinition.identifier.table [89-90行目]
    |
    +-- postToAll(CreateTablePreEvent(db, name)) [93行目]
    |
    +-- delegate.createTable() [94行目]
    |
    +-- postToAll(CreateTableEvent(db, name)) [95行目] ← 本イベント
            |
            +-- doPostEvent() [34行目]
                    |
                    +-- ExternalCatalogEventListener.onEvent()
```

### データフロー図

```
[入力]                          [処理]                              [出力]

CatalogTable ---------> ExternalCatalogWithListener -------> CreateTablePreEvent (操作前)
  (tableDefinition)          .createTable()                    (database, name)
                                    |                               |
                                    +-- delegate.createTable() -----+
                                    |                               |
                                    v                               v
                              (操作成功時のみ)              CreateTableEvent (操作後)
                                                              (database, name)
                                                                    |
                                                                    v
                                                         ExternalCatalogEventListener
                                                                .onEvent()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| events.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/events.scala` | ソース | CreateTableEventイベントクラスおよびTableEventトレイトの定義 |
| ExternalCatalogWithListener.scala | `sql/catalyst/src/main/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogWithListener.scala` | ソース | イベント発火のラッパークラス |
| ExternalCatalogEventSuite.scala | `sql/catalyst/src/test/scala/org/apache/spark/sql/catalyst/catalog/ExternalCatalogEventSuite.scala` | テスト | イベント発火パターンのテスト |
