# 通知設計書 75-QueryTerminatedEvent

## 概要

本ドキュメントは、Apache SparkのStructured Streamingにおいて、ストリーミングクエリが終了（正常またはエラー）した際に発火されるイベント通知「QueryTerminatedEvent」の設計を記述する。

### 本通知の処理概要

QueryTerminatedEventは、Structured Streamingのクエリが何らかの理由で終了した際に発火されるイベント通知である。正常終了、ユーザーによる停止、例外による異常終了のいずれの場合にも発火される。例外が発生した場合は例外メッセージとエラークラス情報が含まれ、正常終了の場合はこれらはNoneとなる。

**業務上の目的・背景**：ストリーミングクエリの終了は運用上の重要イベントである。特に予期しない異常終了の場合、速やかに検知して対応する必要がある。QueryTerminatedEventにより、クエリの終了をリアルタイムに検知し、アラート発報、自動再起動ロジックのトリガー、運用ダッシュボードの更新、障害レポートの生成などのアクションを実行できる。

**通知の送信タイミング**：StreamExecution.runStream()のfinally句において、StreamingQueryManagerへの終了通知の後にpostEventで発火される。クエリスレッドの終了処理中に発火されるため、クエリは既に停止状態にある。

**通知の受信者**：StreamingQueryListenerインタフェースを実装し、SparkSessionのStreamingQueryManagerに登録されたすべてのリスナーが受信する。配信完了後、StreamingQueryListenerBusのactiveQueryRunIdsからrunIdが除去される。

**通知内容の概要**：クエリの永続ID（id: UUID）、実行ID（runId: UUID）、例外メッセージ（exception: Option[String]）、例外のエラークラス（errorClassOnException: Option[String]）が含まれる。

**期待されるアクション**：リスナーは通知を受けて、クエリ終了の記録、異常終了時のアラート発報、クエリの自動再起動判定、リソースのクリーンアップ、監視ダッシュボードの更新などを実行することが期待される。

## 通知種別

アプリ内通知（StreamingQueryListenerBus + Spark LiveListenerBus経由の非同期イベント通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（Spark LiveListenerBus経由で非同期配信） |
| 優先度 | 高 |
| リトライ | 無し |

### 送信先決定ロジック

StreamingQueryListenerBusに登録されたすべてのStreamingQueryListenerに対して配信される。QueryTerminatedEventの配信完了後、postToAllのオーバーライドにより、activeQueryRunIdsからrunIdが除去される（SPARK-19594対応）。これにより、以降のイベントはこのクエリに対して配信されなくなる。

## 通知テンプレート

### メール通知の場合

該当なし。本イベントはSpark内部のリスナーバスを通じたプログラム内通知である。

### 本文テンプレート

```json
{
  "id": "{id}",
  "runId": "{runId}",
  "exception": "{exception}",
  "errorClassOnException": "{errorClassOnException}"
}
```

### 添付ファイル

該当なし。

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| id | クエリの永続一意ID | StreamExecution.id | Yes |
| runId | 実行ごとの一意ID | StreamExecution.runId | Yes |
| exception | 例外メッセージ（正常終了時はNone） | StreamExecution.exception.map(_.cause).map(Utils.exceptionString) | No |
| errorClassOnException | 例外のエラークラス（SparkThrowableの場合） | SparkThrowable.getErrorClass() | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| クエリ終了 | StreamExecution.runStream()のfinally句 | 常に発火（正常・異常問わず） | クエリの停止時に必ず1回発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | クエリが終了した場合は必ずQueryTerminatedEventが発火される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[クエリ終了: 正常/例外/停止] --> B[StreamExecution.runStream finally句]
    B --> C[StreamingQueryManager.notifyQueryTermination]
    C --> D[QueryTerminatedEvent生成]
    D --> E{例外あり?}
    E -->|Yes| F[exception, errorClassOnException設定]
    E -->|No| G[exception=None, errorClassOnException=None]
    F --> H[postEvent QueryTerminatedEvent]
    G --> H
    H --> I[StreamingQueryListenerBus.post]
    I --> J[SparkListenerBus.post 非同期]
    J --> K[doPostEvent: listener.onQueryTerminated]
    K --> L[postToAll完了後: activeQueryRunIdsからrunId除去]
    L --> M[terminationLatch解放]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし。本イベントはメモリ上の状態情報から生成される。

### 更新テーブル一覧

該当なし。

#### 送信ログテーブル

Sparkイベントログが有効な場合、SparkListenerEventとしてイベントログに記録される。

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー例外 | リスナーのonQueryTerminatedメソッドで例外発生 | ListenerBusが例外をキャッチしてログ出力。他のリスナーへの配信は継続 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0回（リトライなし） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | 制限なし（クエリ終了時のみ発火） |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし。クエリ終了時に即座に配信される。

## セキュリティ考慮事項

- exceptionフィールドにはスタックトレースが含まれる可能性があり、内部実装の詳細が露出する
- エラーメッセージに接続文字列やファイルパスなどの機密情報が含まれる可能性がある
- イベントログに記録される場合、例外情報がログに残る点に注意が必要

## 備考

- QueryTerminatedEventはSince 2.1.0で導入。errorClassOnExceptionフィールドはSince 3.5.0で追加
- 後方互換のため、errorClassOnExceptionなしのコンストラクタ（286-288行目）も提供されている
- activeQueryRunIdsからの除去はpostToAllオーバーライド（87-94行目）で行われ、全リスナーへの配信完了後に実行される
- QueryStartedEventが発火されている場合、QueryTerminatedEventも必ず発火されることが保証されている（277-278行目のコメント）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | StreamingQueryListener.scala | `sql/api/src/main/scala/org/apache/spark/sql/streaming/StreamingQueryListener.scala` | QueryTerminatedEventクラス（278-298行目）のフィールド定義（id, runId, exception, errorClassOnException）。後方互換コンストラクタ（286-288行目）を確認 |

**読解のコツ**: exceptionとerrorClassOnExceptionはOption[String]型であり、正常終了時はNone。errorClassOnExceptionはSparkのエラークラスフレームワークに対応する例外の場合のみ値が設定される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | StreamExecution.scala | `sql/core/src/main/scala/org/apache/spark/sql/execution/streaming/runtime/StreamExecution.scala` | runStreamのfinally句（425-428行目）でのQueryTerminatedEvent生成。notifyQueryTermination呼び出し後にpostEventが実行される |

**主要処理フロー**:
1. **425行目**: `sparkSession.streams.notifyQueryTermination(StreamExecution.this)` - QueryManager通知
2. **426-428行目**: `postEvent(new QueryTerminatedEvent(id, runId, exception..., errorClassOpt))` - イベント発火

#### Step 3: リスナーバスのクリーンアップを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | StreamingQueryListenerBus.scala | `sql/core/src/main/scala/org/apache/spark/sql/execution/streaming/runtime/StreamingQueryListenerBus.scala` | postToAllオーバーライド（87-94行目）でのactiveQueryRunIds除去。QueryTerminatedEventの配信完了後にrunIdが除去される |
| 3-2 | StreamingQueryListenerBus.scala | `sql/core/src/main/scala/org/apache/spark/sql/execution/streaming/runtime/StreamingQueryListenerBus.scala` | doPostEvent（141-143行目）でshouldReportチェック後にlistener.onQueryTerminatedを呼び出し |

### プログラム呼び出し階層図

```
StreamExecution.runStream()  [finally句]
    |
    +-- sparkSession.streams.notifyQueryTermination(this)
    |
    +-- postEvent(new QueryTerminatedEvent(id, runId, exception, errorClassOpt))
            |
            +-- SparkSession.streams.postListenerEvent(event)
                    |
                    +-- StreamingQueryListenerBus.post(event)
                            |
                            +-- sparkListenerBus.post(event)  [非同期]
                                    |
                                    +-- postToAll(event)  [super.postToAll]
                                    |       |
                                    |       +-- doPostEvent(listener, event)
                                    |               |
                                    |               +-- listener.onQueryTerminated(event)
                                    |
                                    +-- activeQueryRunIds -= runId  [SPARK-19594]
```

### データフロー図

```
[入力]                            [処理]                           [出力]

StreamExecution               StreamExecution                QueryTerminatedEvent
  id, runId            --->     .runStream() finally   --->    |
  exception                     |                              v
  errorClassOpt                 postEvent()              StreamingQueryListenerBus
                                                               |
                                                               v
                                                         StreamingQueryListener
                                                           .onQueryTerminated()
                                                               |
                                                               v
                                                         activeQueryRunIds除去
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| StreamingQueryListener.scala | `sql/api/src/main/scala/org/apache/spark/sql/streaming/StreamingQueryListener.scala` | ソース | QueryTerminatedEventクラス定義（278-298行目） |
| StreamExecution.scala | `sql/core/src/main/scala/org/apache/spark/sql/execution/streaming/runtime/StreamExecution.scala` | ソース | イベント発火元（425-428行目） |
| StreamingQueryListenerBus.scala | `sql/core/src/main/scala/org/apache/spark/sql/execution/streaming/runtime/StreamingQueryListenerBus.scala` | ソース | イベント配信バス・runIdクリーンアップ（87-94行目、141-143行目） |
| StreamingQueryListenerSuite.scala | `sql/core/src/test/scala/org/apache/spark/sql/streaming/StreamingQueryListenerSuite.scala` | テスト | リスナーイベントのテスト |
