# 通知設計書 8-SparkListenerJobEnd

## 概要

本ドキュメントは、Apache Sparkのコアイベント通知である`SparkListenerJobEnd`の設計について記述する。このイベントは、ジョブの実行が終了した際に発火するイベント通知である。

### 本通知の処理概要

SparkListenerJobEndは、Sparkジョブの完了（成功または失敗）をリスナーに通知するイベントである。

**業務上の目的・背景**：ジョブの終了はアプリケーション実行の重要なマイルストーンである。Spark UIのジョブ一覧で完了状態を更新する、ジョブの実行時間を確定する、失敗ジョブのエラー情報を表示する、イベントログに記録するなどの用途に利用される。動的リソース割り当てではジョブ終了時にExecutorの解放判断が行われる。ジョブの成功・失敗はJobResult（JobSucceeded/JobFailed）で表現される。

**通知の送信タイミング**：DAGSchedulerの複数箇所から発火する。(1) ジョブの全タスク完了時: cleanupStateForJobAndIndependentStages後にpost（L3472）。(2) ジョブ失敗時: abortStage等のエラーハンドリングパスでpost（L1342, L3368）。(3) SparkContext停止時: cleanUpAfterSchedulerStopでpost（L1342）。(4) 空パーティション時: submitJobでJobStartに続いて即座にpost（L975-976, L1055）。

**通知の受信者**：SparkListenerInterfaceを実装し、LiveListenerBusに登録されたすべてのリスナーが受信する。

**通知内容の概要**：jobId、time（完了時刻ミリ秒）、jobResult（JobSucceeded/JobFailed）を含む。

**期待されるアクション**：受信したリスナーは、ジョブの完了状態をUIに反映する、イベントログに記録する、リソースの解放判断を行う、失敗時にはエラー情報を表示する。

## 通知種別

アプリ内通知（Sparkイベントバス経由の非同期インプロセス通知）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 非同期（LiveListenerBus経由） |
| 優先度 | 中 |
| リトライ | 無し |

### 送信先決定ロジック

LiveListenerBusに登録された全リスナーに対してブロードキャスト配信される。

## 通知テンプレート

### メール通知の場合

本イベントはメール通知ではなく、Sparkイベントバス経由のインプロセス通知である。

### 本文テンプレート

```
イベントクラス: SparkListenerJobEnd
シリアライズ形式: JSON（Jackson, @JsonTypeInfo）

{
  "Event": "org.apache.spark.scheduler.SparkListenerJobEnd",
  "Job ID": <int>,
  "Completion Time": <long>,
  "Job Result": {
    "Result": "JobSucceeded" | "JobFailed"
    // JobFailedの場合: "Exception": { ... }
  }
}
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| 該当なし | - | - | インプロセス通知のため添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| jobId | ジョブID | job.jobId | Yes |
| time | 完了時刻（ミリ秒） | clock.getTimeMillis() | Yes |
| jobResult | ジョブの結果 | JobSucceeded / JobFailed(error) | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| スケジューラ内部処理 | ジョブの全タスク完了 | 正常完了時 | DAGScheduler.scala L3472 |
| スケジューラ内部処理 | abortStage/handleJobCancellation | ジョブ失敗・キャンセル時 | DAGScheduler.scala L3368等 |
| スケジューラ内部処理 | cleanUpAfterSchedulerStop | SparkContext停止時 | DAGScheduler.scala L1342 |
| スケジューラ内部処理 | submitJob/runApproximateJob（空パーティション） | 0タスクのジョブ | DAGScheduler.scala L975-976, L1055 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| LiveListenerBus停止後 | イベントは配信されない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ジョブ実行中] --> B{完了パターン}
    B -->|全タスク成功| C[cleanupStateForJobAndIndependentStages]
    B -->|エラー発生| D[abortStage/handleJobCancellation]
    B -->|SparkContext停止| E[cleanUpAfterSchedulerStop]
    B -->|空パーティション| F[submitJob内で即座処理]
    C --> G[SparkListenerJobEnd - JobSucceeded]
    D --> H[SparkListenerJobEnd - JobFailed]
    E --> H
    F --> G
    G --> I[listenerBus.post]
    H --> I
    I --> J[各リスナーのonJobEnd呼び出し]
    J --> K[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| KVStore | AppStatusListenerがジョブ完了情報を格納 | UI表示用 |

### テーブル別参照項目詳細

#### KVStore

| 参照項目（カラム名） | 用途 | 取得条件 |
|-------------------|------|---------|
| JobDataWrapper | ジョブの完了状態情報 | jobIdで検索 |

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| KVStore | UPDATE | ジョブ状態をSUCCEEDED/FAILEDに更新 |
| イベントログファイル | APPEND | イベントをJSON記録 |

#### 送信ログテーブル

| 操作 | 項目（カラム名） | 更新値 | 備考 |
|-----|-----------------|-------|------|
| UPDATE | JobDataWrapper.status | SUCCEEDED または FAILED | AppStatusListenerによる更新 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| キューオーバーフロー | イベント生成速度超過 | イベントドロップ |
| リスナー例外 | onJobEnd内で例外 | 例外キャッチ、ログ出力 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0 |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| キュー容量 | デフォルト10,000イベント |

### 配信時間帯

時間帯制限なし。

## セキュリティ考慮事項

- JobFailedの場合、例外情報にスタックトレースが含まれる可能性がある
- イベントログファイルのアクセス制御が必要

## 備考

- SparkListenerJobEndは3フィールド（jobId: Int, time: Long, jobResult: JobResult）のケースクラスである（SparkListener.scala L91-95）
- JobResultはsealedトレイトであり、JobSucceeded（オブジェクト）とJobFailed（case class、exception: Exception含む）の2つの実装がある
- SparkContext停止時のcleanUpAfterSchedulerStopでは、全アクティブジョブに対してJobFailed(SparkException)でpostされる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | L91-95: SparkListenerJobEndケースクラス。3フィールド（jobId, time, jobResult） |

**読解のコツ**: JobResult型の定義（JobSucceeded, JobFailed）も確認する。JobFailedはexceptionフィールドを持つ。

#### Step 2: エントリーポイントを理解する（成功パス）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | DAGScheduler.scala | `core/src/main/scala/org/apache/spark/scheduler/DAGScheduler.scala` | L3472: markMapStageJobAsFinished内でJobSucceededをpost。他にも正常完了パスが存在 |

#### Step 3: エントリーポイントを理解する（失敗パス）

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DAGScheduler.scala | `core/src/main/scala/org/apache/spark/scheduler/DAGScheduler.scala` | L1342: cleanUpAfterSchedulerStop内でJobFailed(SparkException)をpost。L3368: abortStage系のエラーハンドリングパスでJobFailedをpost |

**主要処理フロー（失敗パス・L3368）**:
1. **L3368**: `listenerBus.post(SparkListenerJobEnd(job.jobId, clock.getTimeMillis(), JobFailed(error)))`

#### Step 4: イベントディスパッチを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | L38-39: SparkListenerJobEndのパターンマッチ → listener.onJobEnd()呼び出し |

### プログラム呼び出し階層図

```
[成功パス]
DAGScheduler (全タスク完了)
    +-- cleanupStateForJobAndIndependentStages()
    +-- listenerBus.post(SparkListenerJobEnd(jobId, time, JobSucceeded)) [L3472]

[失敗パス]
DAGScheduler.abortStage() / handleJobCancellation()
    +-- listenerBus.post(SparkListenerJobEnd(jobId, time, JobFailed(error))) [L3368]

[停止パス]
DAGScheduler.cleanUpAfterSchedulerStop()
    +-- listenerBus.post(SparkListenerJobEnd(jobId, time, JobFailed(error))) [L1342]

共通:
    +-- LiveListenerBus.post()
            +-- AsyncEventQueue.post() [各キュー]
                    +-- listener.onJobEnd()
```

### データフロー図

```
[入力]                     [処理]                           [出力]

DAGScheduler           LiveListenerBus                  各リスナー
(jobId, time,         (AsyncEventQueue x N)
 jobResult)
                                                         AppStatusListener
jobId, time ---------> SparkListenerJobEnd ------------> (KVStore更新)
jobResult                                                EventLoggingListener
 - JobSucceeded                                          (イベントログ)
 - JobFailed(error)                                      ExecutorAllocationManager
                                                         (Executor解放判断)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SparkListener.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListener.scala` | ソース | SparkListenerJobEndケースクラス定義（L91-95） |
| DAGScheduler.scala | `core/src/main/scala/org/apache/spark/scheduler/DAGScheduler.scala` | ソース | 複数箇所からのイベント発火（L975, L1055, L1342, L3368, L3472） |
| SparkListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/SparkListenerBus.scala` | ソース | イベントディスパッチ（L38-39） |
| LiveListenerBus.scala | `core/src/main/scala/org/apache/spark/scheduler/LiveListenerBus.scala` | ソース | 非同期配信制御 |
| AppStatusListener.scala | `core/src/main/scala/org/apache/spark/status/AppStatusListener.scala` | ソース | UI用ジョブ完了記録 |
