# API設計書

## 概要

本ドキュメントは、SQLiteデータベースエンジンのC言語ライブラリAPIについて記載した設計書である。SQLiteはサーバーレスで自己完結型のトランザクション対応SQLデータベースエンジンであり、アプリケーションから直接C言語関数として呼び出される形でAPIを提供する。

**注意**: 本プロジェクトはWeb API（HTTP/REST API）ではなく、C言語のライブラリAPIを提供するプロジェクトである。

## 共通仕様

### ライブラリ情報

```
ライブラリ名: SQLite
ヘッダーファイル: sqlite3.h (src/sqlite.h.in からビルド時に生成)
バージョン定義: SQLITE_VERSION, SQLITE_VERSION_NUMBER
```

### APIの呼び出し方式

C言語関数として直接リンクして呼び出す。すべての公開API関数は `sqlite3_` プレフィックスで始まる。

### 共通データ型

| 型名 | 説明 |
| --- | --- |
| sqlite3* | データベース接続ハンドル |
| sqlite3_stmt* | プリペアドステートメントハンドル |
| sqlite3_value* | SQL値を保持する構造体 |
| sqlite3_context* | ユーザー定義関数のコンテキスト |
| sqlite3_int64 | 64ビット符号付き整数 |
| sqlite3_uint64 | 64ビット符号なし整数 |
| sqlite3_blob* | BLOBハンドル |
| sqlite3_backup* | バックアップハンドル |
| sqlite3_mutex* | ミューテックスハンドル |
| sqlite3_vfs* | VFS（仮想ファイルシステム）オブジェクト |
| sqlite3_file* | オープンファイルハンドル |
| sqlite3_filename | ファイル名（const char*型） |

### 共通戻り値（リザルトコード）

| ステータスコード | 値 | 説明 |
| --- | --- | --- |
| SQLITE_OK | 0 | 成功 |
| SQLITE_ERROR | 1 | 一般的なエラー |
| SQLITE_INTERNAL | 2 | SQLite内部ロジックエラー |
| SQLITE_PERM | 3 | アクセス権限拒否 |
| SQLITE_ABORT | 4 | コールバックがアボートを要求 |
| SQLITE_BUSY | 5 | データベースファイルがロックされている |
| SQLITE_LOCKED | 6 | テーブルがロックされている |
| SQLITE_NOMEM | 7 | メモリ割り当て失敗 |
| SQLITE_READONLY | 8 | 読み取り専用データベースへの書き込み試行 |
| SQLITE_INTERRUPT | 9 | sqlite3_interrupt()により操作が中断 |
| SQLITE_IOERR | 10 | ディスクI/Oエラー |
| SQLITE_CORRUPT | 11 | データベースディスクイメージが破損 |
| SQLITE_NOTFOUND | 12 | 不明なオペコード |
| SQLITE_FULL | 13 | データベースが満杯で挿入失敗 |
| SQLITE_CANTOPEN | 14 | データベースファイルを開けない |
| SQLITE_PROTOCOL | 15 | データベースロックプロトコルエラー |
| SQLITE_SCHEMA | 17 | データベーススキーマが変更された |
| SQLITE_TOOBIG | 18 | 文字列またはBLOBがサイズ制限を超過 |
| SQLITE_CONSTRAINT | 19 | 制約違反によりアボート |
| SQLITE_MISMATCH | 20 | データ型不一致 |
| SQLITE_MISUSE | 21 | ライブラリの不正使用 |
| SQLITE_NOLFS | 22 | ホストでサポートされていないOS機能を使用 |
| SQLITE_AUTH | 23 | 認可拒否 |
| SQLITE_RANGE | 25 | sqlite3_bindの第2パラメータが範囲外 |
| SQLITE_NOTADB | 26 | データベースファイルでないファイルを開いた |
| SQLITE_ROW | 100 | sqlite3_step()で次の行の準備完了 |
| SQLITE_DONE | 101 | sqlite3_step()の実行完了 |

## API一覧

| カテゴリ | API関数名 | 説明 |
| --- | --- | --- |
| ライブラリ初期化 | sqlite3_initialize | SQLiteライブラリの初期化 |
| ライブラリ初期化 | sqlite3_shutdown | SQLiteライブラリのシャットダウン |
| ライブラリ初期化 | sqlite3_config | グローバル設定の変更 |
| バージョン情報 | sqlite3_libversion | バージョン文字列の取得 |
| バージョン情報 | sqlite3_libversion_number | バージョン番号の取得 |
| バージョン情報 | sqlite3_sourceid | ソースIDの取得 |
| バージョン情報 | sqlite3_threadsafe | スレッドセーフ設定の確認 |
| データベース接続 | sqlite3_open | データベースを開く |
| データベース接続 | sqlite3_open16 | データベースを開く（UTF-16） |
| データベース接続 | sqlite3_open_v2 | データベースを開く（拡張版） |
| データベース接続 | sqlite3_close | データベースを閉じる |
| データベース接続 | sqlite3_close_v2 | データベースを閉じる（遅延解放版） |
| SQL実行 | sqlite3_exec | SQL文の実行（簡易版） |
| SQL実行 | sqlite3_prepare | SQL文をコンパイル |
| SQL実行 | sqlite3_prepare_v2 | SQL文をコンパイル（v2） |
| SQL実行 | sqlite3_prepare_v3 | SQL文をコンパイル（v3） |
| SQL実行 | sqlite3_step | ステートメントを実行 |
| SQL実行 | sqlite3_finalize | ステートメントを破棄 |
| SQL実行 | sqlite3_reset | ステートメントをリセット |
| パラメータバインド | sqlite3_bind_blob | BLOBパラメータをバインド |
| パラメータバインド | sqlite3_bind_blob64 | BLOBパラメータをバインド（64ビット） |
| パラメータバインド | sqlite3_bind_double | DOUBLEパラメータをバインド |
| パラメータバインド | sqlite3_bind_int | INTパラメータをバインド |
| パラメータバインド | sqlite3_bind_int64 | INT64パラメータをバインド |
| パラメータバインド | sqlite3_bind_null | NULLパラメータをバインド |
| パラメータバインド | sqlite3_bind_text | TEXTパラメータをバインド |
| パラメータバインド | sqlite3_bind_text64 | TEXTパラメータをバインド（64ビット） |
| パラメータバインド | sqlite3_bind_value | sqlite3_valueをバインド |
| パラメータバインド | sqlite3_bind_zeroblob | ゼロフィルBLOBをバインド |
| パラメータバインド | sqlite3_bind_zeroblob64 | ゼロフィルBLOBをバインド（64ビット） |
| パラメータバインド | sqlite3_bind_pointer | ポインタをバインド |
| パラメータバインド | sqlite3_bind_parameter_count | パラメータ数を取得 |
| パラメータバインド | sqlite3_bind_parameter_index | パラメータインデックスを取得 |
| パラメータバインド | sqlite3_bind_parameter_name | パラメータ名を取得 |
| パラメータバインド | sqlite3_clear_bindings | バインドをクリア |
| 結果取得 | sqlite3_column_blob | BLOB値を取得 |
| 結果取得 | sqlite3_column_bytes | バイト数を取得 |
| 結果取得 | sqlite3_column_bytes16 | バイト数を取得（UTF-16） |
| 結果取得 | sqlite3_column_count | カラム数を取得 |
| 結果取得 | sqlite3_column_double | DOUBLE値を取得 |
| 結果取得 | sqlite3_column_int | INT値を取得 |
| 結果取得 | sqlite3_column_int64 | INT64値を取得 |
| 結果取得 | sqlite3_column_name | カラム名を取得 |
| 結果取得 | sqlite3_column_name16 | カラム名を取得（UTF-16） |
| 結果取得 | sqlite3_column_text | TEXT値を取得 |
| 結果取得 | sqlite3_column_text16 | TEXT値を取得（UTF-16） |
| 結果取得 | sqlite3_column_type | カラム型を取得 |
| 結果取得 | sqlite3_column_value | sqlite3_valueを取得 |
| 結果取得 | sqlite3_data_count | 結果データ数を取得 |
| エラー処理 | sqlite3_errcode | エラーコードを取得 |
| エラー処理 | sqlite3_extended_errcode | 拡張エラーコードを取得 |
| エラー処理 | sqlite3_errmsg | エラーメッセージを取得 |
| エラー処理 | sqlite3_errmsg16 | エラーメッセージを取得（UTF-16） |
| エラー処理 | sqlite3_errstr | エラーコードの説明文字列を取得 |
| エラー処理 | sqlite3_error_offset | SQLエラー位置を取得 |
| トランザクション | sqlite3_get_autocommit | オートコミット状態を取得 |
| 変更追跡 | sqlite3_changes | 変更行数を取得 |
| 変更追跡 | sqlite3_changes64 | 変更行数を取得（64ビット） |
| 変更追跡 | sqlite3_total_changes | 累計変更行数を取得 |
| 変更追跡 | sqlite3_total_changes64 | 累計変更行数を取得（64ビット） |
| 変更追跡 | sqlite3_last_insert_rowid | 最後に挿入した行IDを取得 |
| BLOB操作 | sqlite3_blob_open | BLOBを開く |
| BLOB操作 | sqlite3_blob_close | BLOBを閉じる |
| BLOB操作 | sqlite3_blob_bytes | BLOBサイズを取得 |
| BLOB操作 | sqlite3_blob_read | BLOBを読み取り |
| BLOB操作 | sqlite3_blob_write | BLOBに書き込み |
| BLOB操作 | sqlite3_blob_reopen | BLOBを別行に移動 |
| バックアップ | sqlite3_backup_init | バックアップを初期化 |
| バックアップ | sqlite3_backup_step | バックアップステップ実行 |
| バックアップ | sqlite3_backup_finish | バックアップ完了 |
| バックアップ | sqlite3_backup_remaining | 残りページ数を取得 |
| バックアップ | sqlite3_backup_pagecount | 総ページ数を取得 |
| メモリ管理 | sqlite3_malloc | メモリ割り当て |
| メモリ管理 | sqlite3_malloc64 | メモリ割り当て（64ビット） |
| メモリ管理 | sqlite3_realloc | メモリ再割り当て |
| メモリ管理 | sqlite3_realloc64 | メモリ再割り当て（64ビット） |
| メモリ管理 | sqlite3_free | メモリ解放 |
| メモリ管理 | sqlite3_msize | メモリサイズ取得 |
| メモリ管理 | sqlite3_memory_used | 使用メモリ量を取得 |
| メモリ管理 | sqlite3_memory_highwater | 最大使用メモリ量を取得 |
| ユーザー定義関数 | sqlite3_create_function | 関数を登録 |
| ユーザー定義関数 | sqlite3_create_function_v2 | 関数を登録（v2） |
| ユーザー定義関数 | sqlite3_create_window_function | ウィンドウ関数を登録 |
| コールバック | sqlite3_busy_handler | ビジーハンドラを設定 |
| コールバック | sqlite3_busy_timeout | ビジータイムアウトを設定 |
| コールバック | sqlite3_commit_hook | コミットフックを設定 |
| コールバック | sqlite3_rollback_hook | ロールバックフックを設定 |
| コールバック | sqlite3_update_hook | 更新フックを設定 |
| コールバック | sqlite3_trace_v2 | トレースコールバックを設定 |
| VFS | sqlite3_vfs_find | VFSを検索 |
| VFS | sqlite3_vfs_register | VFSを登録 |
| VFS | sqlite3_vfs_unregister | VFSを登録解除 |
| WAL | sqlite3_wal_checkpoint | WALチェックポイント実行 |
| WAL | sqlite3_wal_checkpoint_v2 | WALチェックポイント実行（v2） |
| WAL | sqlite3_wal_autocheckpoint | 自動チェックポイント設定 |
| WAL | sqlite3_wal_hook | WALフックを設定 |

## 各API関数定義

### ライブラリ初期化

#### 1. sqlite3_initialize

SQLiteライブラリを初期化する。通常は自動的に呼び出されるため、明示的な呼び出しは不要。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_initialize(void)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

なし

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

**使用例**

```c
int rc = sqlite3_initialize();
if (rc != SQLITE_OK) {
    // エラー処理
}
```

---

#### 2. sqlite3_shutdown

SQLiteライブラリをシャットダウンする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_shutdown(void)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | No |

**パラメータ**

なし

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK |

**注意事項**

- 未解放のデータベース接続やメモリ割り当てがある状態で呼び出してはならない
- SQLiteの他の部分が使用中の状態で呼び出してはならない

---

#### 3. sqlite3_config

SQLiteライブラリのグローバル設定を変更する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_config(int op, ...)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | No |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| op | int | Yes | 設定オプション（SQLITE_CONFIG_*定数） |
| ... | varies | - | オプションに応じた追加パラメータ |

**主要な設定オプション**

| オプション | 説明 |
| --- | --- |
| SQLITE_CONFIG_SINGLETHREAD | シングルスレッドモードに設定 |
| SQLITE_CONFIG_MULTITHREAD | マルチスレッドモードに設定 |
| SQLITE_CONFIG_SERIALIZED | シリアライズドモードに設定 |
| SQLITE_CONFIG_MALLOC | カスタムメモリアロケータを設定 |
| SQLITE_CONFIG_LOG | ログコールバックを設定 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

---

### バージョン情報

#### 4. sqlite3_libversion

SQLiteライブラリのバージョン文字列を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `const char *sqlite3_libversion(void)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**戻り値**

| 型 | 説明 |
| --- | --- |
| const char* | "X.Y.Z"形式のバージョン文字列 |

---

#### 5. sqlite3_libversion_number

SQLiteライブラリのバージョン番号を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_libversion_number(void)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | X*1000000 + Y*1000 + Z形式のバージョン番号 |

---

#### 6. sqlite3_threadsafe

SQLiteがスレッドセーフでコンパイルされているかを確認する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_threadsafe(void)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | 0=スレッドセーフ無効、1または2=スレッドセーフ有効 |

---

### データベース接続

#### 7. sqlite3_open

データベースファイルを開く。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_open(const char *filename, sqlite3 **ppDb)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| filename | const char* | Yes | データベースファイル名（UTF-8） |
| ppDb | sqlite3** | Yes | データベースハンドルを受け取るポインタ |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

**使用例**

```c
sqlite3 *db;
int rc = sqlite3_open("test.db", &db);
if (rc != SQLITE_OK) {
    fprintf(stderr, "Cannot open database: %s\n", sqlite3_errmsg(db));
    sqlite3_close(db);
    return rc;
}
```

---

#### 8. sqlite3_open_v2

データベースファイルを開く（拡張版）。オープンフラグを指定可能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_open_v2(const char *filename, sqlite3 **ppDb, int flags, const char *zVfs)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| filename | const char* | Yes | データベースファイル名（UTF-8） |
| ppDb | sqlite3** | Yes | データベースハンドルを受け取るポインタ |
| flags | int | Yes | オープンフラグの組み合わせ |
| zVfs | const char* | No | 使用するVFS名（NULLでデフォルト） |

**主要なフラグ**

| フラグ | 説明 |
| --- | --- |
| SQLITE_OPEN_READONLY | 読み取り専用で開く |
| SQLITE_OPEN_READWRITE | 読み書き可能で開く |
| SQLITE_OPEN_CREATE | ファイルが存在しない場合は作成 |
| SQLITE_OPEN_URI | URI形式のファイル名を許可 |
| SQLITE_OPEN_MEMORY | インメモリデータベースとして開く |
| SQLITE_OPEN_NOMUTEX | マルチスレッドモードで開く |
| SQLITE_OPEN_FULLMUTEX | シリアライズドモードで開く |
| SQLITE_OPEN_SHAREDCACHE | 共有キャッシュモードを有効化 |
| SQLITE_OPEN_PRIVATECACHE | プライベートキャッシュモードを有効化 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

**使用例**

```c
sqlite3 *db;
int rc = sqlite3_open_v2(
    "test.db",
    &db,
    SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE,
    NULL
);
```

---

#### 9. sqlite3_close

データベース接続を閉じる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_close(sqlite3 *db)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| db | sqlite3* | Yes | データベースハンドル |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）、未解放のリソースがある場合はSQLITE_BUSY |

---

#### 10. sqlite3_close_v2

データベース接続を閉じる（遅延解放版）。未解放のリソースがあっても即座にSQLITE_OKを返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_close_v2(sqlite3 *db)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| db | sqlite3* | Yes | データベースハンドル |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK |

---

### SQL実行

#### 11. sqlite3_exec

SQL文を実行する簡易インターフェース。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_exec(sqlite3 *db, const char *sql, int (*callback)(void*,int,char**,char**), void *arg, char **errmsg)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| db | sqlite3* | Yes | データベースハンドル |
| sql | const char* | Yes | 実行するSQL文（セミコロン区切りで複数可） |
| callback | function pointer | No | 各行に対して呼び出されるコールバック関数 |
| arg | void* | No | コールバックの第1引数に渡される値 |
| errmsg | char** | No | エラーメッセージを受け取るポインタ |

**コールバック関数のシグネチャ**

```c
int callback(void *arg, int argc, char **argv, char **colNames)
```

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

**使用例**

```c
char *errmsg = NULL;
int rc = sqlite3_exec(db, "CREATE TABLE test (id INTEGER PRIMARY KEY, name TEXT)", NULL, NULL, &errmsg);
if (rc != SQLITE_OK) {
    fprintf(stderr, "SQL error: %s\n", errmsg);
    sqlite3_free(errmsg);
}
```

---

#### 12. sqlite3_prepare_v2

SQL文をコンパイルしてプリペアドステートメントを作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_prepare_v2(sqlite3 *db, const char *zSql, int nByte, sqlite3_stmt **ppStmt, const char **pzTail)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| db | sqlite3* | Yes | データベースハンドル |
| zSql | const char* | Yes | SQL文（UTF-8） |
| nByte | int | Yes | SQL文のバイト長（-1で自動計算） |
| ppStmt | sqlite3_stmt** | Yes | ステートメントハンドルを受け取るポインタ |
| pzTail | const char** | No | 未処理部分の先頭へのポインタを受け取る |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

**使用例**

```c
sqlite3_stmt *stmt;
const char *sql = "SELECT * FROM test WHERE id = ?";
int rc = sqlite3_prepare_v2(db, sql, -1, &stmt, NULL);
if (rc != SQLITE_OK) {
    fprintf(stderr, "Failed to prepare: %s\n", sqlite3_errmsg(db));
}
```

---

#### 13. sqlite3_step

プリペアドステートメントを実行する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_step(sqlite3_stmt *pStmt)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_ROW（次の行あり）、SQLITE_DONE（完了）、またはエラーコード |

**使用例**

```c
while (sqlite3_step(stmt) == SQLITE_ROW) {
    int id = sqlite3_column_int(stmt, 0);
    const char *name = (const char *)sqlite3_column_text(stmt, 1);
    printf("id: %d, name: %s\n", id, name);
}
```

---

#### 14. sqlite3_finalize

プリペアドステートメントを破棄する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_finalize(sqlite3_stmt *pStmt)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル（NULLも可） |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | 最後の評価のリザルトコード、またはSQLITE_OK（NULLの場合） |

---

#### 15. sqlite3_reset

プリペアドステートメントを初期状態にリセットする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_reset(sqlite3_stmt *pStmt)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | 最後の評価のリザルトコード |

---

### パラメータバインド

#### 16. sqlite3_bind_int

整数値をパラメータにバインドする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_bind_int(sqlite3_stmt *pStmt, int i, int iValue)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル |
| i | int | Yes | パラメータインデックス（1から開始） |
| iValue | int | Yes | バインドする整数値 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

---

#### 17. sqlite3_bind_text

テキスト値をパラメータにバインドする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_bind_text(sqlite3_stmt *pStmt, int i, const char *zData, int nData, void (*xDel)(void*))` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル |
| i | int | Yes | パラメータインデックス（1から開始） |
| zData | const char* | Yes | バインドするテキスト（UTF-8） |
| nData | int | Yes | テキストのバイト長（-1でNUL終端まで） |
| xDel | function pointer | No | デストラクタ（SQLITE_STATIC, SQLITE_TRANSIENT等） |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

---

### 結果取得

#### 18. sqlite3_column_int

現在行のカラムから整数値を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_column_int(sqlite3_stmt *pStmt, int iCol)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル |
| iCol | int | Yes | カラムインデックス（0から開始） |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | カラムの整数値 |

---

#### 19. sqlite3_column_text

現在行のカラムからテキスト値を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `const unsigned char *sqlite3_column_text(sqlite3_stmt *pStmt, int iCol)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル |
| iCol | int | Yes | カラムインデックス（0から開始） |

**戻り値**

| 型 | 説明 |
| --- | --- |
| const unsigned char* | UTF-8テキストへのポインタ |

---

#### 20. sqlite3_column_count

結果セットのカラム数を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_column_count(sqlite3_stmt *pStmt)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pStmt | sqlite3_stmt* | Yes | ステートメントハンドル |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | カラム数 |

---

### エラー処理

#### 21. sqlite3_errcode

最後のエラーコードを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_errcode(sqlite3 *db)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| db | sqlite3* | Yes | データベースハンドル |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | エラーコード |

---

#### 22. sqlite3_errmsg

最後のエラーメッセージを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `const char *sqlite3_errmsg(sqlite3 *db)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| db | sqlite3* | Yes | データベースハンドル |

**戻り値**

| 型 | 説明 |
| --- | --- |
| const char* | UTF-8エラーメッセージ |

---

### メモリ管理

#### 23. sqlite3_malloc

メモリを割り当てる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `void *sqlite3_malloc(int n)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| n | int | Yes | 割り当てるバイト数 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| void* | 割り当てられたメモリへのポインタ、失敗時はNULL |

---

#### 24. sqlite3_free

sqlite3_malloc()等で割り当てたメモリを解放する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `void sqlite3_free(void *p)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| p | void* | Yes | 解放するメモリへのポインタ（NULLも可） |

**戻り値**

なし

---

### BLOB操作

#### 25. sqlite3_blob_open

インクリメンタルI/O用にBLOBを開く。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `int sqlite3_blob_open(sqlite3 *db, const char *zDb, const char *zTable, const char *zColumn, sqlite3_int64 iRow, int flags, sqlite3_blob **ppBlob)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| db | sqlite3* | Yes | データベースハンドル |
| zDb | const char* | Yes | データベース名（"main"等） |
| zTable | const char* | Yes | テーブル名 |
| zColumn | const char* | Yes | カラム名 |
| iRow | sqlite3_int64 | Yes | 行のROWID |
| flags | int | Yes | 0=読み取り専用、1=読み書き可能 |
| ppBlob | sqlite3_blob** | Yes | BLOBハンドルを受け取るポインタ |

**戻り値**

| 型 | 説明 |
| --- | --- |
| int | SQLITE_OK（成功）またはエラーコード |

---

### バックアップ

#### 26. sqlite3_backup_init

オンラインバックアップを初期化する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| 関数シグネチャ | `sqlite3_backup *sqlite3_backup_init(sqlite3 *pDest, const char *zDestName, sqlite3 *pSource, const char *zSourceName)` |
| ヘッダー | sqlite3.h |
| スレッドセーフ | Yes |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| pDest | sqlite3* | Yes | コピー先データベースハンドル |
| zDestName | const char* | Yes | コピー先データベース名 |
| pSource | sqlite3* | Yes | コピー元データベースハンドル |
| zSourceName | const char* | Yes | コピー元データベース名 |

**戻り値**

| 型 | 説明 |
| --- | --- |
| sqlite3_backup* | バックアップハンドル、失敗時はNULL |

---

## 備考

### API公開ファイル

- 公開APIは `src/sqlite.h.in` で定義され、ビルド時に `sqlite3.h` として生成される
- 拡張機能用のAPIは `src/sqlite3ext.h` で定義される

### 主要な実装ファイル

- `src/main.c` - データベース接続、初期化などのコアAPI実装
- `src/vdbeapi.c` - VDBE（仮想マシン）関連API実装（sqlite3_step, sqlite3_column_*, sqlite3_bind_*等）
- `src/prepare.c` - SQL文のコンパイル関連API実装

### スレッドセーフティ

SQLiteは3つのスレッドモードをサポートする：

1. **シングルスレッド** (SQLITE_THREADSAFE=0): ミューテックス無効化
2. **マルチスレッド** (SQLITE_THREADSAFE=2): 複数スレッドで使用可能だが、同一接続を同時使用不可
3. **シリアライズド** (SQLITE_THREADSAFE=1): 完全スレッドセーフ

### 参考ドキュメント

- SQLite公式ドキュメント: https://sqlite.org/cintro.html
- SQLite C/C++ Interface: https://sqlite.org/c3ref/intro.html
