# アーキテクチャ設計書

## 概要

本ドキュメントは、SQLiteデータベースエンジン（バージョン3.52.0）のアーキテクチャ設計を記述する。SQLiteは、サーバーレスで自己完結型のトランザクショナルSQLデータベースエンジンであり、世界で最も広く使用されているデータベースエンジンの一つである。

SQLiteは、アプリケーションに埋め込まれるライブラリとして設計されており、独立したサーバープロセスを必要としない。データベース全体が単一のクロスプラットフォームファイルに格納される。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md)を参照

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| アプリケーション | C API (sqlite3.h) | SQLite機能の利用、SQL文の実行 |
| ファイルシステム | VFS (Virtual File System) | データベースファイル、ジャーナルファイルの読み書き |
| オペレーティングシステム | OS抽象化層 | ファイルロック、メモリ管理、スレッド同期 |
| TCLインタプリタ | TCL拡張 (tclsqlite.c) | テスト実行、スクリプトからのアクセス |
| 外部拡張機能 | 拡張API (sqlite3ext.h) | FTS5、R-Tree、JSONなどの拡張機能 |

## レイヤー構成

### アーキテクチャスタイル

**階層型アーキテクチャ（Layered Architecture）**

SQLiteは明確に分離された階層構造を持ち、各レイヤーが下位レイヤーの機能のみを利用する設計となっている。これにより、モジュール間の結合度が低く、保守性と移植性が高い。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Interface Layer | 外部APIの提供、初期化処理 | main.c, sqlite3.h, loadext.c |
| SQL Compiler Layer | SQL構文解析、クエリ最適化、コード生成 | tokenize.c, parse.y, select.c, where.c, expr.c |
| Virtual Machine Layer | バイトコードの実行、カーソル管理 | vdbe.c, vdbeapi.c, vdbeaux.c, vdbemem.c |
| B-Tree Layer | テーブル・インデックスのデータ構造管理 | btree.c, btmutex.c |
| Pager Layer | ページキャッシュ、トランザクション、ジャーナリング | pager.c, pcache.c, pcache1.c, wal.c |
| OS Abstraction Layer | OS依存機能の抽象化 | os.c, os_unix.c, os_win.c |

### レイヤー間の依存関係ルール

```
Interface Layer
      |
      v
SQL Compiler Layer
      |
      v
Virtual Machine Layer
      |
      v
B-Tree Layer
      |
      v
Pager Layer
      |
      v
OS Abstraction Layer
```

**依存ルール：**
- 上位レイヤーは直下のレイヤーのみを参照可能
- 下位レイヤーは上位レイヤーを参照禁止
- 同一レイヤー内での相互参照は許可（ただし循環依存は禁止）
- sqliteInt.hを通じて内部データ構造の共有が行われる

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| Core Engine | SQL処理のコア機能 | main.c, prepare.c, build.c |
| SQL Parser | SQL文の解析 | parse.y, tokenize.c |
| Query Processor | SELECT文の処理、クエリ最適化 | select.c, where.c, wherecode.c, whereexpr.c |
| Code Generator | 式の評価、VDBE命令生成 | expr.c, insert.c, update.c, delete.c |
| Virtual Machine | バイトコード実行エンジン | vdbe.c, vdbeapi.c, vdbeaux.c |
| Storage Engine | B-Tree、ページ管理 | btree.c, pager.c, pcache.c |
| Utility | 汎用ユーティリティ | util.c, malloc.c, hash.c, utf.c |
| Full-Text Search | 全文検索拡張 | ext/fts3/, ext/fts5/ |
| R-Tree Index | 空間インデックス拡張 | ext/rtree/ |
| JSON Support | JSON処理機能 | json.c |

### パッケージ構造

```
sqlite-master/
├── src/                      # コアソースコード
│   ├── main.c               # メインエントリーポイント、API実装
│   ├── sqlite.h.in          # 公開APIヘッダテンプレート
│   ├── sqliteInt.h          # 内部ヘッダ（データ構造定義）
│   ├── sqliteLimit.h        # 制限値定義
│   │
│   ├── parse.y              # SQL文法定義（Lemon形式）
│   ├── tokenize.c           # 字句解析器
│   │
│   ├── select.c             # SELECT文処理
│   ├── where.c              # WHERE句最適化（クエリオプティマイザ）
│   ├── wherecode.c          # WHERE句コード生成
│   ├── whereexpr.c          # WHERE式処理
│   ├── whereInt.h           # WHERE内部ヘッダ
│   ├── expr.c               # 式評価
│   ├── resolve.c            # 名前解決
│   │
│   ├── build.c              # DDL処理（CREATE/DROP等）
│   ├── insert.c             # INSERT文処理
│   ├── update.c             # UPDATE文処理
│   ├── delete.c             # DELETE文処理
│   ├── trigger.c            # トリガー処理
│   │
│   ├── vdbe.c               # 仮想マシン本体
│   ├── vdbe.h               # VDBEインターフェース
│   ├── vdbeInt.h            # VDBE内部ヘッダ
│   ├── vdbeapi.c            # VDBE API
│   ├── vdbeaux.c            # VDBEユーティリティ
│   ├── vdbemem.c            # VDBEメモリ管理
│   ├── vdbesort.c           # VDBEソート処理
│   │
│   ├── btree.c              # B-Tree実装
│   ├── btree.h              # B-Treeインターフェース
│   ├── btreeInt.h           # B-Tree内部ヘッダ
│   ├── btmutex.c            # B-Treeミューテックス
│   │
│   ├── pager.c              # ページャー（トランザクション管理）
│   ├── pager.h              # ページャーインターフェース
│   ├── pcache.c             # ページキャッシュ
│   ├── pcache.h             # ページキャッシュインターフェース
│   ├── pcache1.c            # デフォルトページキャッシュ実装
│   ├── wal.c                # Write-Ahead Logging
│   ├── wal.h                # WALインターフェース
│   │
│   ├── os.c                 # OS抽象化レイヤー
│   ├── os.h                 # OSインターフェース
│   ├── os_unix.c            # Unix/Linux VFS実装
│   ├── os_win.c             # Windows VFS実装
│   │
│   ├── func.c               # 組み込みSQL関数
│   ├── date.c               # 日付時刻関数
│   ├── json.c               # JSON関数
│   ├── printf.c             # 文字列フォーマット
│   ├── utf.c                # UTF-8処理
│   │
│   └── shell.c.in           # sqlite3 CLIシェル
│
├── ext/                      # 拡張機能
│   ├── fts3/                # Full-Text Search 3
│   ├── fts5/                # Full-Text Search 5
│   ├── rtree/               # R-Tree空間インデックス
│   ├── session/             # セッション拡張
│   ├── rbu/                 # Resumable Bulk Update
│   ├── icu/                 # ICU拡張
│   ├── misc/                # その他の拡張
│   └── wasm/                # WebAssembly対応
│
├── tool/                     # ビルド・開発ツール
│   ├── lemon.c              # Lemonパーサージェネレータ
│   ├── mksqlite3c.tcl       # アマルガメーション生成
│   ├── mksqlite3h.tcl       # sqlite3.h生成
│   └── mkopcodeh.tcl        # opcodes.h生成
│
├── test/                     # テストコード
│
└── doc/                      # ドキュメント
```

### コンポーネント依存関係

**主要コンポーネント間の依存関係：**

1. **main.c** -> sqliteInt.h, 全サブシステム
2. **SQL Parser (parse.y, tokenize.c)** -> sqliteInt.h
3. **Query Processor (select.c, where.c)** -> VDBE, B-Tree
4. **VDBE (vdbe.c)** -> B-Tree, Pager
5. **B-Tree (btree.c)** -> Pager
6. **Pager (pager.c)** -> OS Layer, Page Cache, WAL
7. **OS Layer (os_unix.c, os_win.c)** -> なし（最下位）

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| 組み込みRDB | SQLite | 3.52.0 | 本体（自己完結型） |

SQLiteは外部データベースに依存せず、自身がデータベースエンジンとして機能する。

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| Page Cache (pcache1.c) | 内蔵 | データベースページのメモリキャッシュ | PRAGMA cache_sizeで設定 |
| Schema Cache | 内蔵 | スキーマ情報のキャッシュ | データベース接続ライフタイム |
| Statement Cache | 内蔵 | prepared statementのキャッシュ | アプリケーション管理 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| 該当なし | - | SQLiteは同期処理ベースでMQを使用しない |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| FTS3/FTS4 | 内蔵拡張 | 全文検索（レガシー） |
| FTS5 | 内蔵拡張 | 全文検索（推奨） |
| R-Tree | 内蔵拡張 | 空間インデックス検索 |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Lemon Parser Generator | 内蔵 | SQL文法からパーサー生成 |
| ICU (オプション) | 外部 | 国際化文字列処理 |
| zlib (オプション) | 外部 | 圧縮機能 |

## データフロー

### リクエスト処理の流れ

**SQL文実行フロー（sqlite3_prepare -> sqlite3_step -> sqlite3_finalize）：**

1. **API呼び出し** (main.c)
   - アプリケーションがsqlite3_prepare_v2()でSQL文を渡す
   - 入力検証、データベース接続状態確認

2. **字句解析** (tokenize.c)
   - SQL文字列をトークンに分解
   - キーワード、識別子、リテラルの識別

3. **構文解析** (parse.y -> parse.c)
   - Lemonパーサーによる構文木生成
   - 文法エラーの検出

4. **意味解析・名前解決** (resolve.c, build.c)
   - テーブル名、カラム名の解決
   - 型チェック

5. **クエリ最適化** (where.c, select.c)
   - インデックス選択
   - 結合順序の決定
   - コスト推定

6. **コード生成** (expr.c, select.c, insert.c等)
   - VDBEバイトコードの生成
   - prepared statementの作成

7. **バイトコード実行** (vdbe.c)
   - sqlite3_step()でVDBE命令を順次実行
   - B-Tree操作、結果セット生成

8. **B-Tree操作** (btree.c)
   - テーブル/インデックスの読み書き
   - カーソル管理

9. **ページアクセス** (pager.c, pcache.c)
   - ページキャッシュ検索/更新
   - ディスクI/O

10. **ファイルI/O** (os_unix.c, os_win.c)
    - 実際のファイル読み書き
    - ロック管理

### 非同期処理の流れ

SQLiteは基本的に同期処理を行うが、以下の非同期的な要素がある：

1. **Write-Ahead Logging (WAL)**
   - 書き込みをWALファイルに記録
   - チェックポイントで非同期的にデータベースファイルに反映
   - 読み取りと書き込みの並行処理が可能

2. **Auto-Checkpoint**
   - WALサイズが閾値を超えると自動チェックポイント
   - バックグラウンドでのデータ統合

3. **Busy Handler**
   - ロック競合時のリトライメカニズム
   - sqlite3_busy_handler()でカスタマイズ可能

### データ永続化の流れ

**トランザクションコミット（ジャーナルモード: DELETE）：**

1. トランザクション開始（BEGIN）
2. 変更前ページをジャーナルファイルに書き出し
3. 変更をメモリ上のページキャッシュに適用
4. COMMIT時、ジャーナルをfsync
5. 変更ページをデータベースファイルに書き出し
6. データベースファイルをfsync
7. ジャーナルファイルを削除

**トランザクションコミット（ジャーナルモード: WAL）：**

1. トランザクション開始
2. 変更をWALファイルに追記
3. WALファイルをfsync
4. COMMIT完了
5. チェックポイント時にWAL内容をDBファイルに統合

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| 認可コールバック | auth.c | SQL文実行前の権限チェック |
| コンパイル時無効化 | SQLITE_OMIT_AUTHORIZATION | 認可機能の除去 |
| ファイルレベル | VFS/OS Layer | ファイルシステム権限に依存 |

SQLiteはアプリケーション組み込み型のため、ユーザー認証機能は標準では持たない。sqlite3_set_authorizer()で認可コールバックを設定可能。

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| エラーログ | sqlite3_log() | アプリケーション定義 |
| トレースログ | sqlite3_trace_v2() | コールバック関数 |
| プロファイリング | SQLITE_ENABLE_STMT_SCANSTATUS | VDBEオペコード統計 |
| OSTRACE | SQLITE_DEBUG_OS_TRACE | 標準エラー出力 |

### エラーハンドリング

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| SQL構文エラー | parse.y | SQLITE_ERROR + エラーメッセージ |
| 制約違反 | vdbe.c (OP_Halt) | SQLITE_CONSTRAINT |
| I/Oエラー | os_*.c | SQLITE_IOERR_* |
| メモリ不足 | malloc.c | SQLITE_NOMEM |
| 型変換エラー | vdbemem.c | SQLITE_MISMATCH |
| ロック競合 | pager.c | SQLITE_BUSY |
| 破損検出 | btree.c, pager.c | SQLITE_CORRUPT |

エラーコードは全てsqlite3.hで定義され、詳細メッセージはsqlite3_errmsg()で取得可能。

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| 単一データベース | pager.c | SERIALIZABLE（デフォルト） |
| 複数データベース | main.c | 複合トランザクション |
| セーブポイント | vdbe.c, pager.c | ネスト可能 |

**ロックレベル（os.h）：**
- NO_LOCK (0)
- SHARED_LOCK (1) - 読み取りロック
- RESERVED_LOCK (2) - 書き込み予約
- PENDING_LOCK (3) - 書き込み待機
- EXCLUSIVE_LOCK (4) - 排他ロック

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 階層化 | 全体構造 | Interface -> Compiler -> VM -> B-Tree -> Pager -> OS |
| 抽象化 | VFS | os.hで抽象インターフェース定義、os_unix.c/os_win.cで具象実装 |
| 単一責任 | 各モジュール | pager.c=ページ管理、btree.c=B-Tree操作など明確な分離 |
| 情報隠蔽 | ヘッダファイル構成 | 公開API(sqlite3.h)と内部実装(sqliteInt.h)の分離 |
| 防御的プログラミング | 全体 | NEVER/ALWAYS/testcaseマクロによる事前条件・事後条件の検証 |
| ゼロコピー | メモリ管理 | VDBEのMem構造体でのポインタ共有 |

### コーディング規約

**命名規則：**
- 公開API関数: `sqlite3_` プレフィックス
- 内部関数: `sqlite3` プレフィックス（アンダースコアなし）
- 型名: パスカルケース（Btree, Pager, Vdbe）
- マクロ: 大文字スネークケース（SQLITE_OK, BTREE_MEMORY）

**コメント規則：**
- 各ファイル先頭にライセンス（Public Domain宣言）
- 関数前にdoxygenスタイルのコメント
- 特殊コメント: NO_TEST, OPTIMIZATION-IF-TRUE/FALSE

**テスト規約：**
- testcaseマクロでブランチカバレッジを追跡
- assertマクロで不変条件を検証
- 全てのエラーパスをテスト可能にする設計

**移植性：**
- C89/C99準拠
- OS依存コードはVFSレイヤーに隔離
- コンパイルオプションによる機能の有効化/無効化（SQLITE_OMIT_*, SQLITE_ENABLE_*）

## 備考

**パフォーマンス最適化：**
- アマルガメーション（sqlite3.c）として単一ファイルにまとめることで、コンパイラの最適化効率が約5%向上
- ページサイズはデフォルト4096バイト、PRAGMAで変更可能
- メモリアロケータはカスタマイズ可能（mem1.c, mem2.c, mem3.c, mem5.c）

**ビルドオプション：**
- SQLITE_THREADSAFE: スレッドセーフティレベル（0, 1, 2）
- SQLITE_ENABLE_*: オプション機能の有効化
- SQLITE_OMIT_*: 不要機能の除去によるフットプリント削減

**拡張性：**
- 拡張関数: sqlite3_create_function()
- 仮想テーブル: sqlite3_create_module()
- VFS: sqlite3_vfs_register()
- コレーション: sqlite3_create_collation()
