# SQLite コードリーディングガイドライン

## はじめに

このガイドラインは、SQLiteのコードベースを効率的に理解するための手引きです。
C言語に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

**SQLiteの概要:**
SQLiteは、サーバーレス、ゼロ設定、トランザクション対応のSQLデータベースエンジンです。
単一のCファイルである「アマルガメーション（amalgamation）」としてコンパイル可能で、
組み込みシステムからデスクトップアプリケーションまで幅広く利用されています。

---

## 1. 言語基礎

> このセクションでは、C言語の基本構文とSQLiteで使用される概念を解説します。

### 1.1 プログラム構造

SQLiteはANSI C（C89/C99）で記述されています。各ソースファイルは標準的なC言語の構造に従い、
ファイル先頭に著作権放棄（パブリックドメイン宣言）と祝福文が記載されています。

```c
// ファイル: src/main.c:1-16
/*
** 2001 September 15
**
** The author disclaims copyright to this source code.  In place of
** a legal notice, here is a blessing:
**
**    May you do good and not evil.
**    May you find forgiveness for yourself and forgive others.
**    May you share freely, never taking more than you give.
**
*************************************************************************
** Main file for the SQLite library.  The routines in this file
** implement the programmer interface to the library.
*/
#include "sqliteInt.h"
```

**解説:**
- すべてのソースファイルは統一された祝福文から始まる
- `#include "sqliteInt.h"` は内部ヘッダーで、ほぼ全てのソースファイルでインクルードされる

### 1.2 データ型と変数

SQLiteでは、移植性を確保するために独自の型定義を多用しています。

```c
// ファイル: src/sqliteInt.h より抜粋
typedef sqlite_int64 i64;          /* 64ビット符号付き整数 */
typedef sqlite_uint64 u64;         /* 64ビット符号なし整数 */
typedef unsigned char u8;          /* 8ビット符号なし整数 */
typedef unsigned short int u16;    /* 16ビット符号なし整数 */
typedef unsigned int u32;          /* 32ビット符号なし整数 */
```

**主要なデータ型:**

| 型名 | 説明 | 使用例 |
|-----|------|--------|
| `i64` | 64ビット符号付き整数 | ROWIDなど |
| `u64` | 64ビット符号なし整数 | ファイルサイズなど |
| `u8` | 8ビット符号なし整数 | フラグ、オペコード |
| `u16` | 16ビット符号なし整数 | オフセット値 |
| `u32` | 32ビット符号なし整数 | ページ番号など |

### 1.3 制御構造

SQLiteでは、コードカバレッジテスト用の特殊なマクロがあります。

```c
// ファイル: src/sqliteInt.h:18-43
/* Special Comments:
**
**    NO_TEST                     - The branches on this line are not
**                                  measured by branch coverage.
**
**    OPTIMIZATION-IF-TRUE        - This branch is allowed to always be false
**                                  and the correct answer is still obtained.
**
**    OPTIMIZATION-IF-FALSE       - This branch is allowed to always be true
**                                  and the correct answer is still obtained.
**
**    PREVENTS-HARMLESS-OVERREAD  - This branch prevents a buffer overread
**                                  that would be harmless and undetectable.
*/
```

### 1.4 関数/メソッド定義

SQLiteの関数は命名規則に従って定義されています。公開API関数は`sqlite3_`プレフィックスを持ちます。

```c
// ファイル: src/main.c:98-111
/* sqlite3_libversion() function returns a pointer to the sqlite3_version[] string */
const char *sqlite3_libversion(void){ return sqlite3_version; }

/* sqlite3_sourceid() function returns a pointer to a string constant */
const char *sqlite3_sourceid(void){ return SQLITE_SOURCE_ID; }

/* sqlite3_libversion_number() function returns an integer */
int sqlite3_libversion_number(void){ return SQLITE_VERSION_NUMBER; }
```

### 1.5 モジュール/インポート

SQLiteはヘッダーファイルで依存関係を管理しています。主要なヘッダーの階層構造は以下の通りです。

```c
// 一般的なインクルード構造
#include "sqliteInt.h"    // 内部インターフェース（全ソースで使用）
#include "btreeInt.h"     // B-Tree内部定義（btree.cで使用）
#include "vdbeInt.h"      // VDBE内部定義（vdbe*.cで使用）
#include "whereInt.h"     // クエリオプティマイザ内部定義
```

---

## 2. プロジェクト固有の概念

> このセクションでは、SQLite特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### VDBE（Virtual DataBase Engine）

SQLiteは、SQLクエリを仮想マシンのバイトコードにコンパイルして実行します。

```c
// ファイル: src/vdbe.h:11-17
/*
** Header file for the Virtual DataBase Engine (VDBE)
**
** This header defines the interface to the virtual database engine
** or VDBE.  The VDBE implements an abstract machine that runs a
** simple program to access and modify the underlying database.
*/
```

#### VdbeOp構造体（仮想マシン命令）

```c
// ファイル: src/vdbe.h:55-93
struct VdbeOp {
  u8 opcode;          /* What operation to perform */
  signed char p4type; /* One of the P4_xxx constants for p4 */
  u16 p5;             /* Fifth parameter is an unsigned 16-bit integer */
  int p1;             /* First operand */
  int p2;             /* Second parameter (often the jump destination) */
  int p3;             /* The third parameter */
  union p4union {     /* fourth parameter */
    int i;                 /* Integer value if p4type==P4_INT32 */
    void *p;               /* Generic pointer */
    char *z;               /* Pointer to data for string types */
    // ... その他の型
  } p4;
};
```

### 2.2 プロジェクト独自のパターン

#### コンパイル時オプション（SQLITE_OMIT_* / SQLITE_ENABLE_*）

SQLiteは多数のコンパイル時オプションで機能を有効化/無効化できます。

```c
// ファイル: src/btree.c:59-63
#ifndef SQLITE_OMIT_AUTOVACUUM
#define IfNotOmitAV(expr) (expr)
#else
#define IfNotOmitAV(expr) 0
#endif
```

| プレフィックス | 意味 | 例 |
|--------------|------|-----|
| `SQLITE_OMIT_*` | 機能を除外 | `SQLITE_OMIT_AUTOVACUUM` |
| `SQLITE_ENABLE_*` | 機能を有効化 | `SQLITE_ENABLE_FTS5` |

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `*.c` | C言語ソースファイル | `main.c`, `btree.c` |
| `*.h` | ヘッダーファイル | `sqlite3.h`, `vdbe.h` |
| `*Int.h` | 内部専用ヘッダー | `sqliteInt.h`, `btreeInt.h` |
| `*.c.in` | テンプレートファイル（ビルド時に変換） | `shell.c.in`, `sqlite.h.in` |
| `test*.c` | テスト用コード | `test1.c`, `test_func.c` |
| `*.y` | Lemonパーサー文法ファイル | `parse.y` |
| `*.tcl` | Tclスクリプト（ビルド/テスト用） | `mksqlite3c.tcl` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `sqlite3_` | 公開API関数 | `sqlite3_open()`, `sqlite3_exec()` |
| `sqlite3` | 内部関数（プレフィックス） | `sqlite3BtreeOpen()` |
| `SQLITE_` | マクロ/定数 | `SQLITE_OK`, `SQLITE_ERROR` |
| `OP_` | VDBEオペコード | `OP_OpenRead`, `OP_Column` |
| `TK_` | トークン定数 | `TK_SELECT`, `TK_WHERE` |
| `P4_` | VdbeOp.p4の型定数 | `P4_INT32`, `P4_KEYINFO` |
| `Bt` | B-Tree関連 | `BtCursor`, `BtShared` |
| `Pager` | ページャー関連 | `PagerOpen()` |
| `Vdbe` | 仮想マシン関連 | `VdbeOp`, `VdbeFrame` |

### 3.3 プログラム分類一覧

| 分類 | ファイル名パターン | 説明 |
|-----|------------------|------|
| コア機能 | `main.c`, `sqlite3.c` | メインAPIと初期化 |
| パーサー | `parse.y`, `tokenize.c` | SQL文の構文解析 |
| クエリ実行 | `vdbe*.c` | 仮想マシン実行エンジン |
| ストレージ | `btree.c`, `pager.c` | データ永続化 |
| クエリ最適化 | `where*.c` | クエリプランの最適化 |
| 拡張機能 | `ext/` 配下 | FTS5, RTree等の拡張 |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
sqlite-master/
├── src/                  # メインソースコード
│   ├── main.c           # 公開API実装
│   ├── btree.c          # B-Treeストレージエンジン
│   ├── vdbe.c           # 仮想マシン
│   ├── pager.c          # ページキャッシュ/トランザクション
│   ├── select.c         # SELECT文処理
│   ├── where.c          # クエリオプティマイザ
│   ├── parse.y          # SQL文法定義（Lemon）
│   ├── sqliteInt.h      # 内部共通ヘッダー
│   ├── sqlite.h.in      # 公開APIヘッダーテンプレート
│   └── test*.c          # テスト用コード
├── ext/                  # 拡張機能
│   ├── fts5/            # 全文検索エンジン v5
│   ├── rtree/           # R-Tree空間インデックス
│   ├── session/         # セッション拡張
│   ├── rbu/             # Resumable Bulk Update
│   └── misc/            # その他小規模拡張
├── test/                 # テストスクリプト/データ
├── tool/                 # ビルド/ユーティリティツール
│   ├── lemon.c          # Lemonパーサージェネレータ
│   ├── mksqlite3c.tcl   # アマルガメーション生成
│   └── mksqlite3h.tcl   # ヘッダー生成
├── doc/                  # 内部ドキュメント
├── autoconf/             # autoconf設定
├── Makefile.in           # Makefileテンプレート
└── configure             # 設定スクリプト
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | SQLiteコアの実装 | `main.c`, `btree.c`, `vdbe.c` |
| `ext/` | 拡張機能の実装 | `fts5/`, `rtree/`, `session/` |
| `test/` | テストスクリプト | `*.test`（Tclテスト） |
| `tool/` | ビルド/生成ツール | `lemon.c`, `*.tcl` |
| `doc/` | 内部技術ドキュメント | `lemon.html`, `*.md` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

SQLiteはレイヤードアーキテクチャを採用しており、以下の階層構造を持ちます。

```
┌─────────────────────────────────────────────────────────────┐
│                    Interface (sqlite3.h)                    │
│                  sqlite3_open(), sqlite3_exec()             │
├─────────────────────────────────────────────────────────────┤
│                    SQL Compiler                             │
│         Tokenizer → Parser → Code Generator                 │
├─────────────────────────────────────────────────────────────┤
│                Virtual Machine (VDBE)                       │
│              Bytecode Interpreter (vdbe.c)                  │
├─────────────────────────────────────────────────────────────┤
│                   B-Tree Module                             │
│            B-Tree Cursor (btree.c, btree.h)                │
├─────────────────────────────────────────────────────────────┤
│                    Pager Module                             │
│       Page Cache + Transaction (pager.c, pager.h)          │
├─────────────────────────────────────────────────────────────┤
│                 OS Interface (VFS)                          │
│            os_unix.c / os_win.c / os_kv.c                  │
└─────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| Interface | 公開API提供 | `main.c`, `sqlite.h.in` |
| SQL Compiler | SQL解析・コード生成 | `parse.y`, `tokenize.c`, `select.c` |
| VDBE | バイトコード実行 | `vdbe.c`, `vdbeaux.c` |
| B-Tree | インデックス管理 | `btree.c`, `btreeInt.h` |
| Pager | ページキャッシュ・トランザクション | `pager.c`, `wal.c` |
| OS Interface | ファイルI/O抽象化 | `os_unix.c`, `os_win.c` |

### 5.3 データフロー

SQLクエリの実行フローは以下のようになります：

1. **API呼び出し** (`sqlite3_prepare_v2`) → SQL文字列を受け取る
2. **トークン化** (`tokenize.c`) → SQLをトークン列に分解
3. **構文解析** (`parse.y` → `parse.c`) → 構文木（AST）を生成
4. **コード生成** (`select.c`, `where.c`) → VDBEバイトコードを生成
5. **実行** (`vdbe.c`) → バイトコードを逐次実行
6. **データアクセス** (`btree.c`) → B-Treeカーソルでデータ取得
7. **ページI/O** (`pager.c`) → ページキャッシュ経由でディスクアクセス

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

SQLiteの主要なエントリーポイントは`main.c`に実装されています。

```c
// ファイル: src/main.c より抜粋
/*
** Main file for the SQLite library.  The routines in this file
** implement the programmer interface to the library.  Routines in
** other files are for internal use by SQLite and should not be
** accessed by users of the library.
*/
```

**主要な公開API関数:**

| 関数 | 役割 |
|------|------|
| `sqlite3_open()` | データベース接続を開く |
| `sqlite3_prepare_v2()` | SQL文をコンパイル |
| `sqlite3_step()` | コンパイル済みSQL文を実行 |
| `sqlite3_finalize()` | ステートメントを解放 |
| `sqlite3_close()` | データベース接続を閉じる |

### 6.2 ビジネスロジック

SQLiteの「ビジネスロジック」に相当するのは、SQLコンパイラとクエリオプティマイザです。

```c
// ファイル: src/where.c:11-18
/*
** This module contains C code that generates VDBE code used to process
** the WHERE clause of SQL statements.  This module is responsible for
** generating the code that loops through a table looking for applicable
** rows.  Indices are selected and used to speed the search when doing
** so is applicable.  Because this module is responsible for selecting
** indices, you might also think of this module as the "query optimizer".
*/
```

### 6.3 データアクセス

データアクセス層は、B-TreeモジュールとPagerモジュールで構成されます。

```c
// ファイル: src/btree.c:11-15
/*
** This file implements an external (disk-based) database using BTrees.
** See the header comment on "btreeInt.h" for additional information.
** Including a description of file format and an overview of operation.
*/
```

```c
// ファイル: src/pager.c:11-20
/*
** This is the implementation of the page cache subsystem or "pager".
**
** The pager is used to access a database disk file.  It implements
** atomic commit and rollback through the use of a journal file that
** is separate from the database file.  The pager also implements file
** locking to prevent two processes from writing the same database
** file simultaneously, or one process from reading the database while
** another is writing.
*/
```

### 6.4 ユーティリティ/共通機能

共通で使用されるユーティリティ関数は、各モジュールに分散しています。

| ファイル | 機能 |
|---------|------|
| `util.c` | 汎用ユーティリティ関数 |
| `hash.c` | ハッシュテーブル実装 |
| `printf.c` | フォーマット文字列処理 |
| `malloc.c` | メモリ管理 |
| `mutex.c` | 排他制御 |

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| SQLITE_OK/エラーコード | 戻り値による成否判定 | 高 | 全ファイル |
| コンパイル時条件 | `#ifdef SQLITE_OMIT_*` | 高 | 全ファイル |
| テストカバレッジマクロ | `testcase()`, `assert()` | 高 | 全ファイル |
| メモリ管理パターン | `sqlite3_malloc/free` | 中 | 全ファイル |

### 各パターンの詳細

#### パターン1: SQLITE_OK/エラーコードパターン

**目的:** 関数の成否を明確に伝達し、エラーハンドリングを統一する

**実装例:**
```c
// ファイル: src/btree.c:89-92
int sqlite3_enable_shared_cache(int enable){
  sqlite3GlobalConfig.sharedCacheEnabled = enable;
  return SQLITE_OK;
}
```

**主要なエラーコード:**

| コード | 値 | 意味 |
|-------|-----|------|
| `SQLITE_OK` | 0 | 成功 |
| `SQLITE_ERROR` | 1 | 一般エラー |
| `SQLITE_BUSY` | 5 | データベースがロック中 |
| `SQLITE_NOMEM` | 7 | メモリ不足 |
| `SQLITE_CORRUPT` | 11 | データベース破損 |

#### パターン2: コンパイル時条件パターン

**目的:** 機能の有効化/無効化をコンパイル時に制御

**実装例:**
```c
// ファイル: src/main.c:19-27
#ifdef SQLITE_ENABLE_FTS3
# include "fts3.h"
#endif
#ifdef SQLITE_ENABLE_RTREE
# include "rtree.h"
#endif
#if defined(SQLITE_ENABLE_ICU) || defined(SQLITE_ENABLE_ICU_COLLATIONS)
# include "sqliteicu.h"
#endif
```

#### パターン3: TRACEマクロパターン

**目的:** デバッグ時のトレース出力を制御

**実装例:**
```c
// ファイル: src/btree.c:28-33
#if 0
int sqlite3BtreeTrace=1;  /* True to enable tracing */
# define TRACE(X)  if(sqlite3BtreeTrace){printf X;fflush(stdout);}
#else
# define TRACE(X)
#endif
```

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定（公開API関数）
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認（構造体の変化）
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: SELECT文の実行フロー

**概要:** `sqlite3_prepare_v2()` から始まるSELECT文のコンパイルと実行

**処理フロー:**
```
sqlite3_prepare_v2() → sqlite3Prepare() → sqlite3RunParser()
    → sqlite3Parser() → sqlite3Select() → sqlite3WhereBegin()
        → VDBEコード生成 → sqlite3_step() → sqlite3VdbeExec()
```

**詳細な追跡:**

1. **API呼び出し** (`src/prepare.c`)
   - `sqlite3_prepare_v2()` がSQL文字列を受け取る
   - 内部で `sqlite3Prepare()` を呼び出す

2. **構文解析** (`src/parse.y` → 生成される `parse.c`)
   - Lemonパーサーが構文木を構築
   - `sqlite3Parser()` が各トークンを処理

3. **コード生成** (`src/select.c`)
   - `sqlite3Select()` がSELECT文のVDBEコードを生成
   - WHERE句は `sqlite3WhereBegin()` で処理

4. **実行** (`src/vdbe.c`)
   - `sqlite3VdbeExec()` がバイトコードを逐次実行
   - 各オペコードに対応する処理を実行

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか（`sqlite3_*` 関数）
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか（Parse構造体、Vdbe構造体）
- [ ] エラーハンドリングを確認したか（`SQLITE_OK` チェック）
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要とビルド方法
2. 公式サイトの[アーキテクチャ解説](https://sqlite.org/arch.html)
3. `src/sqliteInt.h` - 内部データ構造の定義

#### 特定機能を理解したい場合
1. 該当機能のヘッダーファイル（例: `btree.h`, `vdbe.h`）
2. 実装ファイル（例: `btree.c`, `vdbe.c`）
3. 対応するテストファイル（`test/` 配下）

#### 改修作業を行う場合
1. 対象機能のソースファイルのヘッダーコメント
2. 関連する `*Int.h` 内部ヘッダー
3. 対応するテストケース

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | ビルド方法、ソースツリー構造 | 最初に参照 |
| `doc/lemon.html` | Lemonパーサージェネレータ | パーサー修正時 |
| `doc/compile-for-*.md` | プラットフォーム別ビルド | ビルド時 |
| `src/sqlite.h.in` | 公開API定義 | API理解時 |
| 公式サイトドキュメント | 詳細な技術解説 | 深い理解が必要な時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `sqliteInt.h` が巨大すぎて全容を把握できない
A: このファイルは全内部定義を含むため約7000行あります。必要な構造体/関数だけを検索して参照してください。主要な構造体は`sqlite3`（データベース接続）、`Vdbe`（仮想マシン）、`Parse`（解析コンテキスト）です。

#### Q: `SQLITE_OMIT_*` マクロで条件コンパイルされている部分の理解が困難
A: まず標準ビルド（全機能有効）のコードパスを理解してください。`SQLITE_OMIT_*` はオプション機能の除外用で、コア機能は常に有効です。

#### Q: 生成されるファイル（parse.c, opcodes.h）が見つからない
A: これらはビルド時に生成されます。`make target_source` を実行すると、`tsrc/` ディレクトリに全ソースファイルが生成されます。

#### Q: アマルガメーション（sqlite3.c）の構造がわからない
A: アマルガメーションは全ソースを1ファイルにまとめたものです。`tool/mksqlite3c.tcl` がソースの結合順序を定義しています。コードリーディングには個別ソースファイルの方が適しています。

### デバッグのヒント

1. **TRACEマクロの有効化**: `#if 0` を `#if 1` に変更するとトレース出力が有効になる
2. **EXPLAIN文の活用**: `EXPLAIN` や `EXPLAIN QUERY PLAN` でVDBEコードを確認
3. **sqlite3_trace_v2()**: 実行時のSQL文トレースを有効化
4. **assert()の活用**: デバッグビルドでは多数のアサーションが有効

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| VDBE | Virtual DataBase Engine - SQLiteの仮想マシン |
| B-Tree | データとインデックスの格納に使用するバランス木構造 |
| Pager | ページキャッシュとトランザクション管理モジュール |
| WAL | Write-Ahead Logging - 先行書き込みログ方式 |
| VFS | Virtual File System - ファイルI/O抽象化レイヤー |
| Amalgamation | 全ソースを1ファイルにまとめたもの |
| Lemon | SQLiteが使用するLALR(1)パーサージェネレータ |
| FTS | Full-Text Search - 全文検索拡張 |
| R-Tree | 空間インデックス用のツリー構造 |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/main.c` | 公開API実装 | `sqlite3_open()`, `sqlite3_exec()` |
| `src/btree.c` | B-Treeストレージ | インデックス管理、データ格納 |
| `src/vdbe.c` | 仮想マシン | バイトコード実行エンジン |
| `src/pager.c` | ページキャッシュ | トランザクション、ジャーナル |
| `src/where.c` | クエリオプティマイザ | WHERE句の最適化 |
| `src/select.c` | SELECT処理 | SELECT文のコード生成 |
| `src/parse.y` | SQL文法 | Lemon形式の文法定義 |
| `src/sqliteInt.h` | 内部ヘッダー | 全内部定義（約7000行） |
| `src/sqlite.h.in` | 公開APIヘッダー | 公開インターフェース定義 |
| `tool/lemon.c` | パーサー生成器 | LALR(1)パーサー生成 |

### C. 参考資料

- [SQLite公式サイト](https://sqlite.org/)
- [SQLiteアーキテクチャ解説](https://sqlite.org/arch.html)
- [SQLiteファイルフォーマット](https://sqlite.org/fileformat2.html)
- [VDBE（仮想マシン）解説](https://sqlite.org/opcode.html)
- [クエリプランナー概要](https://sqlite.org/optoverview.html)
- [Lemonパーサージェネレータ](https://sqlite.org/lemon.html)
