# セキュリティ設計書

## 概要

本ドキュメントはSQLiteデータベースエンジンのセキュリティ設計について記載する。SQLiteは組み込み型リレーショナルデータベースであり、サーバーレスアーキテクチャを採用している。本書では、SQLiteが提供するセキュリティ機能、アクセス制御メカニズム、データ保護機能について詳述する。

## 認証設計

### 認証方式

SQLiteはサーバーレスデータベースエンジンであり、従来のユーザー名/パスワードベースの認証機構は**標準では提供されていない**。

認証に関する特徴:
- データベースファイルへのアクセスは、ファイルシステムレベルの権限に依存
- アプリケーションレベルでの認証は利用者側で実装が必要
- 暗号化拡張（SQLite Encryption Extension - SEE）を使用することで、データベースレベルでの暗号化とパスワード保護が可能（別売りの商用拡張）

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | N/A（サーバーレス） | SQLiteはサーバーレスのためセッション概念なし |
| セッション固定化対策 | N/A | ファイルベースのため対象外 |
| 接続管理 | `sqlite3_open()` / `sqlite3_close()` | データベースハンドル単位での管理 |
| 接続タイムアウト | `busy_timeout` PRAGMA | デフォルトは無制限（0） |

## 認可設計

### 権限体系

SQLiteはAuthorizerコールバック機構（`sqlite3_set_authorizer()`）を提供し、SQLステートメントレベルでのアクセス制御を可能にする。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| SQLITE_OK | 許可 | 操作の実行を許可 |
| SQLITE_DENY | 拒否 | 操作を拒否しエラーを返す |
| SQLITE_IGNORE | 無視 | 読み取り時はNULL返却、書き込み時は無視 |

### アクセス制御

SQLiteは`sqlite3_set_authorizer()`関数を通じて、以下の操作に対するアクセス制御コールバックを提供する:

**データ操作（DML）:**
- `SQLITE_SELECT` - SELECT文の実行
- `SQLITE_INSERT` - INSERT文の実行
- `SQLITE_UPDATE` - UPDATE文の実行
- `SQLITE_DELETE` - DELETE文の実行
- `SQLITE_READ` - カラムの読み取り

**スキーマ操作（DDL）:**
- `SQLITE_CREATE_TABLE` / `SQLITE_CREATE_TEMP_TABLE` - テーブル作成
- `SQLITE_CREATE_INDEX` / `SQLITE_CREATE_TEMP_INDEX` - インデックス作成
- `SQLITE_CREATE_VIEW` / `SQLITE_CREATE_TEMP_VIEW` - ビュー作成
- `SQLITE_CREATE_TRIGGER` / `SQLITE_CREATE_TEMP_TRIGGER` - トリガー作成
- `SQLITE_DROP_TABLE` / `SQLITE_DROP_INDEX` / `SQLITE_DROP_VIEW` / `SQLITE_DROP_TRIGGER` - 削除操作

**その他:**
- `SQLITE_PRAGMA` - PRAGMA文の実行
- `SQLITE_TRANSACTION` - トランザクション操作

実装例（`src/auth.c`より）:
```c
int sqlite3_set_authorizer(
  sqlite3 *db,
  int (*xAuth)(void*,int,const char*,const char*,const char*,const char*),
  void *pArg
);
```

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | N/A（ファイルベースDB） |
| HSTS | N/A（ファイルベースDB） |
| TLS/SSL | N/A（ネットワーク通信なし） |
| プロセス間通信 | ファイルロックによる排他制御 |

SQLiteはファイルベースのデータベースであり、ネットワーク通信は行わない。プロセス間でのデータベースアクセスは、ファイルロック機構により制御される。

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | N/A（通信なし） |
| 保存データ（標準） | 暗号化なし（平文） |
| 保存データ（拡張） | AES-256（SQLite Encryption Extension使用時） |
| ハッシュ関数（拡張） | SHA-1, SHA-3（ext/misc/sha1.c, ext/misc/shathree.c） |

**標準機能での暗号化:**
SQLiteコアには暗号化機能は含まれていない。`src/pager.c`には暗号化拡張のためのフックポイントが用意されている。

**利用可能なハッシュ関数拡張:**
- `sha1(X)` - SHA-1ハッシュの計算
- `sha3(X, SIZE)` - SHA-3ハッシュの計算（224, 256, 384, 512ビット対応）
- `sha3_agg(Y, SIZE)` - 集約SHA-3ハッシュ
- `sha3_query(Z, SIZE)` - クエリ結果のSHA-3ハッシュ

### 機密情報管理

SQLiteでは以下の方法で機密情報を保護することを推奨:

1. **ファイルシステム権限**: データベースファイルのアクセス権限を適切に設定（デフォルト: 0644）
2. **ディレクトリ権限**: データベースを格納するディレクトリの権限管理
3. **メモリ内データベース**: 機密データは`:memory:`データベースを使用
4. **アプリケーションレベル暗号化**: 機密カラムは格納前にアプリケーション側で暗号化

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| SQLインジェクション | パラメータバインディング機能（`sqlite3_bind_*` API群） |
| 整数オーバーフロー | メモリ割り当てサイズの安全マージン（256バイト） |
| バッファオーバーフロー | 厳格な長さ制限と境界チェック |
| DoS攻撃（リソース枯渇） | 各種SQLITE_MAX_*制限値 |

### SQLインジェクション対策

SQLiteは準備済みステートメント（Prepared Statement）とパラメータバインディングを提供:

```c
// 安全なパラメータバインディング
sqlite3_prepare_v2(db, "SELECT * FROM users WHERE id = ?", -1, &stmt, NULL);
sqlite3_bind_int(stmt, 1, user_id);
```

### リソース制限（DoS対策）

`src/sqliteLimit.h`で定義される制限値:

| 制限項目 | デフォルト値 | 説明 |
| --- | --- | --- |
| SQLITE_MAX_LENGTH | 1,000,000,000 | TEXT/BLOBの最大長（バイト） |
| SQLITE_MAX_SQL_LENGTH | 1,000,000,000 | SQL文の最大長（バイト） |
| SQLITE_MAX_COLUMN | 2,000 | テーブルの最大カラム数 |
| SQLITE_MAX_EXPR_DEPTH | 1,000 | 式ツリーの最大深度 |
| SQLITE_MAX_PARSER_DEPTH | 2,500 | パーサースタックの最大深度 |
| SQLITE_MAX_COMPOUND_SELECT | 500 | 複合SELECTの最大項数 |
| SQLITE_MAX_FUNCTION_ARG | 1,000 | 関数の最大引数数 |
| SQLITE_MAX_ATTACHED | 10 | 最大アタッチデータベース数 |
| SQLITE_MAX_VARIABLE_NUMBER | 32,766 | 最大バインド変数数 |
| SQLITE_MAX_TRIGGER_DEPTH | 1,000 | トリガーの最大再帰深度 |
| SQLITE_MAX_ALLOCATION_SIZE | 2,147,483,391 | 単一メモリ割り当ての最大サイズ |

### API Armor（防御的プログラミング）

`SQLITE_ENABLE_API_ARMOR`コンパイルオプションにより、API呼び出し時の引数検証が強化される:

- データベースハンドルの有効性チェック（`sqlite3SafetyCheckOk()`）
- NULL引数のチェック
- 範囲外インデックスのチェック
- 不正な状態での呼び出し検出

対象ファイル: `src/main.c`, `src/auth.c`, `src/backup.c`, `src/mutex.c`など21ファイル

## 排他制御・同時実行制御

### ファイルロック機構

SQLiteは複数のロック方式をサポート（`src/os_unix.c`）:

| ロック方式 | 説明 | プラットフォーム |
| --- | --- | --- |
| POSIX Advisory Locks | デフォルトのロック方式 | Unix/Linux |
| flock() | BSD系ファイルロック | Unix |
| dot-file | 補助ファイルによるロック | クロスプラットフォーム |
| AFP Locking | Apple Filing Protocol | macOS |
| Named Semaphore | 名前付きセマフォ | VxWorks |
| Proxy Locking | プロキシロック | macOS |

### ロックレベル

| レベル | 説明 |
| --- | --- |
| UNLOCKED | ロックなし |
| SHARED | 読み取り用共有ロック |
| RESERVED | 書き込み予約（他の読み取りは許可） |
| PENDING | 書き込み保留中 |
| EXCLUSIVE | 排他ロック |

### スレッドセーフティ

`SQLITE_THREADSAFE`コンパイルオプション:

| 値 | モード | 説明 |
| --- | --- | --- |
| 0 | シングルスレッド | mutex無効、スレッドセーフなし |
| 1 | シリアライズ | 完全スレッドセーフ（デフォルト） |
| 2 | マルチスレッド | 同一接続を複数スレッドで共有不可 |

Mutex実装: `src/mutex.c`, `src/mutex_unix.c`, `src/mutex_w32.c`

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| SQLログ | 実行されたSQL文（SQLITE_ENABLE_SQLLOG有効時） | アプリケーション依存 |
| 拡張トレース | I/O操作のトレース（SQLITE_ENABLE_IOTRACE有効時） | デバッグ用 |
| VFSトレース | VFS操作のトレース（ext/misc/vfstrace.c） | 拡張機能 |
| メモリトレース | メモリ割り当て追跡（ext/misc/memtrace.c） | 拡張機能 |

**標準機能:**
SQLiteコアには監査ログ機能は組み込まれていない。以下のコールバック機構を使用してアプリケーション側で実装可能:
- `sqlite3_trace_v2()` - SQL実行のトレース
- `sqlite3_profile()` - SQL実行時間の計測

## データ整合性保護

### トランザクション管理

- ACID特性の完全サポート
- ロールバックジャーナル / WAL（Write-Ahead Logging）モード
- 自動チェックポイント（WALモード時、デフォルト1000フレーム）

### チェックサム

- `PRAGMA integrity_check` - データベース整合性検証
- `ext/misc/cksumvfs.c` - VFSレベルでのチェックサム検証（拡張）

## 備考

### コンパイル時セキュリティオプション

| オプション | 効果 |
| --- | --- |
| SQLITE_ENABLE_API_ARMOR | API呼び出しの引数検証強化 |
| SQLITE_OMIT_AUTHORIZATION | 認可機能の無効化（組み込み向け） |
| SQLITE_SECURE_DELETE | 削除時のデータゼロクリア |
| SQLITE_OMIT_LOAD_EXTENSION | 拡張機能ロードの無効化 |
| SQLITE_OMIT_DESERIALIZE | デシリアライズ機能の無効化 |

### セキュリティに関する推奨事項

1. **最小権限の原則**: データベースファイルへのアクセス権限を必要最小限に設定
2. **入力検証**: ユーザー入力はパラメータバインディングを使用
3. **拡張機能の制限**: `sqlite3_enable_load_extension()`を無効化し、信頼できない拡張機能のロードを防止
4. **リソース制限**: アプリケーション要件に応じてSQLITE_MAX_*値を調整
5. **WALモードの活用**: 並行読み取り性能向上とクラッシュリカバリの改善

### 脆弱性報告

セキュリティに関する問題は、SQLite公式フォーラム（https://sqlite.org/forum/）または開発者への直接連絡（drh at sqlite dot org）で報告可能。
