# 機能設計書 1-SQLiteライブラリ初期化

## 概要

本ドキュメントは、SQLiteデータベースエンジンにおけるライブラリ全体の初期化処理について、その設計と実装の詳細を記述する。sqlite3_initialize()関数を中心とした初期化処理の仕組み、依存サブシステム、およびスレッドセーフティの考慮事項について解説する。

### 本機能の処理概要

**業務上の目的・背景**：SQLiteを使用するアプリケーションは、データベース操作を行う前にライブラリの初期化が必要である。この初期化処理により、メモリ管理システム、VFS（Virtual File System）、ミューテックスサブシステムなど、SQLiteの動作に必要な基盤が構築される。適切な初期化により、マルチスレッド環境での安全な動作や、プラットフォーム固有のファイルシステムアクセスが保証される。

**機能の利用シーン**：
- アプリケーション起動時にSQLiteライブラリを使用開始する場面
- sqlite3_open()でデータベース接続を開く際（SQLITE_OMIT_AUTOINITが未定義の場合は自動呼び出し）
- sqlite3_shutdown()後に再度SQLiteを使用する場面
- WASMモジュールの初期化時（sqlite3InitModule()経由）

**主要な処理内容**：
1. 初期化状態の確認（sqlite3GlobalConfig.isInit）によるno-op判定
2. グローバルミューテックスの取得によるスレッドセーフな初期化開始
3. メモリアロケータの初期化（sqlite3MallocInit()）
4. ミューテックスサブシステムの初期化（sqlite3MutexInit()）
5. VFS（Virtual File System）の初期化（sqlite3_os_init()）
6. 拡張初期化処理の実行（SQLITE_EXTRA_INIT定義時）
7. 初期化完了フラグの設定

**関連システム・外部連携**：
- OS抽象化層（VFS）：os_unix.c、os_win.cなどプラットフォーム固有の実装
- メモリ管理サブシステム：malloc.c、mem1.c等
- ミューテックスサブシステム：mutex.c、mutex_unix.c等

**権限による制御**：本機能は権限による制御は行わない。初期化は任意のスレッドから呼び出し可能だが、SQLiteが自動的にシリアライズする。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | WASM テストページインデックス | 主画面 | WASM sqlite3モジュールの初期化処理 |
| 2 | WASM 配布用インデックス | 主画面 | 配布用WASMモジュールの初期化処理 |
| 3 | 基本デモ（UIスレッド版） | 主画面 | sqlite3InitModule()によるWASMモジュール初期化 |
| 4 | 基本デモ（Worker版） | 主画面 | Worker内でのsqlite3InitModule()初期化 |
| 5 | JS Storage デモ | 主画面 | kvvfs対応sqlite3モジュール初期化 |
| 6 | Worker1 API デモ | 主画面 | Worker1 APIでのモジュール初期化 |
| 7 | Worker1 Promiser デモ | 主画面 | Promiser APIでのモジュール初期化 |
| 8 | SQLite3 Fiddle | 主画面 | fiddle.jsによるWASM CLIシェル初期化 |
| 9 | Speedtest1（メインスレッド版） | 主画面 | speedtest1 WASM版の初期化 |
| 10 | Speedtest1（Worker版） | 主画面 | Worker内でのspeedtest1初期化 |

## 機能種別

ライブラリ初期化処理（システム基盤機能）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| （なし） | - | - | sqlite3_initialize()は引数を取らない | - |

### 入力データソース

- sqlite3GlobalConfig構造体（グローバル設定情報）
- SQLITE_THREADSAFE、SQLITE_OMIT_AUTOINITなどのコンパイル時フラグ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | int | SQLITE_OK（成功）、または該当するエラーコード |

### 出力先

- sqlite3GlobalConfig.isInit：初期化完了フラグ
- 各サブシステムの内部状態

## 処理フロー

### 処理シーケンス

```
1. ポインタサイズの整合性アサーション確認
   └─ SQLITE_PTRSIZE==sizeof(char*)を検証

2. 初期化済みチェック
   └─ sqlite3GlobalConfig.isInitがtrueの場合はSQLITE_OKを返して終了

3. 低レベル初期化（ミューテックス保護なし）
   └─ sqlite3MutexInit()でミューテックスサブシステムを初期化

4. メインミューテックス取得
   └─ STATIC_MAINミューテックスを取得してシリアライズ

5. 再帰呼び出しチェック
   └─ 同一スレッドからの再帰呼び出しを検出して対応

6. メモリアロケータ初期化
   └─ sqlite3MallocInit()でメモリ管理システムを初期化

7. VFS初期化
   └─ sqlite3_os_init()でOS抽象化層を初期化

8. 拡張初期化（条件付き）
   └─ SQLITE_EXTRA_INIT定義時のみ追加初期化を実行

9. 初期化完了フラグ設定
   └─ sqlite3GlobalConfig.isInit = 1

10. ミューテックス解放
    └─ メインミューテックスを解放してリターン
```

### フローチャート

```mermaid
flowchart TD
    A[sqlite3_initialize開始] --> B{isInit == true?}
    B -->|Yes| C[SQLITE_OK返却]
    B -->|No| D[sqlite3MutexInit]
    D --> E[メインミューテックス取得]
    E --> F{再帰呼び出し?}
    F -->|Yes| G[早期リターン]
    F -->|No| H[sqlite3MallocInit]
    H --> I[sqlite3_os_init]
    I --> J{SQLITE_EXTRA_INIT定義?}
    J -->|Yes| K[追加初期化実行]
    J -->|No| L[isInit = 1設定]
    K --> L
    L --> M[ミューテックス解放]
    M --> N[SQLITE_OK返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 冪等性保証 | 複数回呼び出しても問題なく動作する | 常時 |
| BR-002 | 自動初期化 | SQLITE_OMIT_AUTOINIT未定義時はsqlite3_open等で自動呼び出し | コンパイル設定による |
| BR-003 | シリアライズ保証 | 複数スレッドからの同時呼び出しを自動的にシリアライズ | マルチスレッド環境 |
| BR-004 | 再帰呼び出し許容 | sqlite3_os_init()からの再帰呼び出しを安全に処理 | VFS登録時 |

### 計算ロジック

特に計算ロジックは存在しない。

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない（ライブラリ初期化のみ）。

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| SQLITE_NOMEM | メモリ不足 | メモリアロケータ初期化失敗時 | メモリを解放して再試行 |
| SQLITE_ERROR | 一般エラー | VFS初期化失敗時 | 環境設定を確認 |
| SQLITE_MISUSE | 不正使用 | sqlite3_shutdown後の不正な状態 | 適切な順序で呼び出し |

### リトライ仕様

初期化処理は冪等性を保証しているため、エラー発生時は原因を解消してから再度呼び出すことで問題なく動作する。

## トランザクション仕様

本機能はトランザクション処理を伴わない。

## パフォーマンス要件

- 初期化処理は通常数ミリ秒以内に完了する
- 2回目以降の呼び出しは即座にSQLITE_OKを返却（no-op）
- スレッドセーフ処理のためのミューテックス取得オーバーヘッドが存在

## セキュリティ考慮事項

- 初期化処理自体にセキュリティ上の考慮事項はない
- カスタムメモリアロケータやVFSを使用する場合は、それらの実装のセキュリティを考慮する必要がある

## 備考

- sqlite3_shutdown()との対称性を持つ設計
- コンパイル時オプションにより動作が変化する（SQLITE_THREADSAFE、SQLITE_OMIT_AUTOINIT等）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、SQLiteのグローバル設定を管理する構造体を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | sqliteInt.h | `src/sqliteInt.h` | Sqlite3Config構造体の定義、isInit等のフィールド |
| 1-2 | global.c | `src/global.c` | sqlite3GlobalConfig変数の定義と初期値 |

**読解のコツ**: sqlite3GlobalConfigはSQLiteの全コンポーネントで共有されるグローバル状態を保持する。特にisInit、isMallocInit、isMutexInitフラグが初期化状態の管理に使用される。

#### Step 2: エントリーポイントを理解する

処理の起点となるsqlite3_initialize()関数を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | main.c | `src/main.c` | sqlite3_initialize()関数の実装 |

**主要処理フロー**:
1. **190-195行目**: 関数シグネチャと変数宣言、ミューテックス用変数pMainMtx
2. **206-216行目**: ポインタサイズ検証とisInitフラグによる早期リターン判定
3. **218-262行目**: ミューテックスサブシステムの初期化（sqlite3MutexInit()呼び出し）
4. **264-271行目**: メインミューテックス取得とシリアライズ処理
5. **280-300行目**: メモリアロケータ初期化（sqlite3MallocInit()）
6. **302-330行目**: VFS初期化（sqlite3_os_init()）
7. **350-360行目**: 初期化完了フラグ設定とリターン

#### Step 3: ミューテックスサブシステムを理解する

スレッドセーフな初期化を実現するミューテックス処理を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | mutex.c | `src/mutex.c` | sqlite3MutexInit()の実装 |
| 3-2 | mutex_unix.c | `src/mutex_unix.c` | UNIX系OSでのpthreadベースミューテックス実装 |

**主要処理フロー**:
- ミューテックス実装の選択（noop、unix、w32）
- 静的ミューテックスの初期化

#### Step 4: メモリ管理サブシステムを理解する

メモリアロケータの初期化処理を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | malloc.c | `src/malloc.c` | sqlite3MallocInit()の実装 |
| 4-2 | mem1.c | `src/mem1.c` | システムmalloc()ベースのメモリアロケータ |

**主要処理フロー**:
- デフォルトメモリアロケータの設定
- カスタムメモリアロケータ対応

#### Step 5: VFSサブシステムを理解する

OS抽象化層の初期化を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | os.c | `src/os.c` | sqlite3_os_init()のディスパッチャ |
| 5-2 | os_unix.c | `src/os_unix.c` | UNIX系VFSの初期化とデフォルトVFS登録 |

**主要処理フロー**:
- プラットフォーム固有VFSの登録
- デフォルトVFSの設定

### プログラム呼び出し階層図

```
sqlite3_initialize()                    [main.c:190]
    │
    ├─ sqlite3MutexInit()              [mutex.c]
    │      └─ プラットフォーム固有初期化
    │
    ├─ sqlite3_mutex_enter()           [mutex.c]
    │      └─ STATIC_MAINミューテックス取得
    │
    ├─ sqlite3MallocInit()             [malloc.c]
    │      └─ メモリアロケータ設定
    │
    ├─ sqlite3_os_init()               [os.c → os_unix.c]
    │      ├─ unixCurrentTimeInt64()   時刻取得関数登録
    │      ├─ sqlite3_vfs_register()   VFS登録
    │      └─ unixDlOpen()等           動的ライブラリ関数登録
    │
    └─ sqlite3_mutex_leave()           [mutex.c]
           └─ ミューテックス解放
```

### データフロー図

```
[入力]                    [処理]                         [出力]

sqlite3GlobalConfig  ───▶ sqlite3_initialize()    ───▶ sqlite3GlobalConfig.isInit = 1
(isInit=0)                    │
                              ├─▶ mutex subsystem    ───▶ ミューテックス初期化完了
                              │
                              ├─▶ malloc subsystem   ───▶ メモリアロケータ初期化完了
                              │
                              └─▶ VFS subsystem      ───▶ デフォルトVFS登録完了
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| main.c | `src/main.c` | ソース | sqlite3_initialize()のメイン実装 |
| sqliteInt.h | `src/sqliteInt.h` | ヘッダー | 内部構造体・定数定義 |
| global.c | `src/global.c` | ソース | グローバル変数定義 |
| mutex.c | `src/mutex.c` | ソース | ミューテックスAPIのディスパッチャ |
| mutex_unix.c | `src/mutex_unix.c` | ソース | UNIX系ミューテックス実装 |
| mutex_noop.c | `src/mutex_noop.c` | ソース | シングルスレッド用ダミー実装 |
| malloc.c | `src/malloc.c` | ソース | メモリ管理API |
| mem1.c | `src/mem1.c` | ソース | システムmalloc使用のアロケータ |
| mem5.c | `src/mem5.c` | ソース | メモリプール使用のアロケータ |
| os.c | `src/os.c` | ソース | OS抽象化層のディスパッチャ |
| os_unix.c | `src/os_unix.c` | ソース | UNIX系VFS実装 |
