# 機能設計書 15-トランザクション管理

## 概要

本ドキュメントは、SQLiteにおけるトランザクション管理機能について記述する。BEGIN、COMMIT、ROLLBACKコマンドによるトランザクション制御、およびSAVEPOINTによるネストされたトランザクションのサポートを提供する。

### 本機能の処理概要

**業務上の目的・背景**：データベースの一貫性と整合性を保証するため、複数のSQL文を単一のアトミックな操作としてグループ化する必要がある。トランザクション管理機能は、ACID特性（原子性、一貫性、独立性、永続性）を実現し、システム障害時のデータ保護と、並行アクセス時のデータ整合性を確保する。

**機能の利用シーン**：
- 複数テーブルへの一括更新（全て成功または全て失敗）
- バッチ処理での高速データ挿入（トランザクション内での一括コミット）
- エラー発生時のデータロールバック
- 複雑なビジネスロジックでのSAVEPOINT利用

**主要な処理内容**：
1. **BEGIN**: 新しいトランザクションの開始（DEFERRED/IMMEDIATE/EXCLUSIVE）
2. **COMMIT/END**: トランザクションの確定とデータの永続化
3. **ROLLBACK**: トランザクションの取り消しと変更の破棄
4. **SAVEPOINT**: ネストされたトランザクションポイントの作成
5. **RELEASE SAVEPOINT**: SAVEPOINTの解放
6. **ROLLBACK TO SAVEPOINT**: 特定SAVEPOINTまでのロールバック

**関連システム・外部連携**：VDBEバイトコードエンジン、B-Treeストレージエンジン、ページャー、WAL

**権限による制御**：sqlite3_set_authorizer()で設定された認可コールバックにより、SQLITE_TRANSACTIONアクションコードでアクセス制御が可能

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | 基本デモ（UIスレッド版） | 補助機能 | db.transaction()によるトランザクション制御 |

## 機能種別

トランザクション制御 / データ整合性管理

## 入力仕様

### 入力パラメータ（BEGIN）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| type | int | Yes | トランザクションタイプ（TK_DEFERRED/TK_IMMEDIATE/TK_EXCLUSIVE） | 有効なトークン |

### 入力パラメータ（COMMIT/ROLLBACK）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| eType | int | Yes | コマンドタイプ（TK_COMMIT/TK_END/TK_ROLLBACK） | 有効なトークン |

### 入力パラメータ（SAVEPOINT）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| op | int | Yes | 操作タイプ（SAVEPOINT_BEGIN/SAVEPOINT_RELEASE/SAVEPOINT_ROLLBACK） | 有効な操作 |
| pName | Token* | Yes | SAVEPOINT名 | 有効な識別子 |

### 入力データソース

- SQL文解析結果（パーサーから渡されるトークン情報）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 戻り値 | void | 直接的な戻り値なし |
| 自動コミットモード | int | db->autoCommit状態 |

### 出力先

- データベースファイル（COMMIT時）
- ジャーナルファイル/WALファイル

## 処理フロー

### 処理シーケンス（BEGIN）

```
1. 認証チェック
   └─ SQLITE_TRANSACTION, "BEGIN"

2. VDBE取得
   └─ sqlite3GetVdbe()

3. トランザクションタイプ判定
   └─ DEFERRED: トランザクション開始なし（遅延）
   └─ IMMEDIATE: 書き込みトランザクション開始
   └─ EXCLUSIVE: 排他トランザクション開始

4. 各データベースへのトランザクション
   └─ OP_Transactionを各DBに対して生成

5. 自動コミット無効化
   └─ OP_AutoCommit (p1=0)
```

### 処理シーケンス（COMMIT/ROLLBACK）

```
1. 認証チェック
   └─ SQLITE_TRANSACTION, "COMMIT"/"ROLLBACK"

2. VDBE取得
   └─ sqlite3GetVdbe()

3. 自動コミット有効化
   └─ OP_AutoCommit (p1=1, p2=isRollback)
```

### 処理シーケンス（SAVEPOINT）

```
1. SAVEPOINT名取得
   └─ sqlite3NameFromToken()

2. 認証チェック
   └─ SQLITE_SAVEPOINT, op名, SAVEPOINT名

3. OP_Savepoint生成
   └─ op, SAVEPOINT名
```

### フローチャート

```mermaid
flowchart TD
    A[トランザクションコマンド] --> B{コマンド種別}
    B -->|BEGIN| C[認証チェック]
    B -->|COMMIT/ROLLBACK| D[認証チェック]
    B -->|SAVEPOINT| E[認証チェック]

    C --> F{認証成功?}
    F -->|No| Z[終了]
    F -->|Yes| G{トランザクションタイプ}
    G -->|DEFERRED| H[OP_AutoCommit生成]
    G -->|IMMEDIATE| I[OP_Transaction生成]
    G -->|EXCLUSIVE| J[OP_Transaction生成]
    I --> H
    J --> H
    H --> Z

    D --> K{認証成功?}
    K -->|No| Z
    K -->|Yes| L[OP_AutoCommit生成]
    L --> Z

    E --> M{認証成功?}
    M -->|No| Z
    M -->|Yes| N[OP_Savepoint生成]
    N --> Z
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-15-01 | DEFERRED遅延 | DEFERREDトランザクションは最初のDB操作時に実際に開始 | BEGIN DEFERRED |
| BR-15-02 | IMMEDIATE即時 | IMMEDIATEは即座に書き込みロック取得 | BEGIN IMMEDIATE |
| BR-15-03 | EXCLUSIVE排他 | EXCLUSIVEは即座に排他ロック取得 | BEGIN EXCLUSIVE |
| BR-15-04 | 暗黙ROLLBACK | エラー発生時は暗黙的にROLLBACK | エラー時 |
| BR-15-05 | READ ONLY DB | 読み取り専用DBへのEXCLUSIVEは読み取りトランザクション | READ ONLY時 |

### 計算ロジック

**トランザクションタイプの判定**:
- `type == TK_DEFERRED`: トランザクション遅延（OP_Transaction生成なし）
- `type == TK_EXCLUSIVE` かつ 読み取り専用: `eTxnType = 0`（読み取り）
- `type == TK_EXCLUSIVE`: `eTxnType = 2`（排他）
- `type == TK_IMMEDIATE`: `eTxnType = 1`（書き込み）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象 | 操作種別 | 概要 |
|-----|-----|---------|------|
| BEGIN | ロック | 取得 | トランザクションロック取得 |
| COMMIT | データファイル | 書き込み | 変更内容の永続化 |
| ROLLBACK | データファイル | 復元 | 変更前状態への復元 |
| SAVEPOINT | ジャーナル | 記録 | 復元ポイントの記録 |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| SQLITE_BUSY | ロックタイムアウト | 他の接続がロック保持 | リトライまたはタイムアウト設定 |
| SQLITE_LOCKED | テーブルロック | 同一接続内でのデッドロック | トランザクション構造見直し |
| SQLITE_READONLY | 読み取り専用 | 読み取り専用DBへの書き込み | 書き込み権限確認 |
| SQLITE_AUTH | 認証エラー | authorizerが拒否 | 権限を確認 |

### リトライ仕様

- SQLITE_BUSYエラー時、sqlite3_busy_handler()またはsqlite3_busy_timeout()でリトライポリシー設定可能
- デフォルトでは即座にエラー返却

## トランザクション仕様

- **自動コミットモード**: デフォルトでは各SQL文が自動的にコミットされる
- **明示的トランザクション**: BEGINで自動コミット無効化、COMMIT/ROLLBACKで再有効化
- **ネストトランザクション**: SAVEPOINTによる疑似的なネストをサポート
- **ジャーナルモード**: DELETE、TRUNCATE、PERSIST、MEMORY、WAL、OFF

## パフォーマンス要件

- トランザクション開始は低コスト（数マイクロ秒）
- COMMITはディスクI/Oを伴うため、数ミリ秒〜数十ミリ秒
- バッチ挿入時はトランザクション内で実行することで大幅な高速化

## セキュリティ考慮事項

- sqlite3_set_authorizer()によるトランザクションコマンドの制御
- SQLITE_TRANSACTIONアクションでBEGIN/COMMIT/ROLLBACKを監視
- SQLITE_SAVEPOINTアクションでSAVEPOINT操作を監視

## 備考

- SQLiteは軽量なファイルベースDBのため、長時間のトランザクションは他の接続をブロックする可能性
- WALモードでは読み取りと書き込みの並行実行が可能
- 自動コミットモードでもDDL文（CREATE TABLE等）は暗黙的トランザクションで実行

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | sqliteInt.h | `src/sqliteInt.h` | sqlite3構造体のautoCommitフィールド |
| 1-2 | vdbeInt.h | `src/vdbeInt.h` | Vdbe構造体のトランザクション関連フィールド |

**読解のコツ**: db->autoCommitが0の場合、明示的トランザクション中であることを示す。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | build.c | `src/build.c` | sqlite3BeginTransaction()関数（5239行目〜）がBEGINのエントリー |
| 2-2 | build.c | `src/build.c` | sqlite3EndTransaction()関数（5275行目〜）がCOMMIT/ROLLBACKのエントリー |
| 2-3 | build.c | `src/build.c` | sqlite3Savepoint()関数（5297行目〜）がSAVEPOINTのエントリー |

**主要処理フロー（BEGIN）**:
1. **5247行目**: 認証チェック（SQLITE_TRANSACTION, "BEGIN"）
2. **5250行目**: VDBE取得
3. **5252-5265行目**: DEFERREDでない場合、各DBにOP_Transaction生成
4. **5256-5262行目**: トランザクションタイプ判定（読み取り/書き込み/排他）
5. **5263-5264行目**: OP_Transaction、OP_UsesBtree
6. **5267行目**: OP_AutoCommit (p1=0)

**主要処理フロー（COMMIT/ROLLBACK）**:
1. **5281行目**: eType判定（TK_COMMIT/TK_END/TK_ROLLBACK）
2. **5282行目**: isRollbackフラグ設定
3. **5283-5285行目**: 認証チェック
4. **5289行目**: OP_AutoCommit (p1=1, p2=isRollback)

**主要処理フロー（SAVEPOINT）**:
1. **5298行目**: SAVEPOINT名取得
2. **5302-5303行目**: 操作名配列（"BEGIN", "RELEASE", "ROLLBACK"）
3. **5305行目**: 認証チェック（SQLITE_SAVEPOINT）
4. **5309行目**: OP_Savepoint生成

### プログラム呼び出し階層図

```
sqlite3BeginTransaction() [build.c:5239]
    │
    ├─ sqlite3AuthCheck() [auth.c]
    │      └─ SQLITE_TRANSACTION, "BEGIN"
    │
    ├─ sqlite3GetVdbe() [vdbeaux.c]
    │
    ├─ (各DBに対してループ)
    │      ├─ sqlite3BtreeIsReadonly() [btree.c]
    │      ├─ sqlite3VdbeAddOp2(OP_Transaction)
    │      └─ sqlite3VdbeUsesBtree()
    │
    └─ sqlite3VdbeAddOp0(OP_AutoCommit)

sqlite3EndTransaction() [build.c:5275]
    │
    ├─ sqlite3AuthCheck() [auth.c]
    │      └─ SQLITE_TRANSACTION, "COMMIT"/"ROLLBACK"
    │
    ├─ sqlite3GetVdbe() [vdbeaux.c]
    │
    └─ sqlite3VdbeAddOp2(OP_AutoCommit, 1, isRollback)

sqlite3Savepoint() [build.c:5297]
    │
    ├─ sqlite3NameFromToken() [build.c]
    │
    ├─ sqlite3AuthCheck() [auth.c]
    │      └─ SQLITE_SAVEPOINT, op名, SAVEPOINT名
    │
    └─ sqlite3VdbeAddOp4(OP_Savepoint, op, 0, 0, zName)
```

### データフロー図

```
[入力]                     [処理]                      [出力]

BEGIN TRANSACTION ──▶ sqlite3BeginTransaction()
                              │
                              ▼
                      ┌───────┴───────┐
                      ▼               ▼
               DEFERRED         IMMEDIATE/EXCLUSIVE
                      │               │
                      ▼               ▼
              OP_AutoCommit    OP_Transaction
                   (0)         各DBに対して
                      │               │
                      └───────┬───────┘
                              ▼
                      OP_AutoCommit(0)
                              │
                              ▼
                    自動コミット無効 ──────▶ db->autoCommit = 0


COMMIT/ROLLBACK ────▶ sqlite3EndTransaction()
                              │
                              ▼
                    OP_AutoCommit(1, isRollback)
                              │
                              ▼
                    自動コミット有効 ──────▶ db->autoCommit = 1
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| build.c | `src/build.c` | ソース | BEGIN/COMMIT/ROLLBACK/SAVEPOINTのコード生成 |
| vdbe.c | `src/vdbe.c` | ソース | OP_Transaction, OP_AutoCommit, OP_Savepointの実行 |
| pager.c | `src/pager.c` | ソース | トランザクションのディスク操作 |
| wal.c | `src/wal.c` | ソース | WALモードでのトランザクション処理 |
| btree.c | `src/btree.c` | ソース | B-Treeレベルのトランザクション管理 |
| auth.c | `src/auth.c` | ソース | 認証処理 |
| sqliteInt.h | `src/sqliteInt.h` | ヘッダー | sqlite3構造体定義（autoCommit等） |
| vdbe.h | `src/vdbe.h` | ヘッダー | VDBE命令定義 |
