# 機能設計書 30-JSON関数

## 概要

本ドキュメントは、SQLiteのJSON関数（json.c）の機能設計を記述する。これらの関数はJSONデータの解析、生成、操作を行う機能を提供する。

### 本機能の処理概要

JSON関数は、RFC-8259準拠のJSONテキストおよびSQLite独自のJSONB（バイナリJSON）形式の操作をサポートする。JSON-5拡張も入力として受け入れる。

**業務上の目的・背景**：現代のアプリケーションではJSONが広く使用されている。SQLiteでJSONを直接操作できることで、JSONデータの保存・検索・更新を効率的に行え、アプリケーション層でのJSON処理を削減できる。

**機能の利用シーン**：
- WebAPIレスポンスの保存と検索
- 構造化ログの分析
- 設定データの管理
- ドキュメント指向データの操作
- JSONBによる高速な内部処理

**主要な処理内容**：
1. JSONテキストの解析とJSONBへの変換
2. JSONパスによる値の抽出
3. JSON値の挿入・更新・削除
4. JSON配列/オブジェクトの操作
5. JSONテキストとJSONB間の変換

**関連システム・外部連携**：特になし（自己完結型）

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | CLIメイン | 主機能 | JSONクエリ実行 |

## 機能種別

SQL関数 / データ変換

## 入力仕様

### 入力パラメータ

| 関数名 | 引数 | 説明 |
|--------|-----|------|
| json(X) | JSON | JSONを正規化して返却 |
| jsonb(X) | JSON | JSONBバイナリを返却 |
| json_array(...) | values | JSON配列を生成 |
| json_object(...) | key-value pairs | JSONオブジェクトを生成 |
| json_extract(X, path...) | JSON, paths | パスで値を抽出 |
| json_type(X, path) | JSON, path | 型名を返却 |
| json_valid(X) | JSON | 有効性チェック |
| json_set(X, path, value...) | JSON, path-value pairs | 値を設定 |
| json_insert(X, path, value...) | JSON, path-value pairs | 値を挿入（既存は保持） |
| json_replace(X, path, value...) | JSON, path-value pairs | 値を置換（新規は無視） |
| json_remove(X, path...) | JSON, paths | 値を削除 |
| json_each(X) | JSON | テーブル値関数 |
| json_tree(X) | JSON | 再帰テーブル値関数 |

### JSONパス構文

| 構文 | 説明 | 例 |
|------|------|-----|
| $ | ルート | $ |
| .key | オブジェクトキー | $.name |
| [N] | 配列インデックス | $[0] |
| [#] | 配列末尾（挿入用） | $[#] |
| [#-N] | 配列末尾からN番目 | $[#-1] |

## 出力仕様

### 出力データ

| 関数 | 戻り値型 | 例 |
|------|----------|-----|
| json | TEXT | '{"a":1}' |
| jsonb | BLOB | x'...' |
| json_extract | varies | 抽出値の型 |
| json_type | TEXT | 'object', 'array', 'string', 'integer', 'real', 'true', 'false', 'null' |
| json_valid | INTEGER | 0 or 1 |

### 出力先

- クエリ結果として返却

## 処理フロー

### 処理シーケンス

```
1. 入力解析
   └─ テキストJSON → JsonParse構造体
   └─ またはJSONBをそのまま使用

2. 操作実行
   └─ パス評価とノード探索
   └─ 値の取得/設定/削除

3. 出力変換
   └─ JsonParse → テキストJSON
   └─ またはJSONBをそのまま返却
```

### フローチャート

```mermaid
flowchart TD
    A[JSON入力] --> B{入力形式?}
    B -->|TEXT| C[JSONパース]
    B -->|BLOB| D[JSONBバリデーション]
    C --> E[JsonParse構造体]
    D --> E
    E --> F{操作種別}
    F -->|抽出| G[パス評価]
    F -->|変更| H[ノード編集]
    F -->|生成| I[新規構築]
    G --> J[結果返却]
    H --> K{出力形式}
    I --> K
    K -->|TEXT| L[JSON文字列化]
    K -->|BLOB| M[JSONB返却]
    L --> J
    M --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-30-01 | RFC-8259準拠 | 出力はRFC-8259準拠JSON | json*関数 |
| BR-30-02 | JSON-5入力対応 | 入力はJSON-5拡張を許容 | 全関数 |
| BR-30-03 | subtype設定 | TEXT出力にはJSON subtypeを設定 | json*関数 |
| BR-30-04 | ネスト制限 | 最大ネスト深度1000 | 全関数 |
| BR-30-05 | パス解決 | 存在しないパスはNULL返却 | 抽出系 |

### JSONBエンコーディング

| 要素型 | コード | 説明 |
|--------|--------|------|
| NULL | 0 | null |
| TRUE | 1 | true |
| FALSE | 2 | false |
| INT | 3 | RFC-8259整数 |
| INT5 | 4 | JSON-5整数（0x形式） |
| FLOAT | 5 | RFC-8259浮動小数点 |
| FLOAT5 | 6 | JSON-5浮動小数点 |
| TEXT | 7 | SQL/JSON両用テキスト |
| TEXTJ | 8 | JSONエスケープ付きテキスト |
| TEXT5 | 9 | JSON-5エスケープ付きテキスト |
| TEXTRAW | 10 | 未エスケープテキスト |
| ARRAY | 11 | 配列 |
| OBJECT | 12 | オブジェクト |

### JSONBヘッダ形式

ヘッダの上位4ビットがペイロードサイズを示す:
| 上位4ビット | サイズバイト数 | ヘッダ合計サイズ |
|------------|--------------|----------------|
| 0-11 | 0（値がサイズ） | 1 |
| 12 | 1 | 2 |
| 13 | 2 | 3 |
| 14 | 4 | 5 |
| 15 | 8 | 9 |

## データベース操作仕様

### 操作別データベース影響一覧

JSON関数はデータベースを直接変更しない（読み取り専用）。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NULL返却 | 解析エラー | 無効なJSON | 正しいJSONを入力 |
| SQLITE_ERROR | ネスト超過 | 深度1000超過 | ネストを減らす |
| SQLITE_NOMEM | メモリ不足 | 大容量JSON処理 | メモリ確保後再試行 |

### リトライ仕様

特になし。

## トランザクション仕様

- JSON関数はトランザクション状態に依存しない

## パフォーマンス要件

- JSONパース: O(n)（入力サイズ比例）
- パス評価: O(d)（深度比例）
- JSONB処理はテキストJSONの約3倍高速

## セキュリティ考慮事項

- 大量のネストでスタックオーバーフローの可能性（制限1000で防止）
- 信頼できないJSON入力は適切にバリデーション

## 備考

- SQLITE_OMIT_JSONでコンパイル時に無効化可能
- JSON_SUBTYPE=74（'J'のASCII）でJSON値を識別
- jsonb*関数はJSONBバイナリを返却（高速な内部処理用）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | json.c | `src/json.c` | JSONB要素型定数（125-137行目） |
| 1-2 | json.c | `src/json.c` | JsonString構造体（303-311行目） |
| 1-3 | json.c | `src/json.c` | JsonParse構造体（352-373行目） |

**読解のコツ**:
- JSONB_NULL〜JSONB_OBJECTの13種類の要素型
- JsonStringは文字列アキュムレータ（JSON生成用）
- JsonParseはパース結果を保持（aBlob[]にJSONBを格納）

#### Step 2: JSONBフォーマットを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | json.c | `src/json.c` | ヘッダフォーマットコメント（40-118行目） |

**主要処理フロー**:
- **40-64行目**: JSONB要素型の詳細説明
- **66-81行目**: ヘッダサイズ計算ルール
- **97-109行目**: ペイロードの意味（型ごと）

#### Step 3: パース処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | json.c | `src/json.c` | jsonIsSpace配列（153-172行目） |
| 3-2 | json.c | `src/json.c` | jsonIsOk配列（216-257行目） |
| 3-3 | json.c | `src/json.c` | JsonCache構造体（289-293行目） |

**主要処理フロー**:
- **153-172行目**: 独自isspace（標準ライブラリより高速）
- **216-257行目**: JSON特殊文字判定テーブル
- **267-268行目**: JSON_CACHE_ID, JSON_CACHE_SIZE定数

#### Step 4: 編集操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | json.c | `src/json.c` | JEDIT定数（376-380行目） |

**編集モード**:
- JEDIT_DEL (1): 削除
- JEDIT_REPL (2): 置換
- JEDIT_INS (3): 挿入（存在しない場合のみ）
- JEDIT_SET (4): 設定（挿入または置換）
- JEDIT_AINS (5): 配列挿入

### プログラム呼び出し階層図

```
json_extract()
    │
    ├─ jsonParseFuncArg()
    │      │
    │      ├─ JSONテキストパース
    │      │      └─ JsonParseへ格納
    │      │
    │      └─ JSONBバリデーション
    │
    └─ jsonLookupStep()
           │
           ├─ パス評価
           │
           └─ 値抽出

json_set()/json_insert()/json_replace()
    │
    ├─ jsonParseFuncArg()
    │
    └─ jsonLookupStep()
           │
           ├─ パス評価
           │
           └─ 編集操作（JEDIT_*）
                  │
                  └─ JSONB更新

jsonb()
    │
    └─ jsonParseFuncArg()
           │
           └─ JsonParse.aBlob[] 返却

json()
    │
    ├─ jsonParseFuncArg()
    │
    └─ jsonReturnText()
           │
           └─ JSONB → TEXT変換
```

### データフロー図

```
[入力]                    [処理]                      [出力]

TEXT JSON ──────────▶ jsonParseFuncArg()
                            │
                    JSONパース
                            │
                    JsonParse.aBlob[] (JSONB)
                            │
                    ┌───────┴───────┐
                    │               │
                 抽出           変更/生成
                    │               │
              jsonLookupStep()  jsonLookupStep()
                    │               │
              値取得           JSONB編集
                    │               │
                    └───────┬───────┘
                            │
                    ┌───────┴───────┐
                    │               │
                 TEXT           BLOB
                    │               │
              jsonReturnText()  JSONB返却
                    │               │
                    └───────┬───────┘
                            │
                            ▼
                    結果返却 ─────────────▶ クライアント
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| json.c | `src/json.c` | ソース | JSON関数主実装 |
| sqliteInt.h | `src/sqliteInt.h` | ヘッダ | 内部定義 |
| vdbe.c | `src/vdbe.c` | ソース | OP_Function実行 |
