# 機能設計書 46-ステータス情報

## 概要

SQLiteにおけるステータス情報の収集・取得機能の設計書。グローバルステータスとデータベース接続ごとのステータスの両方を管理する。

### 本機能の処理概要

SQLiteのステータス情報機能は、ライブラリの動作状態を監視・診断するための基盤機能である。

**業務上の目的・背景**：SQLiteを組み込んだアプリケーションでは、メモリ使用量、キャッシュ効率、パフォーマンス指標などを監視する必要がある。本機能により、現在値（nowValue）と最大値（mxValue/highwater mark）を追跡し、リソース使用状況の把握やチューニングを支援する。

**機能の利用シーン**：
- メモリ使用量の監視
- ページキャッシュのヒット率分析
- スキーマ使用メモリの把握
- プリペアドステートメントのメモリ使用量確認
- ルックアサイドメモリの効率分析

**主要な処理内容**：
1. グローバルステータス：sqlite3_status64()でライブラリ全体の統計を取得
2. DB接続ステータス：sqlite3_db_status64()で接続ごとの統計を取得
3. 内部更新：sqlite3StatusUp/Down/Highwaterで内部から統計を更新
4. ミューテックス保護：statMutex配列で項目ごとの保護ミューテックスを指定

**関連システム・外部連携**：
- メモリアロケータ（SQLITE_MUTEX_STATIC_MEM）
- ページキャッシュ（SQLITE_MUTEX_STATIC_PCACHE1）
- Pagerサブシステム（キャッシュ統計）
- B-treeサブシステム（スキーマ・ステートメント統計）

**権限による制御**：特になし（公開APIのためアプリケーションから自由に呼び出し可能）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | （API機能） | - | sqlite3_status(), sqlite3_db_status()で呼び出し |

## 機能種別

ステータス監視 / 診断機能

## 入力仕様

### 入力パラメータ（sqlite3_status64）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| op | int | Yes | ステータス項目（SQLITE_STATUS_*） | 0-9の範囲 |
| pCurrent | sqlite3_int64* | Yes | 現在値の出力先 | NULL不可 |
| pHighwater | sqlite3_int64* | Yes | 最大値の出力先 | NULL不可 |
| resetFlag | int | Yes | 最大値リセットフラグ | true/false |

### 入力パラメータ（sqlite3_db_status64）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| db | sqlite3* | Yes | データベース接続 | 有効な接続 |
| op | int | Yes | ステータス項目（SQLITE_DBSTATUS_*） | 有効な項目 |
| pCurrent | sqlite3_int64* | Yes | 現在値の出力先 | NULL不可 |
| pHighwtr | sqlite3_int64* | Yes | 最大値の出力先 | NULL不可 |
| resetFlag | int | Yes | 最大値リセットフラグ | true/false |

### 入力データソース

- アプリケーションからの直接呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 現在値 | sqlite3_int64 | 指定項目の現在値 |
| 最大値 | sqlite3_int64 | 指定項目の最大値（ハイウォーターマーク） |
| 戻り値 | int | SQLITE_OK または SQLITE_MISUSE |

### 出力先

- 呼び出し元の指定ポインタ

## 処理フロー

### 処理シーケンス

```
1. sqlite3_status64(op, pCurrent, pHighwater, resetFlag)
   ├─ 範囲チェック（op < 10）
   ├─ API Armorチェック（NULLポインタ）
   ├─ statMutex[op]に応じたミューテックス取得
   │   ├─ 0: sqlite3MallocMutex()
   │   └─ 1: sqlite3Pcache1Mutex()
   ├─ nowValue[op]をpCurrentに格納
   ├─ mxValue[op]をpHighwaterに格納
   ├─ resetFlag時: mxValue[op] = nowValue[op]
   └─ ミューテックス解放

2. sqlite3_db_status64(db, op, pCurrent, pHighwtr, resetFlag)
   ├─ db->mutex取得
   ├─ switch(op)で項目別処理
   │   ├─ LOOKASIDE_USED: ルックアサイド使用数計算
   │   ├─ CACHE_USED: 全Pagerメモリ使用量合計
   │   ├─ SCHEMA_USED: スキーマメモリ使用量計算
   │   ├─ STMT_USED: 全VDBEメモリ使用量計算
   │   ├─ CACHE_HIT/MISS/WRITE: キャッシュ統計取得
   │   └─ その他項目
   └─ db->mutex解放

3. 内部更新関数
   ├─ sqlite3StatusUp(op, N): nowValue += N, mxValue更新
   ├─ sqlite3StatusDown(op, N): nowValue -= N
   └─ sqlite3StatusHighwater(op, X): mxValue更新（現在値なし）
```

### フローチャート

```mermaid
flowchart TD
    A[sqlite3_status64] --> B{op範囲チェック}
    B -->|範囲外| C[SQLITE_MISUSE]
    B -->|OK| D{statMutex[op]}
    D -->|0| E[MallocMutex取得]
    D -->|1| F[Pcache1Mutex取得]
    E --> G[現在値・最大値取得]
    F --> G
    G --> H{resetFlag?}
    H -->|Yes| I[mxValue = nowValue]
    H -->|No| J[ミューテックス解放]
    I --> J
    J --> K[SQLITE_OK]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | ミューテックス選択 | statMutex配列で項目ごとの保護ミューテックスを決定 | 常時 |
| BR-46-02 | ハイウォーターマーク更新 | StatusUpで現在値が最大値を超えたら更新 | nowValue > mxValue時 |
| BR-46-03 | リセット機能 | resetFlag=trueで最大値を現在値にリセット | API呼び出し時 |
| BR-46-04 | 64ビット/32ビット対応 | SQLITE_PTRSIZE>4で64ビット、それ以外で32ビット | コンパイル時 |

### グローバルステータス項目

| 項目 | インデックス | ミューテックス | 説明 |
|------|------------|---------------|------|
| SQLITE_STATUS_MEMORY_USED | 0 | MallocMutex | メモリ使用量 |
| SQLITE_STATUS_PAGECACHE_USED | 1 | Pcache1Mutex | ページキャッシュ使用数 |
| SQLITE_STATUS_PAGECACHE_OVERFLOW | 2 | Pcache1Mutex | ページキャッシュオーバーフロー |
| SQLITE_STATUS_MALLOC_SIZE | 5 | MallocMutex | 最大malloc要求サイズ |
| SQLITE_STATUS_PARSER_STACK | 6 | MallocMutex | パーサースタック深度 |
| SQLITE_STATUS_PAGECACHE_SIZE | 7 | Pcache1Mutex | ページキャッシュサイズ |
| SQLITE_STATUS_MALLOC_COUNT | 9 | MallocMutex | malloc回数 |

### DB接続ステータス項目

| 項目 | 説明 |
|------|------|
| SQLITE_DBSTATUS_LOOKASIDE_USED | ルックアサイドスロット使用数 |
| SQLITE_DBSTATUS_CACHE_USED | キャッシュ使用メモリ |
| SQLITE_DBSTATUS_SCHEMA_USED | スキーマ使用メモリ |
| SQLITE_DBSTATUS_STMT_USED | ステートメント使用メモリ |
| SQLITE_DBSTATUS_CACHE_HIT | キャッシュヒット数 |
| SQLITE_DBSTATUS_CACHE_MISS | キャッシュミス数 |
| SQLITE_DBSTATUS_CACHE_WRITE | キャッシュ書き込み数 |
| SQLITE_DBSTATUS_DEFERRED_FKS | 遅延外部キー制約数 |

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベーステーブルを直接操作しない（ステータス取得機能のため）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| SQLITE_MISUSE | 不正使用 | op範囲外 | 正しい項目を指定 |
| SQLITE_MISUSE | 不正使用 | NULLポインタ | 有効なポインタを渡す |
| SQLITE_ERROR | エラー | 未知のop（db_status） | 正しい項目を指定 |

### リトライ仕様

ステータス取得はリトライ不要（同期処理）

## トランザクション仕様

ステータス取得はトランザクションの対象外（読み取り専用）

## パフォーマンス要件

- ステータス取得は軽量操作
- ミューテックスは最小限の期間のみ保持
- 内部更新関数はインライン展開可能

## セキュリティ考慮事項

- API Armorによるポインタチェック（SQLITE_ENABLE_API_ARMOR時）
- db接続の有効性チェック（sqlite3SafetyCheckOk）

## 備考

- SQLITE_OMIT_WSDで静的データなし環境をサポート
- sqlite3_status()は sqlite3_status64()のラッパー
- sqlite3_db_status()は sqlite3_db_status64()のラッパー

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ステータス情報を格納する構造体を理解。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | status.c | `src/status.c` | sqlite3StatValueType（22-26行目）：値の型 |
| 1-2 | status.c | `src/status.c` | sqlite3StatType（28-31行目）：ステータス構造体 |
| 1-3 | status.c | `src/status.c` | statMutex配列（37-48行目）：項目別ミューテックス |

**読解のコツ**: nowValue[10]が現在値、mxValue[10]が最大値。statMutex[op]が0ならMallocMutex、1ならPcache1Mutexを使用。

#### Step 2: 内部更新関数を理解する

内部から統計を更新する関数群。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | status.c | `src/status.c` | sqlite3StatusValue（69-76行目）：現在値取得 |
| 2-2 | status.c | `src/status.c` | sqlite3StatusUp（89-99行目）：値を加算 |
| 2-3 | status.c | `src/status.c` | sqlite3StatusDown（100-108行目）：値を減算 |
| 2-4 | status.c | `src/status.c` | sqlite3StatusHighwater（114-129行目）：最大値更新 |

**主要処理フロー**:
- **95行目**: 現在値を加算
- **96-98行目**: 最大値を超えたら更新
- **107行目**: 現在値を減算（最大値は変更しない）
- **126-127行目**: 新しい値が最大値より大きければ更新

#### Step 3: グローバルステータスAPIを理解する

公開APIでステータスを取得。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | status.c | `src/status.c` | sqlite3_status64（134-158行目）：64ビット版 |
| 3-2 | status.c | `src/status.c` | sqlite3_status（159-171行目）：32ビット版ラッパー |

**主要処理フロー**:
- **142-144行目**: 範囲チェック
- **148行目**: ミューテックス選択
- **149-155行目**: 値取得とリセット処理

#### Step 4: DB接続ステータスAPIを理解する

接続ごとのステータス取得。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | status.c | `src/status.c` | sqlite3_db_status64（203-421行目）：メイン関数 |
| 4-2 | status.c | `src/status.c` | countLookasideSlots（176-183行目）：ルックアサイドカウント |
| 4-3 | status.c | `src/status.c` | sqlite3LookasideUsed（188-198行目）：ルックアサイド使用量 |

**主要処理フロー**:
- **216行目**: db->mutex取得
- **217-418行目**: switch(op)で項目別処理
- **264-284行目**: CACHE_USED処理（全Pagerをループ）
- **291-330行目**: SCHEMA_USED処理（スキーマメモリ計算）
- **337-354行目**: STMT_USED処理（VDBE走査）

### プログラム呼び出し階層図

```
sqlite3_status(op, pCurrent, pHighwater, resetFlag)
    └─ sqlite3_status64(op, &iCur, &iHwtr, resetFlag)
           ├─ sqlite3MallocMutex() または sqlite3Pcache1Mutex()
           └─ wsdStat.nowValue[op] / mxValue[op]

sqlite3_db_status(db, op, pCurrent, pHighwtr, resetFlag)
    └─ sqlite3_db_status64(db, op, &C, &H, resetFlag)
           ├─ sqlite3_mutex_enter(db->mutex)
           ├─ [LOOKASIDE_USED]
           │      └─ sqlite3LookasideUsed(db, &H)
           │             └─ countLookasideSlots()
           ├─ [CACHE_USED]
           │      ├─ sqlite3BtreeEnterAll(db)
           │      ├─ sqlite3BtreePager(pBt)
           │      └─ sqlite3PagerMemUsed(pPager)
           ├─ [SCHEMA_USED]
           │      ├─ sqlite3DeleteTrigger()（計算モード）
           │      └─ sqlite3DeleteTable()（計算モード）
           └─ sqlite3_mutex_leave(db->mutex)

[内部からの呼び出し]
sqlite3StatusUp(op, N)
sqlite3StatusDown(op, N)
sqlite3StatusHighwater(op, X)
```

### データフロー図

```
[入力]                    [処理]                           [出力]

ステータス項目op ────▶ sqlite3_status64() ────────────▶ 現在値・最大値
                         │
                         ▼
                  ┌─────────────────┐
                  │ sqlite3StatType │
                  │ ・nowValue[10]  │
                  │ ・mxValue[10]   │
                  └─────────────────┘
                         │
                         ▼
                  ┌─────────────────┐
                  │ statMutex[10]   │
                  │ 0: MallocMutex  │
                  │ 1: Pcache1Mutex │
                  └─────────────────┘

DB接続db, 項目op ────▶ sqlite3_db_status64() ────────▶ 現在値・最大値
                         │
                         ├─ db->lookaside（ルックアサイド統計）
                         ├─ db->aDb[i].pBt（B-tree/Pager統計）
                         └─ db->pVdbe（VDBE統計）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| status.c | `src/status.c` | ソース | ステータス情報機能の実装 |
| sqliteInt.h | `src/sqliteInt.h` | ヘッダ | SQLITE_STATUS_*定数定義 |
| sqlite3.h | `src/sqlite3.h` | ヘッダ | 公開API宣言 |
| pager.c | `src/pager.c` | ソース | sqlite3PagerMemUsed, sqlite3PagerCacheStat |
| btree.c | `src/btree.c` | ソース | sqlite3BtreePager, sqlite3BtreeConnectionCount |
| malloc.c | `src/malloc.c` | ソース | sqlite3MallocMutex() |
| pcache1.c | `src/pcache1.c` | ソース | sqlite3Pcache1Mutex() |
