# 機能設計書 49-各種ユーティリティ関数

## 概要

SQLiteにおける各種ユーティリティ関数の設計書。文字列操作、数値変換、可変長整数エンコーディング、エラー処理など、SQLite全体で使用される基盤関数群を提供する。

### 本機能の処理概要

SQLiteのユーティリティ関数は、データベースエンジン全体で使用される基本的な操作を提供する。

**業務上の目的・背景**：SQLiteは移植性が高く、様々なプラットフォームで動作する。標準ライブラリの機能差を吸収し、一貫した動作を保証するため、独自のユーティリティ関数群を提供している。また、SQLiteの内部データ形式（可変長整数など）を扱う専用関数も含まれる。

**機能の利用シーン**：
- 文字列比較（大文字小文字無視）
- 数値文字列の解析（整数・浮動小数点）
- 可変長整数（varint）のエンコード・デコード
- ビッグエンディアン整数の読み書き
- エラーメッセージの設定
- 安全なオーバーフローチェック付き算術

**主要な処理内容**：
1. 文字列操作：sqlite3StrICmp, sqlite3Dequote
2. 数値変換：sqlite3AtoF, sqlite3Atoi64, sqlite3GetInt32
3. 可変長整数：sqlite3PutVarint, sqlite3GetVarint
4. バイトオーダー：sqlite3Get4byte, sqlite3Put4byte
5. エラー処理：sqlite3Error, sqlite3ErrorMsg
6. オーバーフロー検出：sqlite3AddInt64, sqlite3MulInt64

**関連システム・外部連携**：
- 全SQLiteサブシステム（基盤機能として広く使用）

**権限による制御**：特になし（内部ユーティリティ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | （内部機能） | - | 全SQLite機能で内部的に使用 |

## 機能種別

ユーティリティ / 基盤関数

## 入力仕様

### 主要関数の入力パラメータ

#### sqlite3AtoF

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| z | const char* | Yes | 数値文字列 | - |
| pResult | double* | Yes | 結果出力先 | - |
| length | int | Yes | 文字列長 | - |
| enc | u8 | Yes | エンコーディング | UTF8/UTF16LE/BE |

#### sqlite3PutVarint

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| p | unsigned char* | Yes | 出力バッファ | - |
| v | u64 | Yes | 書き込む値 | - |

### 入力データソース

- SQLite内部からの呼び出し

## 出力仕様

### 出力データ

| 関数 | 戻り値型 | 説明 |
|------|---------|------|
| sqlite3Strlen30 | int | 30ビットに制限された文字列長 |
| sqlite3AtoF | int | 1=整数, 2+=小数, 0以下=エラー |
| sqlite3Atoi64 | int | 0=成功, 1=余分テキスト, 2=オーバーフロー, 3=特殊ケース |
| sqlite3PutVarint | int | 書き込んだバイト数（1〜9） |
| sqlite3GetVarint | u8 | 読み取ったバイト数（1〜9） |

### 出力先

- 呼び出し元の戻り値・出力ポインタ

## 処理フロー

### 可変長整数（Varint）エンコーディング

```
エンコーディング形式:
  A = 0xxxxxxx  7ビットのデータ + フラグビット0
  B = 1xxxxxxx  7ビットのデータ + フラグビット1
  C = xxxxxxxx  8ビットのデータ

   7ビット - A
  14ビット - BA
  21ビット - BBA
  28ビット - BBBA
  35ビット - BBBBA
  42ビット - BBBBBA
  49ビット - BBBBBBA
  56ビット - BBBBBBBA
  64ビット - BBBBBBBBC

sqlite3PutVarint処理:
  1. v<=0x7f: 1バイト出力
  2. v<=0x3fff: 2バイト出力
  3. それ以上: putVarint64で1〜9バイト出力

sqlite3GetVarint処理:
  1. p[0]>=0: 1バイト読み取り
  2. p[1]>=0: 2バイト読み取り
  3. それ以上: 複雑なビット操作で3〜9バイト読み取り
```

### 数値変換処理

```
sqlite3AtoF処理（浮動小数点変換）:
  1. 空白スキップ
  2. 符号解析
  3. 有効数字を仮数部に蓄積
  4. 小数点処理
  5. 指数部（e/E）解析
  6. Dekker乗算で高精度変換
  7. 結果を*pResultに格納

sqlite3Atoi64処理（64ビット整数変換）:
  1. エンコーディング判定
  2. 空白スキップ
  3. 符号解析
  4. 先頭ゼロスキップ
  5. 数字を累積
  6. オーバーフロー判定（compare2pow63）
  7. 結果を*pNumに格納
```

### フローチャート

```mermaid
flowchart TD
    A[sqlite3PutVarint] --> B{v<=0x7f?}
    B -->|Yes| C[1バイト出力]
    B -->|No| D{v<=0x3fff?}
    D -->|Yes| E[2バイト出力]
    D -->|No| F[putVarint64]
    F --> G{v上位8ビット設定?}
    G -->|Yes| H[9バイト出力]
    G -->|No| I[1〜8バイト出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-49-01 | 30ビット制限 | sqlite3Strlen30は30ビットに制限 | 常時 |
| BR-49-02 | 大文字小文字無視 | UpperToLowerテーブルで変換 | 文字列比較時 |
| BR-49-03 | Varint最大9バイト | 64ビット値は最大9バイトでエンコード | Varint操作時 |
| BR-49-04 | Dekker高精度 | 倍精度浮動小数点の精度向上 | sqlite3AtoF |

### 主要関数カテゴリ

| カテゴリ | 関数 | 説明 |
|---------|------|------|
| 文字列操作 | sqlite3Strlen30 | 30ビット制限付き文字列長 |
| 文字列操作 | sqlite3StrICmp | 大文字小文字無視比較 |
| 文字列操作 | sqlite3Dequote | クォート除去 |
| 数値変換 | sqlite3AtoF | 文字列→浮動小数点 |
| 数値変換 | sqlite3Atoi64 | 文字列→64ビット整数 |
| 数値変換 | sqlite3GetInt32 | 文字列→32ビット整数 |
| Varint | sqlite3PutVarint | Varint書き込み |
| Varint | sqlite3GetVarint | Varint読み取り |
| Varint | sqlite3VarintLen | Varint長計算 |
| バイト操作 | sqlite3Get4byte | ビッグエンディアン読み取り |
| バイト操作 | sqlite3Put4byte | ビッグエンディアン書き込み |
| エラー | sqlite3Error | エラーコード設定 |
| エラー | sqlite3ErrorMsg | パーサーエラーメッセージ |
| 算術 | sqlite3AddInt64 | オーバーフロー検出付き加算 |
| 算術 | sqlite3MulInt64 | オーバーフロー検出付き乗算 |

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベーステーブルを直接操作しない（ユーティリティ関数）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| SQLITE_MISUSE | 不正使用 | 無効なdbポインタ | logBadConnection |
| SQLITE_NOMEM | メモリ不足 | エラーメッセージ生成失敗 | mallocFailedフラグ設定 |

### リトライ仕様

ユーティリティ関数はリトライ不要（同期処理）

## トランザクション仕様

ユーティリティ関数はトランザクションの対象外

## パフォーマンス要件

- Varintの1〜2バイトケースは高速パス
- sqlite3Get4byteはバイトオーダーに応じて最適化
- GCC組み込み関数（__builtin_add_overflow等）を使用可能な場合は利用

## セキュリティ考慮事項

- sqlite3SafetyCheckOk/SickOrOkでdbポインタ検証
- オーバーフロー検出付き算術演算

## 備考

- SQLITE_OMITマクロで一部機能を無効化可能
- テスト用にsqlite3FaultSim関数を提供

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 文字列操作を理解する

基本的な文字列操作関数。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | util.c | `src/util.c` | sqlite3Strlen30（92-95行目）：30ビット制限 |
| 1-2 | util.c | `src/util.c` | sqlite3StrICmp（416-434行目）：大文字小文字無視比較 |
| 1-3 | util.c | `src/util.c` | sqlite3Dequote（298-319行目）：クォート除去 |

**読解のコツ**: UpperToLowerテーブル（sqliteInt.h）で大文字→小文字変換。0x3fffffffマスクで30ビットに制限。

#### Step 2: 数値変換を理解する

文字列から数値への変換。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | util.c | `src/util.c` | sqlite3GetInt32（935-988行目）：32ビット整数変換 |
| 2-2 | util.c | `src/util.c` | sqlite3Atoi64（798-887行目）：64ビット整数変換 |
| 2-3 | util.c | `src/util.c` | sqlite3AtoF（529-716行目）：浮動小数点変換 |
| 2-4 | util.c | `src/util.c` | dekkerMul2（467-495行目）：Dekker乗算 |

**主要処理フロー**:
- **798-819行目**: UTF8/UTF16エンコーディング処理
- **831-833行目**: 数字の累積
- **869行目**: compare2pow63で2^63との比較

#### Step 3: 可変長整数を理解する

Varintエンコーディング。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | util.c | `src/util.c` | エンコーディング仕様コメント（1158-1175行目） |
| 3-2 | util.c | `src/util.c` | sqlite3PutVarint（1211-1222行目）：書き込み |
| 3-3 | util.c | `src/util.c` | putVarint64（1187-1210行目）：64ビット書き込み |
| 3-4 | util.c | `src/util.c` | sqlite3GetVarint（1241-1389行目）：読み取り |
| 3-5 | util.c | `src/util.c` | sqlite3VarintLen（1435-1439行目）：長さ計算 |

**主要処理フロー**:
- **1212-1214行目**: 1バイトケース（v<=0x7f）
- **1216-1219行目**: 2バイトケース（v<=0x3fff）
- **1244-1247行目**: 読み取り1バイトケース
- **1248-1251行目**: 読み取り2バイトケース

#### Step 4: バイトオーダー操作を理解する

ビッグエンディアン整数の読み書き。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | util.c | `src/util.c` | sqlite3Get4byte（1445-1462行目）：4バイト読み取り |
| 4-2 | util.c | `src/util.c` | sqlite3Put4byte（1463-1478行目）：4バイト書き込み |

**読解のコツ**: SQLITE_BYTEORDERとGCC_VERSION/MSVC_VERSIONで最適な実装を選択。

#### Step 5: エラー処理を理解する

エラーメッセージの設定。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | util.c | `src/util.c` | sqlite3Error（130-138行目）：エラーコード設定 |
| 5-2 | util.c | `src/util.c` | sqlite3ErrorMsg（242-266行目）：パーサーエラー |
| 5-3 | util.c | `src/util.c` | sqlite3SafetyCheckOk（1547-1563行目）：db検証 |

### プログラム呼び出し階層図

```
[文字列操作]
sqlite3_stricmp(zLeft, zRight)
    └─ sqlite3StrICmp(zLeft, zRight)
           └─ UpperToLower[]テーブル参照

[数値変換]
sqlite3AtoF(z, pResult, length, enc)
    ├─ 符号・仮数・指数解析
    └─ dekkerMul2(rr, 10^n, 誤差項)

sqlite3Atoi64(z, pNum, length, enc)
    └─ compare2pow63(zNum, incr)

sqlite3GetInt32(zNum, pValue)
    └─ sqlite3HexToInt(h) [16進数時]

[Varint]
sqlite3PutVarint(p, v)
    └─ putVarint64(p, v) [v>0x3fff時]

sqlite3GetVarint(p, v)
    └─ [複雑なビット操作]

[エラー処理]
sqlite3ErrorMsg(pParse, zFormat, ...)
    ├─ sqlite3VMPrintf(db, zFormat, ap)
    └─ sqlite3DbFree(db, zMsg)
```

### データフロー図

```
[入力]                    [処理]                           [出力]

文字列z ─────────────▶ sqlite3Atoi64() ──────────────▶ 64ビット整数
                         │
                         ├─ エンコーディング判定
                         ├─ 空白スキップ
                         ├─ 符号解析
                         └─ 数字累積

64ビット整数v ──────▶ sqlite3PutVarint() ──────────▶ 1〜9バイト
                         │
                         ├─ v<=0x7f: 1バイト
                         ├─ v<=0x3fff: 2バイト
                         └─ putVarint64: 1〜9バイト

バイト列p ──────────▶ sqlite3GetVarint() ──────────▶ 64ビット整数
                         │
                         ├─ 1バイト高速パス
                         ├─ 2バイト高速パス
                         └─ 3〜9バイト処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| util.c | `src/util.c` | ソース | ユーティリティ関数実装 |
| sqliteInt.h | `src/sqliteInt.h` | ヘッダ | UpperToLowerテーブル、マクロ定義 |
| global.c | `src/global.c` | ソース | sqlite3UpperToLower配列定義 |
