# 画面設計書 4-基本デモ（Worker版）

## 概要

本ドキュメントは、SQLite WASMをWorkerスレッドで動作させる基本デモ（demo-123-worker.html）の画面設計書です。

### 本画面の処理概要

この画面は、SQLite3 WASMライブラリをWorkerスレッドで実行する基本的なデモンストレーションを提供します。

**業務上の目的・背景**：UIスレッド版（demo-123.html）と同じデモロジックをWorkerスレッドで実行することで、UIをブロックしないデータベース操作の実装パターンを示します。大規模なデータベース操作を行う場合や、UIの応答性を維持したい場合に推奨されるアーキテクチャです。同じdemo-123.jsを使用することで、UIスレッドとWorkerスレッドの両方で動作するコードの書き方を学習できます。

**画面へのアクセス方法**：HTTPサーバー経由で `demo-123-worker.html` にアクセスします。demo-123.jsをWorkerとして実行し、sqlite3.dirパラメータでWASMファイルの場所を指定します。

**主要な操作・処理内容**：
1. Workerの作成 - `new Worker("demo-123.js?sqlite3.dir=jswasm")` でWorkerスレッド起動
2. メッセージ受信 - Worker からの `log` タイプメッセージを受信してUIに表示
3. SQLite3操作 - Worker内でdemo-123.jsの全デモロジック（DB作成、INSERT、SELECT、UDF、トランザクション）を実行

**画面遷移**：
- 遷移先画面：なし
- 遷移元画面：index.html、index-dist.html

**権限による表示制御**：なし

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | SQLiteライブラリ初期化 | 主機能 | Worker内でのsqlite3InitModule()初期化 |
| 2 | データベース接続管理 | 主機能 | Worker内でのDB接続管理 |
| 6 | SELECT文処理 | 主機能 | Worker内でのSQL実行処理 |
| 7 | INSERT文処理 | 主機能 | Worker内でのデータ挿入処理 |

## 画面種別

デモ（実行結果表示）

## URL/ルーティング

`ext/wasm/demo-123-worker.html`

## 入出力項目

| 項目名 | 入出力 | 型 | 説明 |
|--------|--------|-----|------|
| なし | - | - | 本画面はユーザー入力を受け付けない自動実行デモ |

## 表示項目

| 項目名 | 表示位置 | 説明 |
|--------|----------|------|
| タイトル | h1 | "1-2-sqlite3 worker demo" |
| ログ出力 | body | Workerからのメッセージをdiv要素として追加表示 |
| SQLiteバージョン | ログ | "sqlite3 version X.X.X" |
| 各操作結果 | ログ | demo-123.jsの実行結果 |

## イベント仕様

### 1-ページロード（Worker起動）

**トリガー**: ページロード完了時

**処理内容**:
1. **行21-42**: 即時関数内でWorkerを作成
2. **行28**: `new Worker("demo-123.js?sqlite3.dir=jswasm")` でWorker起動
3. **行33-40**: `w.onmessage` でWorkerからのメッセージを受信
4. **行35-36**: `log` タイプメッセージの場合、`logHtml()` でUIに表示

**画面遷移**: なし

### 2-Workerメッセージ受信

**トリガー**: Worker内のdemo-123.jsからpostMessage()が呼ばれたとき

**処理内容**:
```javascript
w.onmessage = function({data}){
  switch(data.type){
    case 'log':
      logHtml(data.payload.cssClass, ...data.payload.args);
      break;
    default:
      logHtml('error',"Unhandled message:",data.type);
  };
};
```

**メッセージ形式**:
```javascript
{
  type: 'log',
  payload: {
    cssClass: 'warning' | '' | 'error',
    args: ['メッセージ内容', ...]
  }
}
```

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| テーブル作成 | t | CREATE | Worker内でCREATE TABLE実行 |
| データ挿入 | t | INSERT | Worker内でINSERT実行 |
| データ検索 | t | SELECT | Worker内でSELECT実行 |
| トランザクションテスト | t | DELETE/INSERT | Worker内でトランザクションデモ実行 |

※実際のDB操作はdemo-123.jsのdemo1()関数で実行される（No.3参照）

## メッセージ仕様

| メッセージID | 種別 | メッセージ内容 | 表示条件 |
|-------------|------|---------------|----------|
| LOG-001 | 情報 | "Running demo from Worker thread." | Worker起動時（コンソール） |
| LOG-002 | 情報 | 各種デモ実行結果 | Worker内でdemo1()実行中 |
| ERR-001 | エラー | "Unhandled message: {type}" | 未対応メッセージタイプ受信時 |

## 例外処理

| 例外状況 | 対応処理 |
|---------|----------|
| Worker作成失敗 | ブラウザエラー（JSコンソールに表示） |
| 未対応メッセージタイプ | logHtml('error', ...)でエラー表示 |
| Worker内例外 | demo-123.js内のerror()関数でpostMessage |

## 備考

- demo-123.jsはURLパラメータ `sqlite3.dir` でWASMファイルの場所を指定
- Worker内でimportScripts()を使用してsqlite3.jsをロード
- UIスレッド版と同じdemo-123.jsを使用（スレッド判定で動作が分岐）

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | demo-123-worker.html | `ext/wasm/demo-123-worker.html` | WorkerメッセージのJSONフォーマット |

**読解のコツ**:
- Workerからのメッセージは `{type: 'log', payload: {cssClass, args}}` 形式
- `data.type` でメッセージ種別を判定

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | demo-123-worker.html | `ext/wasm/demo-123-worker.html` | HTML構造とWorker起動処理 |

**主要処理フロー**:
1. **行21**: `<script>` タグ内で即時関数実行
2. **行22-27**: `logHtml()` 関数定義（UIへのログ出力）
3. **行28**: `new Worker("demo-123.js?sqlite3.dir=jswasm")` でWorker作成
4. **行33-40**: `w.onmessage` でメッセージハンドラ設定

#### Step 3: Worker内の処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | demo-123.js | `ext/wasm/demo-123.js` | Worker用コードパス |

**主要処理フロー**:
- **行31**: `globalThis.window !== globalThis` でWorkerスレッドを判定
- **行32-38**: Worker時は `postMessage()` でメインスレッドに通知
- **行253-273**: Worker時は `importScripts()` でsqlite3.jsをロード
- **行268-273**: `sqlite3.dir` URLパラメータでWASMパスを解決

#### Step 4: UIスレッド版との比較

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | demo-123.html | `ext/wasm/demo-123.html` | UIスレッド版との違い |

**主要な違い**:
- UIスレッド版: `<script src="jswasm/sqlite3.js">` で直接読み込み
- Worker版: Worker内で `importScripts()` を使用
- UIスレッド版: DOM直接操作でログ表示
- Worker版: `postMessage()` → `onmessage` でログ表示

### プログラム呼び出し階層図

```
demo-123-worker.html (ページロード)
    │
    ├─ logHtml() 定義
    │
    └─ new Worker("demo-123.js?sqlite3.dir=jswasm")
           │
           └─ demo-123.js (Worker内)
                  │
                  ├─ importScripts(sqlite3.js)
                  │
                  ├─ sqlite3InitModule()
                  │      └─ Promise.then() → demo1()
                  │
                  └─ demo1(sqlite3)
                         │
                         ├─ postMessage({type:'log',...})
                         │        │
                         │        └─ w.onmessage (メインスレッド)
                         │               └─ logHtml() → DOM更新
                         │
                         └─ DB操作（UIスレッド版と同じ）
```

### データフロー図

```
[メインスレッド]                [Workerスレッド]

ページロード
    │
    └── new Worker() ─────────▶ demo-123.js起動
                                    │
                                    ├── importScripts()
                                    │
                                    ├── sqlite3InitModule()
                                    │
                                    └── demo1() 実行
                                           │
    ┌─────────────────────────────────────┘
    │
    ▼
w.onmessage ◀───────────────── postMessage({type:'log',...})
    │
    └── logHtml()
           │
           ▼
        DOM更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| demo-123-worker.html | `ext/wasm/demo-123-worker.html` | テンプレート | Worker版デモページHTML |
| demo-123.js | `ext/wasm/demo-123.js` | ソース | デモロジック（UIスレッド/Worker両対応） |
| sqlite3.js | `ext/wasm/jswasm/sqlite3.js` | ソース | WASM初期化ローダー |
| sqlite3.wasm | `ext/wasm/jswasm/sqlite3.wasm` | バイナリ | SQLite WASMモジュール |
