# 画面設計書 6-Worker1APIデモ

## 概要

本ドキュメントは、Worker-based wrapper of the OO API #1のテストページ（demo-worker1.html）の画面設計書です。

### 本画面の処理概要

この画面は、sqlite3-worker1.jsを使用したWorkerベースのAPIデモンストレーションを提供します。メッセージパッシングによる非同期DB操作のテストを行います。

**業務上の目的・背景**：SQLite WASMをWorkerスレッドで実行する際の低レベルメッセージパッシングAPIのテストページです。Worker1 APIは直接的なメッセージ送受信を行うため、非同期処理の理解と制御が必要です。より使いやすいPromiseベースのラッパー（demo-worker1-promiser.html）が推奨されますが、Worker1 APIを理解することで低レベルの動作原理を学習できます。

**画面へのアクセス方法**：HTTPサーバー経由で `demo-worker1.html` にアクセスします。開発者コンソールで詳細な出力を確認します。

**主要な操作・処理内容**：
1. Worker作成 - sqlite3-worker1.jsをWorkerとしてロード
2. DBオープン - 'open'メッセージでDB接続を確立
3. SQL実行 - 'exec'メッセージでCREATE TABLE、INSERT、SELECT等を実行
4. DBエクスポート - 'export'メッセージでDBをバイト配列として取得
5. DBクローズ - 'close'メッセージでDB接続を終了

**画面遷移**：
- 遷移先画面：なし
- 遷移元画面：index.html、index-dist.html

**権限による表示制御**：なし

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | SQLiteライブラリ初期化 | 主機能 | Worker1 APIでのモジュール初期化 |
| 2 | データベース接続管理 | 主機能 | Worker経由でのDB接続管理 |
| 42 | ミューテックス | 補助機能 | Workerスレッド間の同期処理 |

## 画面種別

テスト（開発者向け）

## URL/ルーティング

`ext/wasm/demo-worker1.html`

## 入出力項目

| 項目名 | 入出力 | 型 | 説明 |
|--------|--------|-----|------|
| なし | - | - | 本画面はユーザー入力を受け付けない自動テスト |

## 表示項目

| 項目名 | 表示位置 | 説明 |
|--------|----------|------|
| タイトルバー | ヘッダー | "sqlite3-worker1.js tests" |
| 説明文 | 中央 | "Most stuff on this page happens in the dev console." |
| テスト出力 | 下部 | テスト結果のログ表示 |

## イベント仕様

### 1-ページロード（Worker起動とテスト実行）

**トリガー**: ページロード完了時

**処理内容**:
1. **行22**: `new Worker("jswasm/sqlite3-worker1.js")` でWorker作成
2. **行303-344**: `SW.onmessage` でメッセージハンドラ設定
3. **行322-327**: 'worker1-ready'イベント受信時に `runTests()` 実行
4. **行256-301**: テストシーケンス開始

### 2-DB オープン

**トリガー**: runTests()関数実行時（行288-299）

**処理内容**:
```javascript
runOneTest('open', {
  filename:'testing2.sqlite3',
  simulateError: simulateOpenError
}, function(ev){
  DbState.id = ev.dbId;
  if(waitForOpen) setTimeout(runTests2, 0);
});
```

**メッセージ形式**:
```javascript
{
  type: 'open',
  args: { filename: 'testing2.sqlite3' },
  messageId: 'open-1'
}
```

### 3-SQL実行（exec）

**トリガー**: runTests2()関数実行時（行151-253）

**処理内容**:
- CREATE TABLE、INSERT、SELECT等の各種SQL実行
- resultRows、columnNames、rowMode等のオプション指定
- callbackによる行ごとの処理

**テスト例**:
```javascript
runOneTest('exec',{
  sql: ["create table t(a,b);",
        "insert into t(a,b) values(1,2),(3,4),(5,6)"],
  lastInsertRowId: true,
  resultRows: [], columnNames: []
}, function(ev){...});
```

### 4-DBエクスポート

**トリガー**: runTests2()関数実行時（行235-242）

**処理内容**:
```javascript
runOneTest('export',{}, function(ev){
  ev = ev.result;
  T.assert('string' === typeof ev.filename)
    .assert(ev.byteArray instanceof Uint8Array)
    .assert(ev.byteArray.length > 1024)
    .assert('application/x-sqlite3' === ev.mimetype);
});
```

### 5-DBクローズ

**トリガー**: runTests2()関数最後（行243-252）

**処理内容**:
```javascript
runOneTest('close',{unlink:true},function(ev){
  ev = ev.result;
  T.assert('string' === typeof ev.filename);
});
```

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| open | - | CONNECT | testing2.sqlite3を開く |
| exec (create) | t | CREATE | CREATE TABLE t(a,b) |
| exec (insert) | t | INSERT | INSERT 3行 |
| exec (select) | t | SELECT | 各種rowModeでのSELECT |
| exec (delete) | t | DELETE | DELETE FROM t WHERE a>3 |
| export | - | READ | DBをバイト配列として取得 |
| close | - | DISCONNECT | DB接続終了、ファイル削除 |

### テーブル別更新項目詳細

#### テーブル t

| 操作 | 項目（カラム名） | 更新値・取得条件 | 備考 |
|-----|-----------------|-----------------|------|
| INSERT | a, b | (1,2), (3,4), (5,6) | 初期テストデータ |
| SELECT | a, b | ORDER BY a | 各種rowModeでテスト |
| DELETE | a | WHERE a > 3 | 条件付き削除 |

## メッセージ仕様

| メッセージID | 種別 | メッセージ内容 | 表示条件 |
|-------------|------|---------------|----------|
| LOG-001 | 情報 | "Init complete, but async init bits may still be running." | ページロード完了時 |
| LOG-002 | 情報 | "Message: {sqlite3-api: worker1-ready}" | Worker準備完了時 |
| LOG-003 | 情報 | "Sending 'open' message..." | テスト開始時 |
| LOG-004 | 警告 | "This is the final test." | 最終テスト実行時 |
| ERR-001 | エラー | "ERROR from the worker: {message}" | Workerエラー時 |

## 例外処理

| 例外状況 | 対応処理 |
|---------|----------|
| Workerエラー | SW.onerrorハンドラでエラーログ出力 |
| 意図的なエラー（intentional_error） | MsgHandlerQueueでエラーを処理し、後続テスト続行 |
| メッセージハンドラ例外 | try-catchでキャッチしてerror()出力 |

## 備考

- 開発者コンソールでの確認が主な用途
- Promise版（demo-worker1-promiser.html）の方が実用的
- messageIdによるリクエスト/レスポンス追跡機構
- MsgHandlerQueueによる非同期コールバック管理

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | demo-worker1.js | `ext/wasm/demo-worker1.js` | メッセージ形式とDbStateオブジェクト |

**読解のコツ**:
- `DbState` オブジェクトでDB IDを管理
- `MsgHandlerQueue` でコールバックをキュー管理
- メッセージには `messageId` を付与してリクエスト/レスポンスを追跡

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | demo-worker1.html | `ext/wasm/demo-worker1.html` | HTML構造とスクリプト読み込み |
| 2-2 | demo-worker1.js | `ext/wasm/demo-worker1.js` | Worker作成とメッセージハンドラ設定 |

**主要処理フロー**:
1. **行22**: `new Worker("jswasm/sqlite3-worker1.js")` でWorker作成
2. **行303-344**: `SW.onmessage` でメッセージルーティング
3. **行322-327**: 'worker1-ready'受信で `runTests()` 実行

#### Step 3: メッセージパッシング機構を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | demo-worker1.js | `ext/wasm/demo-worker1.js` | runOneTest()とMsgHandlerQueue |

**主要処理フロー**:
- **行54-64**: MsgHandlerQueueの実装（push/shiftでコールバック管理）
- **行81-101**: runOneTest()でメッセージ送信とコールバック登録
- **行310-319**: onmessageでコールバック実行

#### Step 4: テストシーケンスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | demo-worker1.js | `ext/wasm/demo-worker1.js` | runTests()とrunTests2() |

**主要処理フロー**:
- **行256-301**: runTests() - DB open
- **行151-254**: runTests2() - exec、export、close

### プログラム呼び出し階層図

```
demo-worker1.html (ページロード)
    │
    ├─ SqliteTestUtil.js (テストユーティリティ)
    │
    └─ demo-worker1.js
           │
           ├─ new Worker("jswasm/sqlite3-worker1.js")
           │
           ├─ SW.onmessage 設定
           │      │
           │      └─ 'worker1-ready' 受信
           │             └─ runTests()
           │                    │
           │                    └─ runOneTest('open',...)
           │                           │
           │                           └─ (callback) runTests2()
           │                                  │
           │                                  ├─ runOneTest('exec',...)
           │                                  │      └─ (callback) アサーション
           │                                  ├─ runOneTest('export',...)
           │                                  └─ runOneTest('close',...)
           │
           └─ MsgHandlerQueue
                  ├─ push() - コールバック登録
                  └─ shift() - コールバック取得・実行
```

### データフロー図

```
[メインスレッド]                              [Workerスレッド]

new Worker() ──────────────────────────────▶ sqlite3-worker1.js起動
                                                    │
                                                    ▼
                ◀─────────────────────────── 'worker1-ready' postMessage
onmessage
    │
    └── runTests()
           │
           └── postMessage({type:'open',...}) ──▶ open処理
                                                    │
                ◀─────────────────────────── {result, dbId} postMessage
onmessage
    │
    └── runTests2()
           │
           └── postMessage({type:'exec',...}) ──▶ exec処理
                                                    │
                ◀─────────────────────────── {result} postMessage
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| demo-worker1.html | `ext/wasm/demo-worker1.html` | テンプレート | テストページHTML |
| demo-worker1.js | `ext/wasm/demo-worker1.js` | ソース | テストロジック |
| sqlite3-worker1.js | `ext/wasm/jswasm/sqlite3-worker1.js` | ソース | Worker側API実装 |
| SqliteTestUtil.js | `ext/wasm/common/SqliteTestUtil.js` | ソース | テストユーティリティ |
