# 画面設計書 9-Speedtest1（メインスレッド版）

## 概要

本ドキュメントは、SQLite3のベンチマークツールspeedtest1のメインスレッド実行版（speedtest1.html）の画面設計書です。

### 本画面の処理概要

この画面は、SQLite3の主要ベンチマークツールであるspeedtest1をWASM版でメインスレッドで実行します。

**業務上の目的・背景**：SQLite3の性能特性を測定・評価するための標準ベンチマークツールです。様々なSQL操作（INSERT、SELECT、UPDATE、DELETE、トランザクション等）の実行速度を測定し、異なる設定やVFSでの性能比較に使用します。メインスレッド版はUIをブロックしますが、シンプルな実行モデルを提供します。

**画面へのアクセス方法**：HTTPサーバー経由で `speedtest1.html` にアクセスします。ページロード時に自動的にベンチマークが開始されます。

**主要な操作・処理内容**：
1. WASM初期化 - speedtest1.js（WASMモジュール）の初期化
2. ベンチマーク自動実行 - ページロード時に即座にspeedtest1を実行
3. 結果表示 - ベンチマーク完了後に結果を表示
4. VFS選択 - URLパラメータでVFS（kvvfs等）を指定可能
5. オプション設定 - URLパラメータでflags、sizeを指定可能

**画面遷移**：
- 遷移先画面：speedtest1-worker.html（Workerスレッド版へのリンク）
- 遷移元画面：index.html

**権限による表示制御**：なし

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | SQLiteライブラリ初期化 | 主機能 | speedtest1 WASM版の初期化 |
| 17 | VDBE（仮想データベースエンジン） | 主機能 | ベンチマーク対象のVDBE実行 |
| 20 | B-Treeストレージエンジン | 主機能 | ベンチマーク対象のB-Tree操作 |
| 21 | ページャー | 主機能 | ベンチマーク対象のページアクセス |

## 画面種別

ベンチマーク（自動実行）

## URL/ルーティング

`ext/wasm/speedtest1.html`

**URLパラメータ**:
- `vfs`: 使用するVFS名（例: kvvfs, opfs）
- `flags`: カンマ区切りのフラグ（例: --memdb,--nosync）
- `size`: テストサイズ（デフォルト: 100）

## 入出力項目

| 項目名 | 入出力 | 型 | 説明 |
|--------|--------|-----|------|
| URLパラメータ（vfs） | 入力 | 文字列 | 使用するVFS名 |
| URLパラメータ（flags） | 入力 | 文字列 | カンマ区切りのフラグ |
| URLパラメータ（size） | 入力 | 数値 | テストサイズ |

## 表示項目

| 項目名 | 表示位置 | 説明 |
|--------|----------|------|
| タイトルバー | ヘッダー | "speedtest1.wasm" |
| Worker版リンク | ヘッダー下 | "See also: A Worker-thread variant of this page." |
| 初期化スピナー | 中央 | WASM初期化中に表示 |
| 警告メッセージ | 中央 | UIブロックに関する警告 |
| Dev tools警告 | 中央 | 開発者ツールによる速度低下警告 |
| 出力説明 | 中央 | 出力がバッファリングされる旨の説明 |
| テスト出力 | 下部 | ベンチマーク結果 |

## イベント仕様

### 1-ページロード（自動実行）

**トリガー**: ページロード完了時

**処理内容**:
1. **行38**: SqliteTestUtil.js読み込み
2. **行39**: jswasm/speedtest1.js（WASMモジュール）読み込み
3. **行175-178**: sqlite3InitModule()でWASM初期化
4. **行93-173**: runTests()でベンチマーク実行

**警告表示（行28-34）**:
```html
<div class='warning'>This page starts running the main exe when it loads,
which will block the UI until it finishes!</div>
<div class='warning'>Achtung: running it with the dev tools open may
<em>drastically</em> slow it down.</div>
```

### 2-ベンチマーク実行

**トリガー**: sqlite3InitModule()完了後

**処理内容**:
```javascript
const runTests = function(sqlite3){
  const capi = sqlite3.capi, wasm = sqlite3.wasm;
  const argv = ["speedtest1"];

  // URLパラメータからフラグを取得
  if(urlParams.has('flags')){
    argv.push(...(urlParams.get('flags').split(',')));
  }

  // VFS指定
  if(urlParams.has('vfs')){
    vfs = urlParams.get('vfs');
    argv.push("--vfs", vfs);
  }

  // kvvfs使用時の特別処理
  if('kvvfs' === vfs){
    forceSize = 5;
    dbFile = 'transient';
  }

  argv.push("--singlethread", "--nomemstat", "--big-transactions", dbFile);

  // ベンチマーク実行
  wasm.xCall('wasm_main', argv.length, wasm.scopedAllocMainArgv(argv));
};
```

**行番号**: 93-173

### 3-結果出力

**トリガー**: ベンチマーク完了後

**処理内容**:
```javascript
setTimeout(function(){
  wasm.xCall('wasm_main', argv.length, wasm.scopedAllocMainArgv(argv));
  if('kvvfs'===vfs){
    logList.unshift("KVVFS "+dbFile+" size = "+
                    capi.sqlite3_js_kvvfs_size(dbFile));
  }
  logList.unshift("Done running native main(). Output:");
  dumpLogList();
  log2('',"WASM heap size:",sqlite3.wasm.heap8().byteLength,"bytes");
}, 50);
```

**行番号**: 160-172

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| ベンチマーク実行 | speedtest1.db | 全種 | ベンチマーク用のCREATE/INSERT/SELECT/UPDATE/DELETE |

## メッセージ仕様

| メッセージID | 種別 | メッセージ内容 | 表示条件 |
|-------------|------|---------------|----------|
| WARN-001 | 警告 | "This page starts running the main exe when it loads..." | 常時表示 |
| WARN-002 | 警告 | "Achtung: running it with the dev tools open may drastically slow it down." | 常時表示 |
| LOG-001 | 情報 | "Using VFS: {vfs}" | VFS指定時 |
| LOG-002 | 警告 | "kvvfs VFS: forcing --size {n} and filename 'transient'." | kvvfs使用時 |
| LOG-003 | 情報 | "WASM heap size: {n} bytes" | 実行前後 |
| LOG-004 | 情報 | "Starting native app: {argv}" | 実行開始時 |
| LOG-005 | 情報 | "Done running native main(). Output:" | 実行完了時 |
| ERR-001 | エラー | "Unknown VFS: {vfs}" | 不明なVFS指定時 |

## 例外処理

| 例外状況 | 対応処理 |
|---------|----------|
| 不明なVFS指定 | log2('error', ...)でエラー表示、処理中断 |
| WASMロード失敗 | コンソールにエラー出力 |

## 備考

- メインスレッドで実行されるためUIがブロックされる
- 開発者ツールを開いていると大幅に速度低下
- 出力は実行完了後に一括表示（バッファリング）
- kvvfsはサイズ制限があるため--size 5を強制
- 32-bit版と64-bit版（speedtest1-64bit.html）が存在

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | speedtest1.html | `ext/wasm/speedtest1.html` | URLパラメータとargv配列の構築 |

**読解のコツ**:
- `urlParams` でURLパラメータを取得
- `argv` 配列でspeedtest1のコマンドライン引数を構築
- `wasm.xCall('wasm_main', ...)` でC言語のmain()を呼び出し

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | speedtest1.html | `ext/wasm/speedtest1.html` | インラインスクリプトの初期化処理 |

**主要処理フロー**:
1. **行40-41**: 即時関数開始
2. **行46-66**: wasmfsDir()でOPFSディレクトリ初期化
3. **行69-87**: ログ関数定義（logList配列にバッファリング）
4. **行93-173**: runTests()関数定義
5. **行175-178**: sqlite3InitModule()実行、完了後runTests()

#### Step 3: ベンチマーク実行を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | speedtest1.html | `ext/wasm/speedtest1.html` | runTests()内のargv構築とwasm_main()呼び出し |

**主要処理フロー**:
- **行103-107**: URLパラメータからフラグを取得
- **行111-129**: VFS指定処理、kvvfs特別処理
- **行141-149**: デフォルトフラグ追加
- **行160-172**: setTimeout内でwasm_main()実行、結果出力

#### Step 4: Worker版との比較

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | speedtest1-worker.html | `ext/wasm/speedtest1-worker.html` | Worker版との違い |

**主要な違い**:
- メインスレッド版: 即座に実行開始、UIブロック
- Worker版: フラグ選択UI、対話的実行可能

### プログラム呼び出し階層図

```
speedtest1.html (ページロード)
    │
    ├─ SqliteTestUtil.js
    ├─ jswasm/speedtest1.js (WASM)
    │
    └─ <script> 即時関数
           │
           ├─ wasmfsDir() 定義
           ├─ log/logErr 定義
           ├─ runTests() 定義
           │
           └─ sqlite3InitModule()
                  │
                  └─ runTests(sqlite3)
                         │
                         ├─ URLパラメータ解析
                         ├─ argv構築
                         ├─ VFS設定
                         │
                         └─ setTimeout
                                │
                                └─ wasm.xCall('wasm_main', ...)
                                       │
                                       └─ dumpLogList() → DOM更新
```

### データフロー図

```
[入力]               [処理]                    [出力]

URLパラメータ ───▶ argv配列構築 ───▶ wasm_main()呼び出し
    │                                       │
    │                                       ▼
    │                              speedtest1実行
    │                                       │
    │                                       ▼
    │                              logList配列
    │                                       │
    └───────────────────────────────────────┴───▶ DOM更新（出力エリア）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| speedtest1.html | `ext/wasm/speedtest1.html` | テンプレート | メインスレッド版ベンチマークページ |
| speedtest1.js | `ext/wasm/jswasm/speedtest1.js` | ソース | WASM初期化ローダー |
| speedtest1.wasm | `ext/wasm/jswasm/speedtest1.wasm` | バイナリ | speedtest1 WASMモジュール |
| SqliteTestUtil.js | `ext/wasm/common/SqliteTestUtil.js` | ソース | テストユーティリティ |
| speedtest1-worker.html | `ext/wasm/speedtest1-worker.html` | テンプレート | Worker版（比較参照用） |
