# StaxRip コードリーディングガイドライン

## はじめに

このガイドラインは、StaxRipのコードベースを効率的に理解するための手引きです。
Visual Basic .NET（VB.NET）に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、VB.NETの基本構文と概念を解説します。

### 1.1 プログラム構造

StaxRipはVB.NET（Visual Basic .NET）で記述されたWindows Forms アプリケーションです。.NET Framework 4.8をターゲットとしており、64ビットプラットフォーム向けにビルドされています。

**基本的なファイル構造:**

```vb
' Source/General/GlobalClass.vb:21-35
Public Class GlobalClass
    Property DefaultCommands As New GlobalCommands
    Property DPI As Integer
    Property IsAdmin As Boolean = New WindowsPrincipal(WindowsIdentity.GetCurrent()).IsInRole(WindowsBuiltInRole.Administrator)
    Property IsJobProcessing As Boolean
    Property MainForm As MainForm
    Shared Property MAX_PATH As Integer = 260
    Property MinimizedWindows As Boolean
    Property ProcForm As ProcessingForm
    Property LastModifiedTemplate As New Project
    Property ProjectPath As String
    Property SavedProject As New Project
    Property StopAfterCurrentJob As Boolean
    Property ActiveForm As Form
End Class
```

VB.NETの特徴:
- `Public Class ClassName` でクラスを定義
- `Property` キーワードでプロパティを宣言
- `As` キーワードで型を指定
- `New` キーワードでインスタンス化

### 1.2 データ型と変数

VB.NETでは、変数宣言は `Dim` または `Property` キーワードを使用します。

**変数宣言の例:**

```vb
' Source/General/Project.vb:16-50
Public AbortOnFrameMismatch As Boolean = True
Public AddAttachmentsToMuxer As Boolean = True
Public AudioFiles As New List(Of AudioProfile)
Public AudioTracks As New List(Of AudioTrack)
Public SourceWidth As Integer = 1920
Public SourceHeight As Integer = 1080
Public TargetSize As Integer = 5000
Public VideoBitrate As Integer = 5000
```

**主な型:**
| VB.NET型 | 説明 | 例 |
|---------|------|-----|
| `Integer` | 32ビット整数 | `1920` |
| `Boolean` | 真偽値 | `True`, `False` |
| `String` | 文字列 | `""` |
| `Double` | 倍精度浮動小数点 | `5.0` |
| `Single` | 単精度浮動小数点 | `95.0F` |
| `List(Of T)` | ジェネリックリスト | `New List(Of String)` |

### 1.3 制御構造

**条件分岐:**

```vb
' Source/General/Audio.vb:46-60
If ((cutting OrElse Not ap.IsInputSupported) AndAlso Not directMux) OrElse
    trackIsSupportedButNotContainer Then

    Select Case ap.File.ExtFull
        Case ".mkv", ".webm"
            mkvDemuxer.Demux(ap.File, {ap.Stream}, Nothing, ap, p, False, False, True, "Demux MKV Audio", True)
        Case ".mp4"
            MP4BoxDemuxer.DemuxAudio(ap.File, ap.Stream, ap, p, True)
        Case Else
            ffmpegDemuxer.DemuxAudio(ap.File, ap.Stream, ap, p, True)
    End Select
End If
```

**ループ:**

```vb
' Source/Video/VideoScript.vb:62-68
For Each filter As VideoFilter In Filters
    If filter.Active Then
        If skipCategory Is Nothing OrElse filter.Category <> skipCategory Then
            sb.AppendLine(filter.Script)
        End If
    End If
Next
```

### 1.4 関数/メソッド定義

VB.NETでは `Sub`（戻り値なし）と `Function`（戻り値あり）を区別します。

**Subの例:**

```vb
' Source/General/Audio.vb:8-11
Shared Sub Process(ap As AudioProfile)
    If Not File.Exists(ap.File) OrElse TypeOf ap Is NullAudioProfile Then
        Exit Sub
    End If
End Sub
```

**Functionの例:**

```vb
' Source/General/Audio.vb:71-98
Shared Function GetBaseNameForStream(filePath As String, stream As AudioStream) As String
    Dim base = If(p.TempDir?.DirExists(), filePath.Base, "")
    Dim ret = base + " ID" & (stream.Index + 1)
    Return ret.Trim
End Function
```

### 1.5 モジュール/インポート

VB.NETでは `Imports` 文で名前空間をインポートします。

```vb
' Source/Forms/MainForm.vb:1-16
Imports System.ComponentModel
Imports System.Drawing.Design
Imports System.Drawing.Drawing2D
Imports System.Globalization
Imports System.Reflection
Imports System.Runtime.InteropServices
Imports System.Security.Cryptography
Imports System.Text
Imports System.Text.RegularExpressions
Imports System.Threading
Imports System.Threading.Tasks
Imports Microsoft.VisualBasic
Imports StaxRip
Imports StaxRip.UI
Imports VB6 = Microsoft.VisualBasic
```

---

## 2. プロジェクト固有の概念

> このセクションでは、当プロジェクト特有の概念を解説します。

### 2.1 フレームワーク固有の概念

StaxRipはWindows Formsアプリケーションであり、以下の主要コンポーネントを使用しています:

**Windows Forms UI:**
- フォーム（Form）: ウィンドウを表すクラス
- コントロール（Control）: ボタン、テキストボックスなどのUI要素
- イベントハンドラ: ユーザー操作への応答

**シリアライズ:**

プロジェクト設定やプロファイルの永続化に `<Serializable()>` 属性と `SafeSerialization` クラスを使用:

```vb
' Source/General/General.vb:294-327
Public Class SafeSerialization
    Shared Sub Serialize(o As Object, path As String)
        Dim list As New List(Of Object)
        For Each i In o.GetType.GetFields(BindingFlags.Public Or BindingFlags.NonPublic Or BindingFlags.Instance)
            If Not i.IsNotSerialized Then
                Dim mc As New FieldContainer With {.Name = i.Name}
                Dim value = i.GetValue(o)
                If value IsNot Nothing Then
                    mc.Value = If(IsSimpleType(i.FieldType), value, GetObjectData(value))
                    list.Add(mc)
                End If
            End If
        Next
        ' バイナリフォーマットでファイルに保存
    End Sub
End Class
```

### 2.2 プロジェクト独自のパターン

**グローバル変数パターン:**

プロジェクト全体で使用される重要なオブジェクトは、グローバルにアクセス可能です:

- `g` : GlobalClassのインスタンス（アプリケーション全体の状態管理）
- `p` : Projectのインスタンス（現在のプロジェクト設定）
- `s` : ApplicationSettingsのインスタンス（アプリケーション設定）

```vb
' 使用例: Source/Encoding/VideoEncoder.vb:96-99
Overridable ReadOnly Property OutputPath As String
    Get
        If TypeOf Muxer Is NullMuxer Then
            Return p.TargetFile
        Else
            Return Path.Combine(p.TempDir, p.TargetFile.Base + "_out." + OutputExt)
        End If
    End Get
End Property
```

**プロファイルパターン:**

設定可能なコンポーネント（エンコーダー、マクサー、デマクサーなど）は `Profile` 基底クラスを継承:

```vb
' Source/Encoding/VideoEncoder.vb:8-11
<Serializable()>
Public MustInherit Class VideoEncoder
    Inherits Profile
    Implements IComparable(Of VideoEncoder)
```

**拡張メソッドパターン:**

文字列やパス操作を簡潔にするため、拡張メソッドを多用:

```vb
' Source/General/Extensions.vb:190-193
<Extension()>
Function FileName(instance As String) As String
    If instance = "" Then Return ""
    Return Path.GetFileName(instance.TrimQuotes())
End Function
```

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `{機能名}Form.vb` | Windowsフォーム | `MainForm.vb`, `CropForm.vb` |
| `{機能名}Form.Designer.vb` | フォームのデザイナーコード | `MainForm.Designer.vb` |
| `{機能名}Enc.vb` | ビデオエンコーダー実装 | `x265Enc.vb`, `NVEnc.vb` |
| `{機能名}Control.vb` | UIコントロール | `x265Control.vb`, `CommandLineControl.vb` |
| `{カテゴリ}.vb` | 機能カテゴリ | `Audio.vb`, `Demux.vb`, `Muxer.vb` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `Enc` | エンコーダークラス | `x265Enc`, `NVEnc`, `QSVEnc` |
| `Form` | フォームクラス | `MainForm`, `PreviewForm` |
| `Demuxer` | デマクサークラス | `mkvDemuxer`, `ffmpegDemuxer` |
| `Muxer` | マクサークラス | `MkvMuxer`, `MP4Muxer` |
| `Profile` | 設定プロファイル | `AudioProfile`, `VideoEncoder` |
| `Control` | UIコントロール | `SimpleUI`, `ListViewEx` |
| `Value` | プライベートバッキングフィールド | `NameValue`, `CropRightValue` |
| `tb` | TextBox/TrackBar | `tbSourceFile`, `tbTargetFile` |
| `lb/la` | Label | `laBitrate`, `lPixel` |
| `bn` | Button | `bnNext` |
| `lgb/gb` | GroupBox | `lgbFilters`, `gbAssistant` |
| `tlp` | TableLayoutPanel | `tlpMain`, `tlpSource` |
| `bl` | ButtonLabel | `blFilesize`, `blCropText` |

### 3.3 プログラム分類一覧

| 分類 | 説明 | 主なファイル |
|-----|------|------------|
| エンコーダー | 動画エンコード機能 | `x265Enc.vb`, `x264Enc.vb`, `NVEnc.vb` |
| デマクサー | コンテナからのストリーム抽出 | `Demux.vb` |
| マクサー | ストリームのコンテナへの格納 | `Muxer.vb` |
| フォーム | UIウィンドウ | `MainForm.vb`, `PreviewForm.vb` |
| コントロール | UI部品 | `Controls.vb`, `SimpleUI.vb` |
| スクリプト | AviSynth/VapourSynth制御 | `VideoScript.vb` |
| ユーティリティ | 共通機能 | `Extensions.vb`, `General.vb` |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
staxrip-master/
├── Source/                      # メインソースコード
│   ├── StaxRip.vbproj          # プロジェクトファイル
│   ├── StaxRip.sln             # ソリューションファイル
│   ├── Controls/               # カスタムUIコントロール
│   │   ├── x265Control.vb      # x265エンコーダー設定UI
│   │   ├── x264Control.vb      # x264エンコーダー設定UI
│   │   ├── CommandLineControl.vb # コマンドライン編集UI
│   │   └── ...
│   ├── Encoding/               # ビデオエンコーダー実装
│   │   ├── VideoEncoder.vb     # エンコーダー基底クラス
│   │   ├── x265Enc.vb          # x265エンコーダー
│   │   ├── x264Enc.vb          # x264エンコーダー
│   │   ├── NVEnc.vb            # NVIDIA NVEncエンコーダー
│   │   ├── QSVEnc.vb           # Intel QSVエンコーダー
│   │   ├── VCEEnc.vb           # AMD VCEエンコーダー
│   │   ├── SvtAv1Enc.vb        # SVT-AV1エンコーダー
│   │   └── ...
│   ├── Forms/                  # Windowsフォーム
│   │   ├── MainForm.vb         # メインウィンドウ
│   │   ├── PreviewForm.vb      # プレビューウィンドウ
│   │   ├── CropForm.vb         # クロップ設定ウィンドウ
│   │   ├── ProcessingForm.vb   # エンコード進捗ウィンドウ
│   │   └── ...
│   ├── General/                # コアビジネスロジック
│   │   ├── GlobalClass.vb      # グローバル状態管理
│   │   ├── Project.vb          # プロジェクト設定
│   │   ├── Audio.vb            # オーディオ処理
│   │   ├── Demux.vb            # デマクサー
│   │   ├── Muxer.vb            # マクサー
│   │   ├── Macro.vb            # マクロ展開
│   │   ├── Extensions.vb       # 拡張メソッド
│   │   └── ...
│   ├── UI/                     # UI共通コンポーネント
│   │   ├── Theme.vb            # テーマ定義
│   │   ├── ThemeManager.vb     # テーマ管理
│   │   ├── Controls/           # 汎用UIコントロール
│   │   └── ...
│   ├── Video/                  # ビデオ処理
│   │   ├── VideoScript.vb      # スクリプト管理
│   │   ├── VideoRenderer.vb    # ビデオレンダリング
│   │   └── FrameServer.vb      # フレームサーバー
│   ├── FrameServer/            # ネイティブフレームサーバー(C++)
│   └── My Project/             # VB.NETプロジェクト設定
├── Docs/                       # ドキュメント
├── Images/                     # 画像リソース
└── README.md                   # プロジェクト説明
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `Source/Controls/` | エンコーダー固有のUI設定パネル | `x265Control.vb`, `SvtAv1EncAppControl.vb` |
| `Source/Encoding/` | ビデオエンコーダーの実装 | `VideoEncoder.vb`（基底）, `x265Enc.vb` |
| `Source/Forms/` | アプリケーションのウィンドウ | `MainForm.vb`, `ProcessingForm.vb` |
| `Source/General/` | ビジネスロジックとユーティリティ | `Project.vb`, `Audio.vb`, `Demux.vb` |
| `Source/UI/` | 共通UIコンポーネントとテーマ | `Theme.vb`, `Controls/SimpleUI.vb` |
| `Source/Video/` | ビデオ処理とスクリプト | `VideoScript.vb`, `FrameServer.vb` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

StaxRipは**モジュラーモノリス**アーキテクチャを採用しています。単一のWindows Formsアプリケーションですが、機能ごとにモジュール化されています。

```
┌─────────────────────────────────────────────────────────────────┐
│                        StaxRip Application                       │
├─────────────────────────────────────────────────────────────────┤
│  ┌────────────────┐  ┌────────────────┐  ┌────────────────┐    │
│  │   Forms (UI)   │  │   Controls     │  │    Dialogs     │    │
│  │   MainForm     │  │   x265Control  │  │   TaskDialog   │    │
│  │   CropForm     │  │   SimpleUI     │  │   ProfilesForm │    │
│  └───────┬────────┘  └───────┬────────┘  └───────┬────────┘    │
│          │                   │                   │              │
│          └───────────────────┼───────────────────┘              │
│                              │                                   │
│  ┌───────────────────────────▼───────────────────────────────┐  │
│  │                    Core Services                           │  │
│  │  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐        │  │
│  │  │   Project   │  │   Macro     │  │ GlobalClass │        │  │
│  │  └─────────────┘  └─────────────┘  └─────────────┘        │  │
│  └───────────────────────────┬───────────────────────────────┘  │
│                              │                                   │
│  ┌───────────────────────────▼───────────────────────────────┐  │
│  │                 Processing Pipeline                        │  │
│  │  ┌─────────┐  ┌─────────┐  ┌─────────┐  ┌─────────┐       │  │
│  │  │ Demuxer │─▶│ Script  │─▶│ Encoder │─▶│  Muxer  │       │  │
│  │  └─────────┘  └─────────┘  └─────────┘  └─────────┘       │  │
│  └───────────────────────────────────────────────────────────┘  │
│                              │                                   │
│  ┌───────────────────────────▼───────────────────────────────┐  │
│  │                   External Tools                           │  │
│  │   ffmpeg, x265, x264, mkvmerge, MP4Box, etc.              │  │
│  └───────────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| UI層 | ユーザーインターフェース | `MainForm.vb`, `PreviewForm.vb`, `Theme.vb` |
| アプリケーション層 | ワークフロー制御 | `GlobalClass.vb`, `JobManager.vb` |
| ドメイン層 | ビジネスロジック | `Project.vb`, `VideoEncoder.vb`, `AudioProfile.vb` |
| インフラ層 | 外部ツール連携 | `Proc.vb`, `Package.vb` |

### 5.3 データフロー

```
1. ソースファイル読み込み
   └─▶ MediaInfo解析 ─▶ Project設定更新

2. エンコード準備
   └─▶ Demuxer実行 ─▶ 音声/字幕抽出
       └─▶ VideoScript生成 (AviSynth/VapourSynth)

3. エンコード実行
   └─▶ VideoEncoder.Encode()
       └─▶ 外部ツール(x265等)起動 ─▶ 進捗監視

4. マクシング
   └─▶ Muxer.Mux()
       └─▶ mkvmerge/MP4Box実行 ─▶ 出力ファイル生成
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

アプリケーションのエントリーポイントは `Sub Main` です。`MainForm` が主要なウィンドウとして起動されます。

```vb
' Source/StaxRip.vbproj:26
<StartupObject>Sub Main</StartupObject>
```

MainFormは `FormBase` を継承し、主要なUI要素を管理:

```vb
' Source/Forms/MainForm.vb:18-19
Partial Public Class MainForm
    Inherits FormBase
```

### 6.2 ビジネスロジック

**Projectクラス** - プロジェクト設定の中心:

```vb
' Source/General/Project.vb:8-160
<Serializable()>
Public Class Project
    Implements ISafeSerialization, INotifyPropertyChanged

    ' ソース情報
    Public SourceFile As String
    Public SourceWidth As Integer = 1920
    Public SourceHeight As Integer = 1080
    Public SourceFrameRate As Decimal

    ' ターゲット設定
    Public TargetWidth As Integer = 1920
    Public TargetHeight As Integer = 1080
    Public VideoBitrate As Integer = 5000

    ' クロップ設定
    Public CropLeft, CropRight, CropTop, CropBottom As Integer

    ' オーディオ
    Public AudioFiles As New List(Of AudioProfile)
    Public AudioTracks As New List(Of AudioTrack)

    ' スクリプト
    Public Script As TargetVideoScript

    ' エンコーダー
    Public VideoEncoder As VideoEncoder
End Class
```

**VideoEncoderクラス** - エンコーダーの抽象基底クラス:

```vb
' Source/Encoding/VideoEncoder.vb:8-26
<Serializable()>
Public MustInherit Class VideoEncoder
    Inherits Profile
    Implements IComparable(Of VideoEncoder)

    MustOverride Sub Encode()
    MustOverride ReadOnly Property Codec As String
    MustOverride ReadOnly Property OutputExt As String

    Overridable Property Bitrate As Integer
    Overridable Property Passes As Integer
    Overridable Property QualityMode As Boolean

    Property Muxer As Muxer = New MkvMuxer
End Class
```

### 6.3 データアクセス

StaxRipはデータベースを使用しません。設定はバイナリシリアライズでファイルに保存されます。

**SafeSerializationクラス** - 安全なシリアライズ:

```vb
' Source/General/General.vb:294-370
Public Class SafeSerialization
    Shared Sub Serialize(o As Object, path As String)
        ' オブジェクトをバイナリフォーマットでファイルに保存
    End Sub

    Shared Function Deserialize(Of T)(instance As T, path As String) As T
        ' ファイルからオブジェクトを復元
    End Function
End Class
```

### 6.4 ユーティリティ/共通機能

**拡張メソッド (Extensions.vb):**

```vb
' Source/General/Extensions.vb:190-200
<Extension()>
Function FileName(instance As String) As String
    If instance = "" Then Return ""
    Return Path.GetFileName(instance.TrimQuotes())
End Function

<Extension()>
Function ChangeExt(instance As String, value As String) As String
    If instance = "" Then Return ""
    ' 拡張子を変更
End Function
```

**マクロ展開 (Macro.vb):**

```vb
' Source/General/Macro.vb:124-150
Shared Function GetMacros(...) As List(Of Macro)
    Dim ret As New List(Of Macro)
    ret.Add(New Macro("source_file", "Source File", GetType(String), "Path of the source file."))
    ret.Add(New Macro("target_file", "Target File", GetType(String), "Path of the target file."))
    ret.Add(New Macro("video_bitrate", "Video Bitrate", GetType(Integer), "Video bitrate."))
    ' ... 他のマクロ
    Return ret
End Function
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| 継承による拡張 | 基底クラスを継承して機能を追加 | 高 | `VideoEncoder.vb`, `Muxer.vb` |
| プロパティパターン | バッキングフィールドを持つプロパティ | 高 | `Project.vb`, `Macro.vb` |
| 拡張メソッド | 既存型に機能を追加 | 高 | `Extensions.vb` |
| シリアライズ | 設定の永続化 | 中 | `SafeSerialization` |
| イベント駆動 | UIイベントの処理 | 高 | `MainForm.vb` |

### 各パターンの詳細

#### パターン1: 継承による拡張

**目的:** 共通機能を基底クラスで定義し、具体的な実装を派生クラスで行う

**実装例:**
```vb
' ファイル: Source/Encoding/x265Enc.vb:1-30
<Serializable()>
Public Class x265Enc
    Inherits BasicVideoEncoder

    Sub New()
        Name = "x265"
        AutoCompCheckValue = 50
    End Sub

    Public Overrides ReadOnly Property Codec As String
        Get
            Return "hevc"
        End Get
    End Property

    Public Overrides Sub Encode()
        ' x265固有のエンコード処理
        p.Script.Synchronize()
        ' コマンドライン構築と実行
    End Sub
End Class
```

**解説:** `VideoEncoder` → `BasicVideoEncoder` → `x265Enc` という継承階層により、共通のエンコード処理を再利用しつつ、x265固有の設定を追加しています。

#### パターン2: プロパティパターン

**目的:** 値の取得・設定時にバリデーションやイベント発火を行う

**実装例:**
```vb
' ファイル: Source/General/Project.vb:233-242
Private CropRightValue As Integer

Property CropRight As Integer
    Get
        Return CropRightValue
    End Get
    Set(value As Integer)
        CropRightValue = If(value >= 0, value, 0)  ' 負の値を防ぐ
    End Set
End Property
```

**解説:** バッキングフィールド（`CropRightValue`）を使用し、Setterでバリデーション（負の値を0にする）を行っています。

#### パターン3: 外部プロセス実行

**目的:** ffmpegやx265などの外部ツールを実行し、出力を監視する

**実装例:**
```vb
' ファイル: Source/General/Audio.vb:107-116
Using proc As New Proc
    proc.Header = "AVS to " + outPath.Ext.ToUpperInvariant
    proc.SkipStrings = {"frame=", "size="}
    proc.Encoding = Encoding.UTF8
    proc.Package = Package.ffmpeg
    proc.Arguments = args
    proc.Start()
End Using
```

**解説:** `Proc` クラスが外部プロセスの実行を抽象化し、進捗表示やエラーハンドリングを統一的に処理しています。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: ビデオエンコードフロー

**概要:** ソースファイルからエンコードされた動画が出力されるまでの流れ

**処理フロー:**
```
MainForm → VideoEncoder.Encode() → 外部ツール実行 → Muxer.Mux() → 出力ファイル
```

**詳細な追跡:**

1. **ジョブ開始** (`Source/General/JobManager.vb`)
   ```vb
   ' JobManager がジョブキューを管理
   Sub ProcessJobs()
       For Each job In Jobs
           If job.Active Then
               ProcessJob(job)
           End If
       Next
   End Sub
   ```

2. **エンコード実行** (`Source/Encoding/VideoEncoder.vb:11`)
   ```vb
   MustOverride Sub Encode()  ' 各エンコーダーで実装
   ```

3. **x265での実装例** (`Source/Encoding/x265Enc.vb`)
   ```vb
   Public Overrides Sub Encode()
       p.Script.Synchronize()  ' スクリプトを同期

       Using proc As New Proc
           proc.Header = "Video Encoding"
           proc.Package = Package.x265
           proc.Arguments = GetCommandLine(True, True)
           proc.Start()
       End Using
   End Sub
   ```

4. **マクシング** (`Source/General/Muxer.vb:22`)
   ```vb
   MustOverride Sub Mux()  ' 各マクサーで実装
   ```

#### 例2: オーディオ処理フロー

**概要:** ソースファイルから音声を抽出・変換する流れ

**処理フロー:**
```
Audio.Process() → Demux → Convert → Cut → Encode
```

**詳細な追跡:**

1. **オーディオ処理開始** (`Source/General/Audio.vb:8-69`)
   ```vb
   Shared Sub Process(ap As AudioProfile)
       If Not File.Exists(ap.File) Then Exit Sub

       ' デマックスが必要か判断
       If ap.HasStream Then
           Select Case ap.File.ExtFull
               Case ".mkv", ".webm"
                   mkvDemuxer.Demux(...)
               Case ".mp4"
                   MP4BoxDemuxer.DemuxAudio(...)
           End Select
       End If

       ' カット処理
       Cut(ap)

       ' 変換処理
       If ap.NeedConvert Then
           Convert(ap, True)
       End If
   End Sub
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要
2. このコードリーディングガイドライン
3. `docs/code-to-docs/アーキテクチャ設計書/` - システム構成

#### 特定機能を理解したい場合
1. `docs/code-to-docs/機能一覧/` - 機能リスト
2. `docs/code-to-docs/機能設計書/` - 詳細設計
3. 該当ソースコード

#### 改修作業を行う場合
1. 対象機能の設計書
2. 関連ソースコード
3. `docs/code-to-docs/単体テストケース一覧/` - テストケース確認

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要 | 最初 |
| 機能一覧 | 全機能のリスト | 機能調査時 |
| 画面設計書 | UI設計 | UI改修時 |
| 機能設計書 | 機能詳細 | 機能理解時 |
| テスト方針書 | テスト方針 | テスト計画時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: グローバル変数 `p`, `g`, `s` は何を表していますか?
A:
- `p` : 現在のProject（プロジェクト設定）
- `g` : GlobalClass（アプリケーション全体の状態）
- `s` : ApplicationSettings（アプリケーション設定）

これらは `Source/General/GlobalClass.vb` で定義されています。

#### Q: `MustOverride` と `Overridable` の違いは?
A:
- `MustOverride`: 派生クラスで必ず実装が必要（C#の `abstract`）
- `Overridable`: 派生クラスでオーバーライド可能（C#の `virtual`）

#### Q: 拡張メソッドの定義場所は?
A: 主に `Source/General/Extensions.vb` に定義されています。`<Extension()>` 属性が付いたメソッドを探してください。

### デバッグのヒント

1. **ブレークポイント設置推奨箇所:**
   - `MainForm.vb` のイベントハンドラ
   - `VideoEncoder.Encode()` メソッド
   - `Proc.Start()` メソッド

2. **ログ確認:**
   - `LogBuilder` クラスがエンコード処理のログを管理
   - `p.Log` でアクセス可能

3. **設定ファイルの場所:**
   - `Folder.Settings` プロパティで取得可能
   - 通常は `%APPDATA%\StaxRip` または起動フォルダ配下

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| AviSynth | フレームサーバー・動画処理スクリプト環境 |
| VapourSynth | Python ベースのフレームサーバー |
| Demux | コンテナから個別ストリームを抽出する処理 |
| Mux | 複数ストリームをコンテナに格納する処理 |
| x265 | HEVCエンコーダー |
| x264 | H.264エンコーダー |
| NVEnc | NVIDIA GPUエンコーダー |
| QSVEnc | Intel Quick Sync Videoエンコーダー |
| VCEEnc | AMD Video Coding Engineエンコーダー |
| MKV | Matroskaコンテナフォーマット |
| MP4 | MPEG-4 Part 14コンテナフォーマット |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `Source/StaxRip.vbproj` | プロジェクト定義 | ビルド設定、参照 |
| `Source/Forms/MainForm.vb` | メインウィンドウ | UI定義、イベント処理 |
| `Source/General/Project.vb` | プロジェクト設定 | エンコード設定全般 |
| `Source/Encoding/VideoEncoder.vb` | エンコーダー基底 | 共通エンコード機能 |
| `Source/General/Macro.vb` | マクロ機能 | パス変数展開 |
| `Source/General/Extensions.vb` | 拡張メソッド | 文字列・パス操作 |

### C. 参考資料

- [VB.NET 公式ドキュメント](https://docs.microsoft.com/ja-jp/dotnet/visual-basic/)
- [Windows Forms ドキュメント](https://docs.microsoft.com/ja-jp/dotnet/desktop/winforms/)
- [StaxRip GitHub Wiki](https://github.com/staxrip/staxrip/wiki)
- [AviSynth Wiki](http://avisynth.nl/index.php/Main_Page)
- [VapourSynth ドキュメント](http://www.vapoursynth.com/doc/)
