# セキュリティ設計書

## 概要

本ドキュメントは、StaxRipプロジェクト（Windows向けビデオエンコーディングGUIアプリケーション）のセキュリティ設計について記述する。StaxRipはデスクトップアプリケーションであり、ネットワーク認証やWeb通信を主目的としないため、従来のWebアプリケーションセキュリティとは異なる観点でのセキュリティ対策が実装されている。

## 認証設計

### 認証方式

StaxRipはローカルデスクトップアプリケーションであり、ユーザー認証機能は実装されていない。

- **ユーザー認証**: なし（ローカルアプリケーション）
- **外部サービス認証**: なし

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | N/A | セッション管理なし（デスクトップアプリ） |
| セッション固定化対策 | N/A | セッション管理なし |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| 標準ユーザー | 通常操作 | 一般的なエンコード操作を実行可能 |
| 管理者 | 管理者操作 | システム設定変更（Long Path有効化等） |

### アクセス制御

**管理者権限チェック機能**:
- `Source/General/GlobalClass.vb` (行24)で`WindowsPrincipal`/`WindowsIdentity`を使用した管理者権限判定を実装
- `IsAdmin`プロパティで現在のユーザーが管理者ロールに属するかを判定

```vb
Property IsAdmin As Boolean = New WindowsPrincipal(WindowsIdentity.GetCurrent()).IsInRole(WindowsBuiltInRole.Administrator)
```

**用途**:
- Long Path有効化時のレジストリ書き込み操作
- システムレベルの設定変更操作

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | WebClient/HttpClientによるダウンロードではHTTPS URLを優先的に使用 |
| HSTS | N/A（ブラウザベースではないため） |

### ネットワーク通信

1. **ファイルダウンロード機能** (`Source/Forms/DownloadForm.vb`):
   - `WebClient`を使用した非同期ダウンロード
   - User-Agentヘッダの設定（ブラウザ偽装）
   - Refererヘッダの設定

2. **更新チェック機能** (`Source/General/StaxRipUpdate.vb`, `Source/General/ToolUpdate.vb`):
   - `HttpClient`を使用したHTTPS通信
   - GitHubリリースAPIへのアクセス

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | HTTPS（SSL/TLS） |
| 保存データ | 暗号化なし（プレーンテキスト/バイナリ） |

### ハッシュ関数の使用

| 用途 | アルゴリズム | 実装箇所 |
| --- | --- | --- |
| メッセージ識別 | MD5 | `Source/General/Extensions.vb` (行896-902) |
| ヒント表示制御 | MD5 | `Source/General/Documentation.vb` (行426) |
| x265エンコーダ設定 | MD5/CRC/Checksum | `Source/Encoding/x265Enc.vb` (行1002-1007) |

**MD5ハッシュ実装**:
```vb
Function MD5Hash(instance As String) As String
    Using m = MD5.Create()
        Dim inputBuffer = Encoding.UTF8.GetBytes(instance)
        Dim hashBuffer = m.ComputeHash(inputBuffer)
        Return BitConverter.ToString(m.ComputeHash(inputBuffer))
    End Using
End Function
```

**注意**: MD5はセキュリティ目的（パスワードハッシュ等）には非推奨だが、本プロジェクトではメッセージの識別用途（重複検出）にのみ使用されている。

### 機密情報管理

- パスワード・認証情報は扱っていない
- APIキー・トークンは使用していない
- 設定情報はローカルに平文で保存

### データ永続化

**BinaryFormatterによるシリアライゼーション** (`Source/General/General.vb`):
- プロジェクトファイル(.srip)の保存/読み込みに使用
- `SafeSerialization`クラスによるラップ実装

| リスク | 対策状況 |
| --- | --- |
| デシリアライゼーション攻撃 | 制限なし（信頼されたローカルファイルのみ想定） |
| .NET 5以降での非推奨 | .NET Framework 4.8使用のため現時点では問題なし |

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | N/A（Webアプリケーションではない） |
| SQLインジェクション | N/A（データベース未使用） |
| CSRF | N/A（Webアプリケーションではない） |
| コマンドインジェクション | 部分的対策（下記参照） |
| パストラバーサル | 部分的対策（下記参照） |

### ファイルパス検証

**FileIOPermissionによるパス検証** (`Source/General/General.vb`, 行289):
```vb
Call New FileIOPermission(FileIOPermissionAccess.PathDiscovery, ret).Demand()
```

**パス検証ユーティリティ** (`Source/General/Extensions.vb`):
- `IsValidFileName`: 無効なファイル名文字のチェック
- `IsValidPath`: 無効なパス文字のチェック
- `LongPathPrefix`: 長いパスの適切な処理

### 外部プロセス実行

StaxRipは多数の外部エンコーダツールを呼び出す設計であり、以下の方式で実行される:

| 実行方式 | 用途 |
| --- | --- |
| Process.Start | 外部エンコーダ（x264, x265, ffmpeg等）の起動 |
| g.Execute | コマンドライン実行のラッパー |
| g.ShellExecute | ファイル/URLのシェル経由オープン |
| PowerShell.Invoke | PowerShellスクリプト実行 |

**PowerShell実行** (`Source/General/PowerShell.vb`):
- ExecutionPolicyを`Unrestricted`に設定してスクリプト実行
- ユーザースクリプトの実行機能あり

```vb
Dim policyCommand = New Runspaces.Command("Set-ExecutionPolicy")
policyCommand.Parameters.Add("ExecutionPolicy", "Unrestricted")
policyCommand.Parameters.Add("Scope", "Process")
```

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| 処理ログ | エンコード処理の詳細、エラー情報 | プロジェクト単位（.log） |
| 難読化ログ | パス情報を難読化したデバッグ用ログ | エクスポート時のみ |

### ログ機能の実装

- `LogBuilder`クラス (`Source/General/LogBuilder.vb`) によるログ管理
- UTF-8エンコーディングでファイル出力
- パス情報の難読化機能（プライバシー保護用）

## レジストリアクセス

| 操作 | レジストリキー | 目的 |
| --- | --- | --- |
| 読み取り | HKLM\SOFTWARE\AviSynth | AviSynthインストールパス取得 |
| 読み取り | HKLM\SOFTWARE\VapourSynth | VapourSynthインストールパス取得 |
| 読み取り | HKLM\HARDWARE\DESCRIPTION\System\CentralProcessor | CPU情報取得 |
| 書き込み | HKCU\Software\StaxRip | 設定保存 |
| 書き込み | HKLM\SYSTEM\CurrentControlSet\Control\FileSystem | Long Path有効化（管理者権限必要） |

## セキュリティ上の考慮事項

### 潜在的リスク

1. **BinaryFormatterの使用**
   - .NET 5以降で非推奨
   - 信頼されていないソースからの.sripファイル読み込みにリスクあり
   - **推奨**: 信頼できるソースからのファイルのみ使用

2. **PowerShell ExecutionPolicy**
   - プロセススコープでUnrestrictedに設定
   - ユーザースクリプトが任意のコードを実行可能
   - **推奨**: スクリプトの内容を確認してから実行

3. **外部プロセス実行**
   - 多数の外部ツールを起動
   - パスインジェクションのリスク（ツールパスの改ざん）
   - **推奨**: ツールの整合性確認

4. **MD5の使用**
   - 暗号学的に脆弱だが、本プロジェクトでは識別用途のみ
   - セキュリティ上の問題なし

### 軽減策の実装状況

| リスク | 実装状況 |
| --- | --- |
| 管理者権限操作 | UAC/権限チェック実装済み |
| ファイルパス検証 | 部分的に実装 |
| ネットワーク通信 | HTTPS使用 |
| データ永続化 | SafeSerializationによるラップ |

## 備考

- StaxRipはローカルデスクトップアプリケーションであり、Webアプリケーション向けのセキュリティ対策（CSRF, XSS等）は適用対象外
- 主なセキュリティ境界はWindowsの権限モデルに依存
- 外部エンコーダツールのセキュリティは各ツールの実装に依存
- 設定ファイル・プロジェクトファイルは信頼されたローカル環境での使用を想定
