# 機能設計書: APNG生成

## 1. 機能概要

### 1.1 機能名
APNG生成（Animation.aPNG）

### 1.2 機能説明
動画ファイルからアニメーションPNG（APNG）画像を生成する機能。ffmpegでAPNGを生成し、オプションでPNGoptを使用した最適化処理を行う。APNGはGIFより高品質（フルカラー・透過対応）なアニメーション画像形式。

### 1.3 関連画面
- メインフォーム（MainForm）- ツールメニューから呼び出し

## 2. 機能詳細

### 2.1 入力仕様

| 項目名 | データ型 | 必須 | 説明 |
|--------|----------|------|------|
| inputFile | String | Yes | 入力動画ファイルパス |
| proj | Project | No | プロジェクトオブジェクト（NullでもOK） |

### 2.2 出力仕様

| 項目名 | データ型 | 説明 |
|--------|----------|------|
| APNGファイル | File | 生成されたアニメーションPNG |

### 2.3 処理フロー

```
開始
  │
  ├─ 入力ファイル存在確認
  │    └─ 存在しない場合 → 終了
  │
  ├─ ffmpegパッケージ検証
  │    └─ 検証失敗 → 終了
  │
  ├─ PNGoptパッケージ検証（最適化有効時）
  │    └─ 検証失敗 → 終了
  │
  ├─ プロジェクト初期化（Nullの場合）
  │
  ├─ パラメータ取得（設定ストレージから）
  │    ├─ フレームレート（PNGFrameRate）
  │    ├─ 開始時間（PNGTime）
  │    ├─ 継続時間（PNGLength）
  │    ├─ スケール（PNGScale）
  │    ├─ 最適化設定（PNGopt）
  │    └─ 出力ディレクトリ設定
  │
  ├─ APNG生成（ffmpeg）
  │    └─ .apng拡張子で一時保存
  │
  ├─ 既存ファイル削除（.png）
  │
  ├─ 最適化処理判定
  │    ├─ 最適化有効の場合
  │    │    ├─ 出力ディレクトリ判定
  │    │    ├─ PNGopt実行
  │    │    └─ 一時APNGファイル削除
  │    └─ 最適化無効の場合
  │         └─ ファイルリネーム（.apng → .png）
  │
終了
```

### 2.4 ビジネスルール

| ルールID | ルール内容 |
|----------|------------|
| BR-001 | 入力ファイルが存在しない場合は処理を中断する |
| BR-002 | ffmpegが正しくインストールされていない場合は処理を中断する |
| BR-003 | 最適化有効時はPNGoptが必要 |
| BR-004 | APNG出力ディレクトリが指定されている場合はそこに出力 |
| BR-005 | 最適化無効時はファイルリネームのみ行う |
| BR-006 | 無限ループ再生（-plays 0）を設定 |

### 2.5 設定パラメータ

| パラメータ名 | デフォルト値 | 説明 |
|--------------|--------------|------|
| PNGFrameRate | 15 | 出力APNGのフレームレート |
| PNGTime | "15.0" | 開始時間（秒） |
| PNGLength | "3.8" | 継続時間（秒） |
| PNGScale | 480 | 出力幅（ピクセル） |
| PNGopt | "-z1" | PNGopt最適化オプション |
| OptSetting | False | 最適化処理有効フラグ |
| PNGOutput | False | カスタム出力ディレクトリ使用フラグ |
| PNGDirectory | DefaultTargetFolder | 出力ディレクトリパス |

## 3. 画面設計

### 3.1 画面レイアウト
本機能は設定ダイアログを介して呼び出される。メインフォームのツールメニューからアクセス可能。

### 3.2 UI要素

| 要素名 | 種類 | 説明 |
|--------|------|------|
| フレームレート | 数値入力 | APNGのフレームレート設定 |
| 開始時間 | 数値入力 | 動画の開始位置（秒） |
| 継続時間 | 数値入力 | APNGの長さ（秒） |
| スケール | 数値入力 | 出力幅（ピクセル） |
| 最適化有効 | チェックボックス | PNGopt最適化の有効/無効 |
| 最適化オプション | テキスト入力 | PNGoptのコマンドラインオプション |

## 4. データ設計

### 4.1 使用データ構造

```vb
' 設定はObjectStorage経由で永続化
s.Storage.GetInt("PNGFrameRate", 15)
s.Storage.GetString("PNGTime", "15.0")
s.Storage.GetString("PNGLength", "3.8")
s.Storage.GetInt("PNGScale", 480)
s.Storage.GetString("PNGopt", "-z1")
s.Storage.GetBool("OptSetting", False)
s.Storage.GetBool("PNGOutput", False)
s.Storage.GetString("PNGDirectory", p.DefaultTargetFolder)
```

## 5. 外部インターフェース

### 5.1 外部ツール連携

| ツール名 | 用途 |
|----------|------|
| ffmpeg | APNG生成 |
| PNGopt | APNG最適化（オプション） |

### 5.2 ffmpegコマンド

```
ffmpeg -ss [Seek] -t [Duration] -i "[inputFile]"
       -lavfi "fps=[Rate],scale=[Size]:-1:flags=spline"
       -plays 0 -loglevel quiet -an -y "[OutputPath.apng]"
```

### 5.3 PNGoptコマンド

```
pngopt [OptSettings] "[InputAPNG]" "[OutputPNG]"
```

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | 入力ファイル未存在 | 処理を中断し、何も出力しない |
| E-002 | ffmpeg検証失敗 | VerifyOK(True)でエラーダイアログ表示 |
| E-003 | PNGopt検証失敗 | VerifyOK(True)でエラーダイアログ表示 |
| E-004 | ファイル削除失敗 | 例外をキャッチして無視（Try-Catch） |

## 7. セキュリティ考慮事項

- ファイルパスにはエスケープ処理（ダブルクォート囲い）を適用
- 一時ファイルは適切に管理・削除

## 8. パフォーマンス考慮事項

- 最適化処理は任意で、品質/ファイルサイズのトレードオフ
- `-loglevel quiet`オプションでログ出力を抑制
- スケール指定により出力サイズを制限

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **Animation.aPNG メソッド** (`Source/General/Extras.vb` 71-141行目)
   - エントリーポイント
   - パラメータ取得とffmpeg/PNGopt呼び出しの全体フロー

2. **Proc クラス** (`Source/General/Proc.vb`)
   - 外部プロセス実行の共通処理

3. **Package.ffmpeg, Package.PNGopt** (`Source/General/Package.vb`)
   - パッケージの定義と検証

### 9.2 プログラム呼び出し階層図

```
Animation.aPNG()
  ├── File.Exists() - ファイル存在確認
  ├── Package.ffmpeg.VerifyOK() - ffmpegパッケージ検証
  ├── Package.PNGopt.VerifyOK() - PNGoptパッケージ検証
  ├── Project.Init() - プロジェクト初期化（必要時）
  ├── s.Storage.GetXxx() - 設定値取得
  ├── Proc.Start() - APNG生成（ffmpeg）
  ├── File.Delete() - 既存ファイル削除
  ├── [最適化有効時]
  │    ├── Proc.Start() - APNG最適化（PNGopt）
  │    └── File.Delete() - 一時ファイル削除
  └── [最適化無効時]
       └── File.Move() - ファイルリネーム
```

### 9.3 データフロー図

```
[入力動画] → [ffmpeg] → [一時APNG(.apng)]
                              ↓
               ┌──────────────┴──────────────┐
               ↓                              ↓
        [最適化有効]                    [最適化無効]
               ↓                              ↓
        [PNGopt]                      [File.Move]
               ↓                              ↓
        [最適化PNG]                   [出力PNG]
```

### 9.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/General/Extras.vb | 実装 | Animation.aPNGメソッド定義 |
| Source/General/Proc.vb | 実装 | 外部プロセス実行クラス |
| Source/General/Package.vb | 実装 | ffmpeg/PNGoptパッケージ定義 |
| Source/General/ApplicationSettings.vb | 実装 | 設定ストレージ |
| Source/Forms/MainForm.vb | UI | メニューからの呼び出し元 |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
