# 機能設計書: プロジェクト保存/読込

## 1. 機能概要

### 1.1 機能名
プロジェクト保存/読込（Project Save/Load）

### 1.2 機能説明
エンコーディングプロジェクトの状態をファイルに保存し、後から読み込んで作業を再開できる機能。プロジェクトには、ソースファイル情報、エンコーダー設定、フィルター設定、オーディオトラック、クロップ設定など、エンコード処理に必要な全ての設定が含まれる。

### 1.3 関連画面
- メインフォーム（MainForm）- ファイルメニューから保存/読込
- JobsForm - ジョブとしてプロジェクトを読込

## 2. 機能詳細

### 2.1 入力仕様（保存時）

| 項目名 | データ型 | 必須 | 説明 |
|--------|----------|------|------|
| Project | Project | Yes | プロジェクトオブジェクト |
| FilePath | String | Yes | 保存先ファイルパス |

### 2.2 出力仕様（読込時）

| 項目名 | データ型 | 説明 |
|--------|----------|------|
| Project | Project | 読み込まれたプロジェクトオブジェクト |

### 2.3 Projectクラスの主要プロパティ

```vb
Public Class Project
    ' ソースファイル情報
    Public SourceFile As String
    Public SourceFiles As List(Of String)
    Public SourceWidth As Integer = 1920
    Public SourceHeight As Integer = 1080
    Public SourceFrameRate As Decimal
    Public SourceFrames As Integer

    ' ターゲット設定
    Public TargetFile As String
    Public TargetWidth As Integer = 1920
    Public TargetHeight As Integer = 1080
    Public TargetFrameRate As Double
    Public VideoBitrate As Integer = 5000
    Public TargetSize As Integer = 5000

    ' クロップ設定
    Public CropLeft As Integer
    Public CropRight As Integer
    Public CropTop As Integer
    Public CropBottom As Integer

    ' エンコーダー/フィルター
    Public VideoEncoder As VideoEncoder
    Public Script As TargetVideoScript
    Public SourceScript As SourceVideoScript

    ' オーディオ
    Public AudioFiles As List(Of AudioProfile)
    Public AudioTracks As List(Of AudioTrack)

    ' その他
    Public Log As LogBuilder
    Public TempDir As String
    Public Ranges As List(Of Range)
End Class
```

### 2.4 処理フロー（保存）

```
開始
  │
  ├─ プロジェクト状態検証
  │    └─ 必須項目チェック
  │
  ├─ シリアライズ処理
  │    ├─ SafeSerializationによる安全なシリアライズ
  │    └─ バイナリ形式で保存
  │
  ├─ ファイル書き込み
  │    └─ .srip拡張子で保存
  │
終了
```

### 2.5 処理フロー（読込）

```
開始
  │
  ├─ ファイル存在確認
  │
  ├─ デシリアライズ処理
  │    ├─ SafeSerializationによる安全なデシリアライズ
  │    └─ バージョン互換性チェック
  │
  ├─ Project.Init() 呼び出し
  │    ├─ Null参照の初期化
  │    ├─ デフォルト値の設定
  │    └─ マイグレーション処理
  │
  ├─ プロジェクト状態復元
  │    └─ UI更新
  │
終了
```

### 2.6 ビジネスルール

| ルールID | ルール内容 |
|----------|------------|
| BR-001 | プロジェクトファイルの拡張子は.srip |
| BR-002 | シリアライズはSafeSerializationを使用 |
| BR-003 | 読込時にInit()で初期化処理を実行 |
| BR-004 | バージョン不整合時はマイグレーション処理を実行 |
| BR-005 | TargetFileに不正な文字が含まれる場合は警告を表示 |

## 3. 画面設計

### 3.1 メニュー項目

| メニュー | 項目 | ショートカット |
|----------|------|---------------|
| File | Save Project | Ctrl+S |
| File | Save Project As... | Ctrl+Shift+S |
| File | Open Project... | Ctrl+O |

### 3.2 保存ダイアログ

- フィルター: "StaxRip Project (*.srip)|*.srip"
- 初期ディレクトリ: 前回使用したディレクトリまたはドキュメントフォルダ

## 4. データ設計

### 4.1 シリアライズ方式

- バイナリシリアライズを使用
- `<Serializable()>` 属性をクラスに付与
- `<NonSerialized>` 属性でシリアライズ対象外を指定

### 4.2 バージョン管理

```vb
Public Versions As Dictionary(Of String, Integer)

Function Check(obj As Object, key As String, version As Integer) As Boolean
    Return SafeSerialization.Check(Me, obj, key, version)
End Function
```

### 4.3 初期化処理（Init）

```vb
Sub Init() Implements ISafeSerialization.Init
    If Versions Is Nothing Then Versions = New Dictionary(Of String, Integer)
    If TempDir Is Nothing Then TempDir = ""
    If Log Is Nothing Then Log = New LogBuilder
    If Storage Is Nothing Then Storage = New ObjectStorage
    ' ... その他の初期化
    Migrate()
End Sub
```

## 5. 外部インターフェース

### 5.1 ISafeSerialization インターフェース

```vb
Public Interface ISafeSerialization
    Property WasUpdated As Boolean
    ReadOnly Property Versions As Dictionary(Of String, Integer)
    Sub Init()
End Interface
```

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | ファイル未存在 | エラーメッセージ表示 |
| E-002 | デシリアライズ失敗 | バージョン不整合の可能性を通知 |
| E-003 | 不正なファイル形式 | エラーメッセージ表示 |
| E-004 | TargetFileに不正文字 | 警告メッセージ表示、プロパティ設定中止 |

## 7. セキュリティ考慮事項

- シリアライズされたデータは信頼できるソースからのみ読み込む想定
- ファイルパスのバリデーションを実施

## 8. パフォーマンス考慮事項

- バイナリシリアライズにより高速な保存/読込を実現
- 大きなプロジェクトでもメモリ効率的に処理

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **Project クラス** (`Source/General/Project.vb` 1-370行目)
   - プロジェクトデータ構造の定義
   - Init()メソッドで初期化ロジックを確認

2. **SafeSerialization** (`Source/General/SafeSerialization.vb`)
   - シリアライズ/デシリアライズの実装

3. **MainForm** (`Source/Forms/MainForm.vb`)
   - LoadProject/SaveProjectメソッド

### 9.2 プログラム呼び出し階層図

```
[保存処理]
MainForm.SaveProject()
  ├── Project → シリアライズ
  └── FileHelp.WriteAllBytes() → ファイル書き込み

[読込処理]
MainForm.LoadProject()
  ├── FileHelp.ReadAllBytes() → ファイル読み込み
  ├── SafeSerialization.Deserialize() → デシリアライズ
  ├── Project.Init() → 初期化
  │    └── Migrate() → マイグレーション
  └── UI更新
```

### 9.3 データフロー図

```
[保存]
Project オブジェクト → SafeSerialization → バイナリデータ → .srip ファイル

[読込]
.srip ファイル → バイナリデータ → SafeSerialization → Project.Init() → Project オブジェクト
```

### 9.4 重要な行番号

- **Project.Init()**: 173-221行目 - 初期化ロジック
- **Migrate()**: 223-231行目 - マイグレーション処理
- **TargetFile プロパティ**: 285-300行目 - バリデーション付きプロパティ
- **Check()**: 169-171行目 - バージョンチェック

### 9.5 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/General/Project.vb | 実装 | プロジェクトクラス定義 |
| Source/General/SafeSerialization.vb | 実装 | シリアライズ/デシリアライズ |
| Source/Forms/MainForm.vb | UI | 保存/読込メニュー |
| Source/General/ObjectStorage.vb | 実装 | 汎用ストレージ |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
