# 機能設計書: ジョブキュー

## 1. 機能概要

### 1.1 機能名
ジョブキュー（Job Queue Management）

### 1.2 機能説明
複数のエンコードプロジェクトをジョブとしてキューに登録し、一括処理を可能にする機能。JobsFormでジョブの追加、削除、順序変更、有効/無効切り替えを行い、JobManagerでジョブデータを永続化する。

### 1.3 関連画面
- JobsForm - ジョブキュー管理ダイアログ
- メインフォーム（MainForm）- ジョブキューへの追加

## 2. 機能詳細

### 2.1 Job クラス構造

```vb
Public Class Job
    Public Path As String           ' プロジェクトファイルパス
    Public Active As Boolean        ' 有効/無効フラグ
    Public Name As String           ' 表示名
    ' ... その他プロパティ
End Class
```

### 2.2 処理フロー

```
開始
  │
  ├─ JobsForm表示
  │
  ├─ ジョブ一覧読込（JobManager.GetJobs()）
  │    └─ Jobs.datファイルからデシリアライズ
  │
  ├─ ListViewにジョブ表示
  │    ├─ チェックボックスで有効/無効
  │    └─ Job.Activeとバインド
  │
  ├─ ユーザー操作待機
  │    ├─ Start（開始）
  │    │    └─ アクティブジョブの処理開始
  │    │
  │    ├─ Load（読込）
  │    │    └─ 選択ジョブをプロジェクトとして開く
  │    │
  │    ├─ Remove（削除）
  │    │    └─ 選択ジョブを削除
  │    │
  │    ├─ Up/Down（並び替え）
  │    │    └─ ジョブ順序変更
  │    │
  │    ├─ Check/Uncheck
  │    │    └─ ジョブの有効/無効切り替え
  │    │
  │    └─ Sort（ソート）
  │         └─ アルファベット順にソート
  │
  ├─ 変更時
  │    └─ JobManager.SaveJobs() → Jobs.dat保存
  │
終了
```

### 2.3 ビジネスルール

| ルールID | ルール内容 |
|----------|------------|
| BR-001 | ジョブデータはJobs.datファイルに保存 |
| BR-002 | Active=Trueのジョブのみ実行対象 |
| BR-003 | ジョブ削除時、Batch Projectsフォルダ内のファイルも削除 |
| BR-004 | FileSystemWatcherでファイル変更を監視し、自動リロード |
| BR-005 | Start押下後、フォームを閉じてジョブ処理開始 |

### 2.4 コンテキストメニュー操作

| 操作 | ショートカット | 説明 |
|------|---------------|------|
| Select All | Ctrl+S | 全項目選択 |
| Select None | Shift+S | 選択解除 |
| Check Selection | Space | 選択項目を有効化 |
| Check All | Ctrl+C | 全項目有効化 |
| Uncheck Selection | Space | 選択項目を無効化 |
| Uncheck All | Ctrl+U | 全項目無効化 |
| Move Selection Up | Ctrl+Up | 上へ移動 |
| Move Selection Down | Ctrl+Down | 下へ移動 |
| Move To Top | Ctrl+Home | 先頭へ移動 |
| Move To Bottom | Ctrl+End | 末尾へ移動 |
| Sort Alphabetically | Ctrl+Shift+S | アルファベット順ソート |
| Remove Selection | Delete | 選択項目削除 |
| Load Selection | Ctrl+L | 選択項目を開く |

## 3. 画面設計

### 3.1 画面レイアウト

```
+----------------------------------------------------------+
|  [ジョブリスト - チェックボックス付きListView]            |
|    [ ] Job 1 - movie1.srip                               |
|    [x] Job 2 - movie2.srip                               |
|    [x] Job 3 - movie3.srip                               |
+----------------------------------------------------------+
| [Start]  [Up] [Down]  [Menu] [Remove] [Load]             |
+----------------------------------------------------------+
```

### 3.2 UI要素

| 要素名 | 種類 | 説明 |
|--------|------|------|
| lv | ListViewEx | ジョブ一覧（チェックボックス付き） |
| bnStart | Button | ジョブ処理開始 |
| bnUp | Button | 選択項目を上へ |
| bnDown | Button | 選択項目を下へ |
| bnMenu | Button | コンテキストメニュー表示 |
| bnRemove | Button | 選択項目削除 |
| bnLoad | Button | 選択項目をプロジェクトとして開く |

## 4. データ設計

### 4.1 永続化

- **保存先**: `{SettingsFolder}/Jobs.dat`
- **形式**: バイナリシリアライズ
- **監視**: FileSystemWatcherによる変更監視

### 4.2 JobManager クラス

```vb
Public Class JobManager
    Shared Function GetJobs() As List(Of Job)
        ' Jobs.datからジョブリストを読込
    End Function

    Shared Sub SaveJobs(jobs As List(Of Job))
        ' Jobs.datにジョブリストを保存
    End Sub
End Class
```

## 5. 外部インターフェース

### 5.1 ファイル監視

```vb
FileWatcher.Path = Folder.Settings
FileWatcher.NotifyFilter = NotifyFilters.LastWrite Or NotifyFilters.CreationTime
FileWatcher.Filter = "Jobs.dat"
FileWatcher.EnableRaisingEvents = True
```

### 5.2 イベントハンドラ

- `FileWatcher.Changed` → Reload()
- `FileWatcher.Created` → Reload()
- `lv.ItemsChanged` → HandleItemsChanged()

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | Jobs.dat読込失敗 | 空のリストで初期化 |
| E-002 | ジョブファイル未存在 | ジョブをリストから削除 |

## 7. セキュリティ考慮事項

- ジョブファイルパスのバリデーション
- Batch Projectsフォルダ外のファイルは削除しない

## 8. パフォーマンス考慮事項

- BlockSaveフラグで一括操作時の重複保存を防止
- FileWatcher.EnableRaisingEventsで保存時の再読込を防止

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **JobsForm クラス** (`Source/Forms/JobsForm.vb` 1-451行目)
   - UIとジョブ管理のメインロジック

2. **JobManager クラス** (`Source/General/JobManager.vb`)
   - ジョブデータの永続化

3. **Job クラス** (`Source/General/Job.vb`)
   - ジョブデータ構造

### 9.2 プログラム呼び出し階層図

```
JobsForm()
  ├── InitializeComponent() - UI初期化
  ├── JobManager.GetJobs() - ジョブ読込
  ├── lv.AddItems() - リスト表示
  │
  ├── bnStart_Click()
  │    ├── g.VerifyRequirements() - 要件確認
  │    ├── Close() - フォーム閉じる
  │    └── g.ProcessJobs() - ジョブ処理開始
  │
  ├── bnLoad_Click()
  │    └── g.MainForm.LoadProject() - プロジェクト読込
  │
  ├── HandleItemsChanged()
  │    ├── SaveJobs() - ジョブ保存
  │    └── UpdateControls() - UI更新
  │
  └── SaveJobs()
       ├── FileWatcher.EnableRaisingEvents = False
       ├── JobManager.SaveJobs()
       └── FileWatcher.EnableRaisingEvents = True
```

### 9.3 データフロー図

```
[Jobs.dat] ← JobManager.GetJobs() ← [JobsForm]
                                         ↓
                                    [ListView表示]
                                         ↓
                                    [ユーザー操作]
                                         ↓
[Jobs.dat] ← JobManager.SaveJobs() ← [ジョブリスト更新]
```

### 9.4 重要な行番号

- **コンストラクタ**: 194-254行目 - UI初期化、イベント登録
- **bnStart_Click**: 389-404行目 - ジョブ実行開始
- **bnLoad_Click**: 406-412行目 - プロジェクト読込
- **SaveJobs**: 371-387行目 - ジョブ保存
- **Reload**: 351-362行目 - ファイル変更時のリロード
- **コンテキストメニュー設定**: 227-244行目

### 9.5 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/Forms/JobsForm.vb | UI | ジョブキュー管理ダイアログ |
| Source/General/JobManager.vb | 実装 | ジョブデータ永続化 |
| Source/General/Job.vb | 実装 | ジョブデータ構造 |
| Source/UI/ListViewEx.vb | UI | 拡張ListViewコントロール |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
