# 機能設計書: ジョブ実行

## 1. 機能概要

### 1.1 機能名
ジョブ実行（Job Processing）

### 1.2 機能説明
ジョブキューに登録されたアクティブなジョブを順次実行する機能。各ジョブに対してプロジェクトの読み込み、エンコード処理の実行、完了後の後処理を行う。複数インスタンス起動やバックグラウンド処理にも対応。

### 1.3 関連画面
- ProcessingForm - エンコード処理進捗表示
- JobsForm - ジョブ実行開始トリガー
- メインフォーム（MainForm）- 処理状態表示

## 2. 機能詳細

### 2.1 処理フロー

```
開始
  │
  ├─ g.VerifyRequirements() - 要件確認
  │    └─ 失敗時 → 終了
  │
  ├─ 別インスタンス実行判定
  │    ├─ g.IsJobProcessing = True の場合
  │    │    └─ 新規インスタンス起動（-StartJobs -NoFocus）
  │    └─ False の場合
  │         └─ 現在インスタンスで処理
  │
  ├─ g.ProcessJobs() 呼び出し
  │
  ├─ アクティブジョブ取得
  │    └─ JobManager.GetJobs().Where(Active = True)
  │
  ├─ 各ジョブに対してループ
  │    ├─ プロジェクト読込（LoadProject）
  │    ├─ エンコード処理実行
  │    │    ├─ ソース準備
  │    │    ├─ フィルター適用
  │    │    ├─ ビデオエンコード
  │    │    ├─ オーディオエンコード
  │    │    └─ マルチプレクス
  │    ├─ ジョブ完了処理
  │    └─ 次のジョブへ
  │
  ├─ 全ジョブ完了後
  │    ├─ イベントコマンド実行（ApplicationEvent.JobsEncoded）
  │    └─ 完了通知
  │
終了
```

### 2.2 ビジネスルール

| ルールID | ルール内容 |
|----------|------------|
| BR-001 | Active=Trueのジョブのみ処理対象 |
| BR-002 | 処理中は g.IsJobProcessing = True |
| BR-003 | 別インスタンスで処理中の場合は新規インスタンス起動 |
| BR-004 | ジョブ完了後はキューから自動削除 |
| BR-005 | エラー発生時は該当ジョブをスキップして次へ |

### 2.3 インスタンス管理

```vb
' 別インスタンス起動
If g.IsJobProcessing Then
    g.ShellExecute(Application.ExecutablePath, "-StartJobs -NoFocus")
Else
    Task.Run(Sub()
        Thread.Sleep(500)
        g.MainForm.Invoke(Sub() g.ProcessJobs())
    End Sub)
End If
```

### 2.4 コマンドライン引数

| 引数 | 説明 |
|------|------|
| -StartJobs | ジョブ処理を開始 |
| -NoFocus | フォーカスを奪わない（バックグラウンド実行） |

## 3. 画面設計

### 3.1 ProcessingForm

エンコード処理中の進捗状況を表示するダイアログ。

### 3.2 主要UI要素

| 要素名 | 種類 | 説明 |
|--------|------|------|
| 進捗バー | ProgressBar | 処理進捗表示 |
| 経過時間 | Label | エンコード経過時間 |
| 残り時間 | Label | 推定残り時間 |
| 現在フレーム | Label | 処理中のフレーム番号 |
| FPS | Label | 処理速度（フレーム/秒） |

## 4. データ設計

### 4.1 ジョブ状態管理

```vb
Public Class Job
    Public Path As String           ' プロジェクトファイルパス
    Public Active As Boolean        ' 有効/無効フラグ
    Public Name As String           ' 表示名
End Class
```

### 4.2 処理状態フラグ

```vb
g.IsJobProcessing As Boolean        ' ジョブ処理中フラグ
```

## 5. 外部インターフェース

### 5.1 イベント通知

| イベント | タイミング |
|----------|------------|
| ApplicationEvent.JobsEncoded | 全ジョブ完了時 |
| ApplicationEvent.BeforeJobProcessing | ジョブ処理開始前 |
| ApplicationEvent.AfterJobProcessing | ジョブ処理完了後 |

### 5.2 グローバル関数

```vb
g.ProcessJobs()         ' ジョブ処理開始
g.VerifyRequirements()  ' 要件確認
g.ShellExecute()        ' 外部プロセス起動
```

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | 要件未満足 | 処理中断、エラーメッセージ表示 |
| E-002 | プロジェクト読込失敗 | 該当ジョブスキップ |
| E-003 | エンコード失敗 | エラーログ記録、次ジョブへ |

## 7. セキュリティ考慮事項

- コマンドライン引数の検証
- プロジェクトファイルパスのバリデーション

## 8. パフォーマンス考慮事項

- 非同期処理（Task.Run）による応答性確保
- Thread.Sleep(500)で状態安定化を待機
- 複数インスタンスによる並列処理対応

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **JobsForm.bnStart_Click** (`Source/Forms/JobsForm.vb` 389-404行目)
   - ジョブ実行開始のトリガー

2. **g.ProcessJobs** (`Source/General/GlobalClass.vb`)
   - ジョブ処理のメインロジック

3. **ProcessingForm** (`Source/Forms/ProcessingForm.vb`)
   - 進捗表示UI

### 9.2 プログラム呼び出し階層図

```
bnStart_Click()
  ├── g.VerifyRequirements() - 要件確認
  ├── Close() - JobsForm閉じる
  │
  ├── [別インスタンス処理中]
  │    └── g.ShellExecute(Application.ExecutablePath, "-StartJobs -NoFocus")
  │
  └── [現在インスタンス]
       └── Task.Run()
            └── g.MainForm.Invoke()
                 └── g.ProcessJobs()
                      ├── JobManager.GetJobs() - ジョブ取得
                      ├── For Each job In activeJobs
                      │    ├── LoadProject()
                      │    ├── Process() - エンコード実行
                      │    └── ジョブ完了処理
                      └── ApplicationEvent.JobsEncoded 発火
```

### 9.3 データフロー図

```
[JobsForm] → [Start] → [g.ProcessJobs()]
                              ↓
                     [JobManager.GetJobs()]
                              ↓
                     [アクティブジョブ抽出]
                              ↓
                     [各ジョブ処理ループ]
                              ↓
                     [プロジェクト読込] → [エンコード] → [完了処理]
                              ↓
                     [全ジョブ完了]
                              ↓
                     [イベント通知]
```

### 9.4 重要な行番号

- **bnStart_Click**: 389-404行目 - 実行開始処理
- **g.IsJobProcessing判定**: 396行目
- **新規インスタンス起動**: 397行目
- **Task.Run非同期実行**: 399-402行目

### 9.5 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/Forms/JobsForm.vb | UI | ジョブ実行トリガー |
| Source/Forms/ProcessingForm.vb | UI | 進捗表示 |
| Source/General/GlobalClass.vb | 実装 | ProcessJobs()メソッド |
| Source/General/JobManager.vb | 実装 | ジョブデータ管理 |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
