# 機能設計書: イベントコマンド

## 1. 機能概要

### 1.1 機能名
イベントコマンド（Event Commands）

### 1.2 機能説明
アプリケーションの特定イベント発生時に自動的にコマンドを実行する機能。エンコード完了、ジョブ完了、アプリケーション終了などのイベントをトリガーとして、ユーザー定義のコマンドを実行できる。条件（Criteria）を設定して、特定の状況下でのみコマンドを実行することも可能。

### 1.3 関連画面
- EventCommandsEditor - イベントコマンド一覧管理
- EventCommandEditor - 個別イベントコマンド編集

## 2. 機能詳細

### 2.1 EventCommand クラス

```vb
Public Class EventCommand
    Public Name As String               ' コマンド名
    Public Enabled As Boolean           ' 有効/無効フラグ
    Public Event As ApplicationEvent    ' トリガーイベント
    Public CriteriaList As List(Of Criteria)  ' 実行条件
    Public CommandParameters As CommandParameters  ' コマンドパラメータ
    Public OrOnly As Boolean            ' 条件マッチモード（Any/All）
End Class
```

### 2.2 ApplicationEvent（トリガーイベント）

| イベント | 説明 |
|----------|------|
| ApplicationExit | アプリケーション終了時 |
| JobProcessed | 個別ジョブ完了時 |
| JobsEncoded | 全ジョブ完了時 |
| BeforeEncoding | エンコード開始前 |
| AfterEncoding | エンコード完了後 |
| VideoEncoded | ビデオエンコード完了時 |
| AudioEncoded | オーディオエンコード完了時 |
| AfterSourceLoaded | ソース読込完了時 |

### 2.3 処理フロー

```
開始
  │
  ├─ EventCommandsEditor表示
  │
  ├─ イベントコマンド一覧表示
  │    └─ ListViewにコマンド名・イベント表示
  │
  ├─ ユーザー操作待機
  │    ├─ Add（追加）
  │    │    └─ EventCommandEditorで新規作成
  │    │
  │    ├─ Edit（編集）
  │    │    └─ EventCommandEditorで編集
  │    │
  │    ├─ Clone（複製）
  │    │    └─ ObjectHelp.GetCopyで複製
  │    │
  │    ├─ Remove（削除）
  │    │    └─ リストから削除
  │    │
  │    └─ Up/Down（並び替え）
  │         └─ 順序変更
  │
  ├─ OK押下時
  │    └─ 設定保存
  │
終了
```

### 2.4 条件（Criteria）システム

```vb
Public Class Criteria
    Public Name As String           ' 条件名
    Public Description As String    ' 説明
    Public Macro As String         ' マクロ名
End Class
```

条件はマクロを使用して評価：
- `%source_file%` - ソースファイルパス
- `%target_file%` - 出力ファイルパス
- `%video_encoder%` - ビデオエンコーダー名
- 他多数のマクロ対応

### 2.5 ビジネスルール

| ルールID | ルール内容 |
|----------|------------|
| BR-001 | Enabled=Trueのコマンドのみ実行対象 |
| BR-002 | Match All: 全条件がTrueの場合のみ実行 |
| BR-003 | Match Any: いずれかの条件がTrueで実行 |
| BR-004 | 条件なしの場合は常に実行 |
| BR-005 | コマンドはCommandManagerから選択 |

## 3. 画面設計

### 3.1 EventCommandsEditor

```
+--------------------------------------------------+
|  [イベントコマンドリスト - ListView]              |
|    [x] Shutdown after encoding                    |
|    [ ] Copy to network drive                      |
|    [x] Send notification                          |
+--------------------------------------------------+
| [Add...] [Edit...] [Clone] [Remove] [Up] [Down]  |
+--------------------------------------------------+
|                    [OK]  [Cancel]                 |
+--------------------------------------------------+
```

### 3.2 EventCommandEditor

```
+--------------------------------------------------+
| Name: [____________________________]              |
| Event: [Dropdown: AfterEncoding ▼]               |
+--------------------------------------------------+
| Criteria                                          |
|   (●) Match All  ( ) Match Any   [Add]           |
|   [条件リスト]                                    |
+--------------------------------------------------+
| Command                                           |
|   [Selected Command Name        ] [▼]            |
|   Parameters:                                     |
|   [PropertyGrid]                                  |
+--------------------------------------------------+
|                    [OK]  [Cancel]                 |
+--------------------------------------------------+
```

### 3.3 UI要素

| 要素名 | 種類 | 説明 |
|--------|------|------|
| tbName | TextBox | コマンド名入力 |
| cbEvent | ComboBox | イベント選択 |
| rbMatchAllCriteria | RadioButton | 全条件マッチ |
| rbMatchAnyCriteria | RadioButton | いずれかマッチ |
| CriteriaControl | Control | 条件リスト管理 |
| tbCommand | TextBox | 選択コマンド表示 |
| bnCommand | Button | コマンド選択メニュー |
| pgParameters | PropertyGrid | コマンドパラメータ |

## 4. データ設計

### 4.1 CommandParameters

```vb
Public Class CommandParameters
    Public MethodName As String         ' メソッド名
    Public Parameters As List(Of Object) ' パラメータリスト
End Class
```

### 4.2 永続化

- 設定は ApplicationSettings に保存
- EventCommands リストとしてシリアライズ

## 5. 外部インターフェース

### 5.1 CommandManager

```vb
g.MainForm.CustomMainMenu.CommandManager.Commands
```

利用可能なコマンド一覧を取得し、メニューに表示。

### 5.2 マクロ連携

条件評価時にMacroクラスのマクロを使用：
```vb
Macro.GetMacros(True, False, True, True, False)
```

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | コマンド未選択 | 保存可能だが実行時に何も起きない |
| E-002 | パラメータ不正 | エラーメッセージ表示 |

## 7. セキュリティ考慮事項

- 任意のコマンド実行が可能なため、信頼できる環境での使用を想定
- 外部プログラム実行コマンドには注意が必要

## 8. パフォーマンス考慮事項

- イベント発生時に条件評価を行うため、複雑な条件は処理遅延の原因になりうる
- コマンド実行は非同期で行うことを推奨

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **EventCommandsEditor** (`Source/Forms/EventCommandsEditor.vb` 1-308行目)
   - イベントコマンド一覧管理UI

2. **EventCommandEditor** (`Source/Forms/EventCommandEditor.vb` 1-535行目)
   - 個別コマンド編集UI

3. **EventCommand クラス** (`Source/General/EventCommand.vb`)
   - データ構造定義

4. **CommandManager** (`Source/General/Command.vb`)
   - コマンド管理

### 9.2 プログラム呼び出し階層図

```
EventCommandsEditor()
  ├── InitializeComponent() - UI初期化
  ├── lv.AddItems(eventCommands) - リスト表示
  │
  ├── bnAdd_Click()
  │    └── EventCommandEditor(New EventCommand)
  │
  ├── bnEdit_Click()
  │    └── EventCommandEditor(selectedCommand)
  │
  └── bnClone_Click()
       └── ObjectHelp.GetCopy()

EventCommandEditor()
  ├── InitializeComponent() - UI初期化
  ├── cbEvent.Items.AddRange() - イベント一覧
  ├── Macro.GetMacros() - 条件用マクロ取得
  ├── CriteriaControl.AllCriteria = allCriteria
  │
  ├── MenuClick(c As Command)
  │    └── CommandParameters設定
  │
  └── OnFormClosed()
       └── EventCommand更新
```

### 9.3 データフロー図

```
[ApplicationSettings] → [EventCommands リスト] → [EventCommandsEditor]
                                                        ↓
                                                  [ListView表示]
                                                        ↓
                                              [Add/Edit] → [EventCommandEditor]
                                                        ↓
                                              [EventCommand更新]
                                                        ↓
                                              [ApplicationSettings保存]
```

### 9.4 重要な行番号（EventCommandEditor.vb）

- **コンストラクタ**: 359-410行目 - 初期化、マクロ取得
- **MenuClick**: 434-437行目 - コマンド選択時の処理
- **SetCommandParameters**: 439-496行目 - パラメータUI設定
- **OnFormClosed**: 498-506行目 - 保存処理

### 9.5 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/Forms/EventCommandsEditor.vb | UI | イベントコマンド一覧 |
| Source/Forms/EventCommandEditor.vb | UI | 個別コマンド編集 |
| Source/General/EventCommand.vb | 実装 | データ構造 |
| Source/General/Command.vb | 実装 | コマンド管理 |
| Source/General/Macro.vb | 実装 | マクロ定義 |
| Source/UI/CriteriaControl.vb | UI | 条件コントロール |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
