# 機能設計書: マクロ機能

## 1. 機能概要

### 1.1 機能名
マクロ機能（Macro System）

### 1.2 機能説明
プロジェクト情報や環境変数を動的にテキストに展開する機能。`%macro_name%`形式のプレースホルダーを実際の値に置換し、コマンドライン引数、ファイル名生成、イベントコマンドなどで活用される。GUI操作による対話的なマクロ（ファイル選択、テキスト入力など）もサポート。

### 1.3 関連画面
- MacrosForm - マクロ一覧表示
- 各種設定ダイアログ - マクロ入力フィールド

## 2. 機能詳細

### 2.1 Macro クラス

```vb
<Serializable()>
Public Class Macro
    Implements IComparable(Of Macro)

    Property Name As String           ' マクロ名（%xxx%形式）
    Property FriendlyName As String   ' 表示名
    Property Type As Type             ' 値の型
    Property Description As String    ' 説明

    Shared Function GetMacros(...) As List(Of Macro)
    Shared Function Expand(value As String) As String
    Shared Function ExpandGUI(value As String) As (Value, Caption, Cancel)
End Class
```

### 2.2 マクロカテゴリ

| カテゴリ | フラグ | 説明 |
|----------|--------|------|
| グローバル | includeGlobals | ソース/ターゲット情報、環境情報 |
| インタラクティブ | includeInteractive | ファイル選択、テキスト入力 |
| パラメータ付き | includeParam | 引数を取るマクロ |
| 処理中 | includeWhileProcessing | エンコード処理中のみ有効 |
| オーディオ固有 | includeAudioSpecific | オーディオエンコード用 |

### 2.3 主要マクロ一覧（一部）

| マクロ名 | 型 | 説明 |
|----------|------|------|
| %source_file% | String | ソースファイルパス |
| %target_file% | String | ターゲットファイルパス |
| %source_width% | Integer | ソース幅 |
| %source_height% | Integer | ソース高さ |
| %target_width% | Integer | ターゲット幅 |
| %target_height% | Integer | ターゲット高さ |
| %video_bitrate% | Integer | ビデオビットレート |
| %audio_bitrate% | Integer | オーディオビットレート |
| %crop_left% | Integer | 左クロップ値 |
| %crop_right% | Integer | 右クロップ値 |
| %encoder% | String | エンコーダー名 |
| %temp_dir% | String | 一時ディレクトリ |
| %current_date% | String | 現在日付 |

### 2.4 対話的マクロ

| マクロ名 | 説明 |
|----------|------|
| $browse_file$ | ファイル選択ダイアログ表示 |
| $enter_text$ | テキスト入力ダイアログ表示 |
| $enter_text:prompt$ | プロンプト付きテキスト入力 |
| $select:param1;param2;...$ | 選択肢ダイアログ |

### 2.5 パラメータ付きマクロ

| マクロ名 | 説明 |
|----------|------|
| %audio_file:X% | X番目のオーディオファイル |
| %app_path:name% | 指定ツールのパス |
| %app_dir:name% | 指定ツールのディレクトリ |
| %eval:expression% | PowerShell式の評価 |
| %filter:name% | フィルタースクリプト取得 |

### 2.6 処理フロー

```
開始
  │
  ├─ Macro.Expand(value) 呼び出し
  │
  ├─ 空文字チェック
  │    └─ 空の場合 → 空文字を返す
  │
  ├─ %empty%処理
  │    └─ 空文字に置換
  │
  ├─ 各マクロの展開（順次処理）
  │    ├─ 日時系マクロ
  │    ├─ ソースファイル系マクロ
  │    ├─ ターゲットファイル系マクロ
  │    ├─ クロップ系マクロ
  │    ├─ MediaInfo系マクロ
  │    ├─ エンコーダー系マクロ
  │    ├─ パス系マクロ
  │    ├─ パラメータ付きマクロ（正規表現）
  │    └─ 環境変数展開
  │
  ├─ 展開完了チェック
  │    └─ %が残っていなければ → 結果を返す
  │
終了
```

## 3. 画面設計

### 3.1 MacrosForm

マクロ一覧をヘルプとして表示するダイアログ。

### 3.2 マクロ入力サポート

各種設定画面でマクロ入力可能なテキストボックスにヒント表示。

## 4. データ設計

### 4.1 マクロ定義

```vb
ret.Add(New Macro("source_file", "Source File Path",
                  GetType(String), "File path of the source video."))
```

### 4.2 展開ロジック

```vb
If value.Contains("%source_file%") Then
    value = value.Replace("%source_file%", proj.SourceFile)
End If
```

### 4.3 正規表現によるパラメータ解析

```vb
matches = Regex.Matches(value, "%audio_file:(\d+)%")
For Each match As Match In matches
    Dim track = match.Groups(1).Value.ToInt() - 1
    ' ...
Next
```

## 5. 外部インターフェース

### 5.1 MediaInfo連携

```vb
MediaInfo.GetVideo(miFile, "Format")
MediaInfo.GetAudio(miFile, trackIndex, property)
```

### 5.2 PowerShell連携

```vb
PowerShell.InvokeAndConvert(expression)
```

### 5.3 Package連携

```vb
Package.Items.Values.FirstOrDefault(...)
```

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | 未知のマクロ | そのまま残す（展開しない） |
| E-002 | eval式エラー | 例外メッセージを値として設定 |
| E-003 | 対話マクロキャンセル | Cancel=Trueを返す |

## 7. セキュリティ考慮事項

- %eval:%マクロはPowerShellコードを実行するため、信頼できる入力のみ使用
- ファイルパスマクロは外部入力を直接受け入れない

## 8. パフォーマンス考慮事項

- Contains()チェックで不要な置換処理をスキップ
- マクロ展開後に%が残っていなければ早期リターン
- MediaInfo呼び出しは必要な場合のみ実行

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **Macro クラス定義** (`Source/General/Macro.vb` 1-120行目)
   - データ構造とコンストラクタ

2. **GetMacros()** (`Source/General/Macro.vb` 124-255行目)
   - 利用可能なマクロの定義一覧

3. **Expand()** (`Source/General/Macro.vb` 340-689行目)
   - マクロ展開のメインロジック

4. **ExpandGUI()** (`Source/General/Macro.vb` 257-337行目)
   - 対話的マクロの展開

### 9.2 プログラム呼び出し階層図

```
Macro.Expand(value, proj)
  ├── 日時マクロ展開
  │    └── Date.Now.ToString()
  │
  ├── ソース/ターゲット情報展開
  │    └── proj.SourceFile, proj.TargetFile, etc.
  │
  ├── MediaInfo展開
  │    └── MediaInfo.GetVideo/GetAudio/GetText()
  │
  ├── パラメータ付きマクロ展開
  │    └── Regex.Matches() → ループ処理
  │
  ├── アプリケーションパス展開
  │    └── Package.Items.Values.FirstOrDefault()
  │
  ├── eval展開
  │    └── PowerShell.InvokeAndConvert()
  │
  └── 環境変数展開
       └── Environment.ExpandEnvironmentVariables()
```

### 9.3 データフロー図

```
[入力文字列（マクロ含む）]
        ↓
[Macro.Expand()]
        ↓
┌───────────────────────────┐
│ 各種マクロパターンマッチ  │
│   ├─ %source_file% → proj.SourceFile
│   ├─ %audio_file:1% → proj.AudioTracks(0).File
│   ├─ %eval:expr% → PowerShell.InvokeAndConvert(expr)
│   └─ %ENV_VAR% → Environment.GetEnvironmentVariable()
└───────────────────────────┘
        ↓
[展開済み文字列]
```

### 9.4 重要な行番号

- **Macroクラス定義**: 8-30行目
- **GetMacros()**: 124-255行目 - マクロ定義
- **ExpandGUI()**: 257-337行目 - 対話的マクロ
- **Expand()**: 340-689行目 - メイン展開ロジック
- **MediaInfo展開**: 375-398行目
- **eval展開**: 654-670行目

### 9.5 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/General/Macro.vb | 実装 | マクロクラス定義 |
| Source/Forms/MacrosForm.vb | UI | マクロ一覧表示 |
| Source/General/PowerShell.vb | 実装 | eval式評価 |
| Source/General/MediaInfo.vb | 実装 | メディア情報取得 |
| Source/General/Package.vb | 実装 | ツールパス取得 |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
