# 機能設計書: PowerShell連携

## 1. 機能概要

### 1.1 機能名
PowerShell連携（PowerShell Integration）

### 1.2 機能説明
PowerShellスクリプトをアプリケーション内から実行する機能。マクロ展開時の式評価、ユーザー定義スクリプトの実行、イベントコマンドでのカスタム処理などに使用される。System.Management.Automationを使用した.NET統合により、高度な自動化とカスタマイズが可能。

### 1.3 関連画面
- メインフォーム（MainForm）- ツールメニューからスクリプト実行
- イベントコマンドエディタ - PowerShellコマンド設定

## 2. 機能詳細

### 2.1 PowerShell クラス

```vb
Public Class PowerShell
    Shared Sub Invoke(code As String)
    Shared Function InvokeAndConvert(code As String) As String
    Shared Sub InvokeAsync(code As String)
End Class
```

### 2.2 主要メソッド

| メソッド | 説明 |
|----------|------|
| Invoke(code) | PowerShellコードを同期実行 |
| InvokeAndConvert(code) | 実行結果を文字列として取得 |
| InvokeAsync(code) | 非同期でPowerShellコードを実行 |

### 2.3 処理フロー

```
開始
  │
  ├─ PowerShell.Invoke(code) 呼び出し
  │
  ├─ System.Management.Automation.PowerShell作成
  │
  ├─ 初期化スクリプト設定
  │    ├─ StaxRipモジュールインポート
  │    ├─ 共通関数定義
  │    └─ グローバル変数設定
  │
  ├─ ユーザーコード追加
  │
  ├─ 実行
  │    ├─ Invoke() - 同期実行
  │    └─ BeginInvoke() - 非同期実行
  │
  ├─ 結果処理
  │    └─ InvokeAndConvert: 結果を文字列変換
  │
  ├─ エラー処理
  │    └─ 例外発生時はエラーメッセージを返す
  │
終了
```

### 2.4 ビジネスルール

| ルールID | ルール内容 |
|----------|------------|
| BR-001 | ユーザースクリプトはFolder.Scriptsに配置 |
| BR-002 | マクロ展開結果として使用される場合は文字列に変換 |
| BR-003 | 実行時エラーは例外として捕捉し、メッセージを返す |
| BR-004 | 非同期実行時は完了を待たない |

### 2.5 ユーザースクリプトディレクトリ

```
{SettingsFolder}/Scripts/
  ├── CustomScript1.ps1
  ├── CustomScript2.ps1
  └── ...
```

## 3. 画面設計

### 3.1 スクリプトメニュー

ツールメニュー > Scripts からユーザー定義スクリプトを実行可能。

### 3.2 マクロ統合

%eval:expression% マクロでPowerShell式を評価：

```
%eval:1 + 2%  → "3"
%eval:%source_width% / 16%  → 解像度計算結果
```

## 4. データ設計

### 4.1 スクリプト実行コンテキスト

PowerShell実行時に利用可能な変数・関数：

| 変数/関数 | 説明 |
|-----------|------|
| $p | 現在のプロジェクト（Project） |
| $g | グローバルオブジェクト |
| $s | アプリケーション設定 |

### 4.2 StaxRipモジュール

StaxRip専用のPowerShellモジュールが自動インポートされ、以下の機能を提供：

- ファイル操作ヘルパー
- MediaInfo連携
- プロジェクト操作

## 5. 外部インターフェース

### 5.1 System.Management.Automation

```vb
Using ps = System.Management.Automation.PowerShell.Create()
    ps.AddScript(code)
    Dim results = ps.Invoke()
End Using
```

### 5.2 マクロとの連携

```vb
' Macro.Expand内での呼び出し
If value.Contains("%eval:") Then
    matches = Regex.Matches(value, "%eval:(.+?)%")
    For Each ma As Match In matches
        Try
            value = value.Replace(ma.Value, PowerShell.InvokeAndConvert(ma.Groups(1).Value))
        Catch ex As Exception
            value = value.Replace(ma.Value, ex.ToString)
        End Try
    Next
End If
```

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | 構文エラー | 例外メッセージを返す |
| E-002 | 実行時エラー | 例外メッセージを返す |
| E-003 | タイムアウト | タイムアウト例外をスロー |

## 7. セキュリティ考慮事項

- PowerShellスクリプトは任意のコードを実行可能
- ユーザースクリプトは信頼できるソースからのみ使用
- 外部入力をそのままスクリプトに渡さない

## 8. パフォーマンス考慮事項

- PowerShellランタイムの初期化にオーバーヘッドあり
- 頻繁な呼び出しは避け、バッチ処理を推奨
- 非同期実行でUIブロッキングを回避

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **PowerShell クラス** (`Source/General/PowerShell.vb`)
   - メイン実装クラス

2. **Macro.Expand 内の eval 処理** (`Source/General/Macro.vb` 654-670行目)
   - マクロからの呼び出し

3. **MainForm スクリプトメニュー** (`Source/Forms/MainForm.vb`)
   - UIからの呼び出し

### 9.2 プログラム呼び出し階層図

```
[呼び出し元]
  ├── Macro.Expand() - %eval:%マクロ
  │    └── PowerShell.InvokeAndConvert()
  │
  ├── EventCommand実行
  │    └── PowerShell.Invoke()
  │
  └── ユーザースクリプト実行
       └── PowerShell.Invoke()

PowerShell.Invoke(code)
  ├── System.Management.Automation.PowerShell.Create()
  ├── ps.AddScript(initScript)  - 初期化
  ├── ps.AddScript(code)        - ユーザーコード
  └── ps.Invoke()               - 実行
```

### 9.3 データフロー図

```
[PowerShellコード]
        ↓
[PowerShell.Invoke()]
        ↓
┌───────────────────────────┐
│ System.Management.Automation │
│   ├─ 初期化スクリプト実行
│   ├─ ユーザーコード実行
│   └─ 結果収集
└───────────────────────────┘
        ↓
[実行結果（文字列/オブジェクト）]
```

### 9.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/General/PowerShell.vb | 実装 | PowerShell実行クラス |
| Source/General/Macro.vb | 実装 | evalマクロからの呼び出し |
| Source/Forms/MainForm.vb | UI | スクリプトメニュー |
| {SettingsFolder}/Scripts/ | スクリプト | ユーザー定義スクリプト |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
