# 機能設計書: 外部ツール管理

## 1. 機能概要

### 1.1 機能名
外部ツール管理（External Tool Management / Package System）

### 1.2 機能説明
StaxRipが使用する外部ツール（ffmpeg、x265、mkvmerge等）を管理する機能。ツールの検索、バージョン確認、自動ダウンロード、パス設定を行う。Packageクラスでツールを定義し、AppsFormでユーザーインターフェースを提供する。

### 1.3 関連画面
- AppsForm - 外部ツール管理ダイアログ

## 2. 機能詳細

### 2.1 Package 基底クラス

```vb
Public Class Package
    Property Name As String           ' ツール名
    Property Filename As String       ' 実行ファイル名
    Property Description As String    ' 説明
    Property WebURL As String         ' 公式サイトURL
    Property DownloadURL As String    ' ダウンロードURL
    Property Version As String        ' 現在バージョン

    ReadOnly Property Path As String  ' 実行ファイルパス
    Function VerifyOK(showError As Boolean) As Boolean
    Sub LaunchAction()
End Class
```

### 2.2 主要な外部ツール

| ツール名 | 用途 |
|----------|------|
| ffmpeg | 動画変換、GIF/APNG生成 |
| x264 | H.264エンコード |
| x265 | H.265/HEVCエンコード |
| mkvmerge | MKVマルチプレクス |
| mp4box | MP4マルチプレクス |
| NVEnc | NVIDIA HWエンコード |
| QSVEnc | Intel QSVエンコード |
| VCEEnc | AMD VCEエンコード |
| AviSynth | スクリプト処理 |
| VapourSynth | スクリプト処理 |
| MediaInfo | メディア情報取得 |

### 2.3 処理フロー

```
開始
  │
  ├─ AppsForm表示
  │
  ├─ ツール一覧読込（Package.Items）
  │    └─ 各Packageのパス・バージョン確認
  │
  ├─ TreeViewにカテゴリ別表示
  │
  ├─ ユーザー操作待機
  │    ├─ ツール選択
  │    │    └─ 詳細情報表示
  │    │
  │    ├─ Download（ダウンロード）
  │    │    ├─ DownloadURL取得
  │    │    └─ ダウンロード・展開
  │    │
  │    ├─ Launch（起動）
  │    │    └─ ツール実行
  │    │
  │    ├─ Explore（エクスプローラで開く）
  │    │    └─ ディレクトリを開く
  │    │
  │    └─ Web（公式サイト）
  │         └─ ブラウザでURL開く
  │
終了
```

### 2.4 ビジネスルール

| ルールID | ルール内容 |
|----------|------------|
| BR-001 | ツールはFolder.AppsまたはFolder.Pluginsに配置 |
| BR-002 | VerifyOK()でツール存在・バージョン確認 |
| BR-003 | 自動ダウンロード時はZIP展開後に配置 |
| BR-004 | カスタムパス設定が可能 |
| BR-005 | %app_path:name%マクロでパス参照可能 |

### 2.5 ツール検索順序

1. カスタム設定パス
2. Folder.Apps/{ToolName}/
3. Folder.Plugins/
4. システムPATH

## 3. 画面設計

### 3.1 AppsForm レイアウト

```
+--------------------------------------------------+
| [TreeView - カテゴリ/ツール]  | [詳細情報]       |
|   Video Encoder               |   Name: x265     |
|     - x265                    |   Path: ...      |
|     - x264                    |   Version: 3.5   |
|     - NVEnc                   |   Status: OK     |
|   Audio Encoder               |                  |
|   Muxer                       | [Download]       |
|   Demuxer                     | [Launch]         |
|   Helper                      | [Explore]        |
|                               | [Web]            |
+--------------------------------------------------+
|                    [OK]                          |
+--------------------------------------------------+
```

### 3.2 UI要素

| 要素名 | 種類 | 説明 |
|--------|------|------|
| tvMain | TreeView | カテゴリ・ツール一覧 |
| 詳細パネル | Panel | 選択ツールの詳細表示 |
| bnDownload | Button | ツールダウンロード |
| bnLaunch | Button | ツール起動 |
| bnExplore | Button | フォルダを開く |
| bnWeb | Button | 公式サイトを開く |

## 4. データ設計

### 4.1 Package.Items

```vb
Public Shared Items As New Dictionary(Of String, Package)

' 初期化時に登録
Items.Add("ffmpeg", New Package With {
    .Name = "ffmpeg",
    .Filename = "ffmpeg.exe",
    .WebURL = "https://ffmpeg.org/",
    ' ...
})
```

### 4.2 パス永続化

- カスタムパス設定はApplicationSettingsに保存
- ツールごとに`{ToolName}Path`キーで管理

## 5. 外部インターフェース

### 5.1 マクロ連携

```vb
' Macro.Expand内
Package.Items.Values.FirstOrDefault(
    Function(pack) pack.Name.ToLowerInvariant = name.ToLowerInvariant)
```

### 5.2 ダウンロード処理

```vb
' WebClientまたはHttpClientでダウンロード
' ZipFile.ExtractToDirectoryで展開
```

## 6. エラー処理

| エラーコード | エラー内容 | 対処方法 |
|--------------|------------|----------|
| E-001 | ツール未検出 | VerifyOK(True)でエラー表示 |
| E-002 | バージョン不整合 | 警告表示、更新推奨 |
| E-003 | ダウンロード失敗 | エラーメッセージ表示 |

## 7. セキュリティ考慮事項

- ダウンロードURLは信頼できるソースのみ使用
- 展開前にZIPファイルの検証
- 実行前にファイルの整合性確認

## 8. パフォーマンス考慮事項

- ツール検索は起動時に一度だけ実行
- バージョン確認は遅延実行（必要時のみ）
- ダウンロードは非同期で実行

## 9. コードリーディングガイド

### 9.1 推奨読解順序

1. **Package クラス** (`Source/General/Package.vb`)
   - ツール定義の基底クラス

2. **AppsForm** (`Source/Forms/AppsForm.vb`)
   - ツール管理UI

3. **Macro.Expand 内の app_path 処理** (`Source/General/Macro.vb` 560-577行目)
   - マクロからのパス取得

### 9.2 プログラム呼び出し階層図

```
AppsForm()
  ├── InitializeComponent() - UI初期化
  ├── Package.Items - ツール一覧取得
  ├── tvMain構築 - TreeView設定
  │
  ├── bnDownload_Click()
  │    ├── Package.DownloadURL取得
  │    ├── WebClient.DownloadFile()
  │    └── ZipFile.ExtractToDirectory()
  │
  ├── bnLaunch_Click()
  │    └── Package.LaunchAction()
  │
  └── bnExplore_Click()
       └── Process.Start("explorer", path.Dir)

Package.VerifyOK(showError)
  ├── Path存在確認
  ├── バージョン確認（必要に応じて）
  └── エラー時 → MsgError表示
```

### 9.3 データフロー図

```
[Package.Items 定義]
        ↓
[AppsForm TreeView表示]
        ↓
[ユーザー操作]
        ↓
┌───────────────────────────┐
│ Download → WebClient/Extract
│ Launch → Process.Start
│ VerifyOK → ファイル存在確認
└───────────────────────────┘
        ↓
[マクロ展開]
   %app_path:name% → Package.Items[name].Path
```

### 9.4 関連ファイル一覧

| ファイルパス | 種別 | 役割 |
|-------------|------|------|
| Source/General/Package.vb | 実装 | Packageクラス定義 |
| Source/Forms/AppsForm.vb | UI | 外部ツール管理ダイアログ |
| Source/General/Macro.vb | 実装 | app_pathマクロ処理 |
| Source/General/Folder.vb | 実装 | ディレクトリパス定義 |

## 10. 改訂履歴

| 版数 | 日付 | 変更内容 | 作成者 |
|------|------|----------|--------|
| 1.0 | 2025-01-21 | 初版作成 | AI |
