# API設計書

## 概要

本ドキュメントは、stdVBAライブラリが提供するHTTP通信APIおよび主要クラスの公開インターフェースについて記載しています。stdVBAはVBA（Visual Basic for Applications）向けのモダンなライブラリ群であり、HTTPリクエスト、JSON処理、配列操作などの機能を提供します。

**注意**: stdVBAはWebアプリケーションのサーバーサイドAPIではなく、VBAクライアントライブラリです。本ドキュメントではライブラリが提供するHTTPクライアント機能および主要クラスのAPIを設計書として記載します。

## 共通仕様

### ライブラリ構成

```
stdVBA
├── stdHTTP         - HTTP通信クライアント
├── stdJSON         - JSON処理
├── stdArray        - 配列操作
├── stdCallback     - コールバック関数
├── stdLambda       - インラインラムダ式
├── stdRegex        - 正規表現
├── stdAcc          - アクセシビリティ自動化
├── stdWindow       - ウィンドウ管理
├── stdProcess      - プロセス管理
├── stdClipboard    - クリップボード操作
└── その他クラス
```

### 命名規則

- クラス名: `std` プレフィックス（例: `stdHTTP`, `stdArray`）
- Excel専用クラス: `xl` プレフィックス
- コンストラクタ: `Create` または `CreateFrom*` メソッド
- 保護メソッド: `prot` プレフィックス（例: `protInit`）

### 対応プラットフォーム

| プラットフォーム | 対応状況 |
| --- | --- |
| VBA7 (64-bit) | 対応 |
| VBA7 (32-bit) | 対応 |
| VBA6 | 一部対応 |
| Mac | 一部対応 |

## API一覧

### HTTP通信 API

| クラス | メソッド | 説明 |
| --- | --- | --- |
| stdHTTP | Create | HTTPリクエストを作成・実行 |
| stdHTTP | CreateOptions | リクエストオプションを作成 |
| stdHTTP | CreateHeaders | リクエストヘッダーを作成 |
| stdHTTP | CreateProxy | プロキシ設定を作成 |
| stdHTTP | Await | 非同期リクエストの完了を待機 |
| stdHTTP | URIEncode | URI文字列をエンコード |
| stdHTTP | URIDecode | URI文字列をデコード |

### JSON処理 API

| クラス | メソッド | 説明 |
| --- | --- | --- |
| stdJSON | Create | 空のJSONオブジェクトを作成 |
| stdJSON | CreateFromString | 文字列からJSONを作成 |
| stdJSON | CreateFromFile | ファイルからJSONを作成 |
| stdJSON | CreateFromVariant | Variantからjsonを作成 |
| stdJSON | Add | 要素を追加 |
| stdJSON | AddArray | 配列を追加 |
| stdJSON | AddObject | オブジェクトを追加 |
| stdJSON | ToString | JSON文字列に変換 |
| stdJSON | ToFile | ファイルに出力 |

### 配列操作 API

| クラス | メソッド | 説明 |
| --- | --- | --- |
| stdArray | Create | 配列を作成 |
| stdArray | CreateFromArray | VBA配列から作成 |
| stdArray | CreateFromCollection | Collectionから作成 |
| stdArray | Push/Pop | 末尾への追加/削除 |
| stdArray | Map/Filter/Reduce | 関数型操作 |
| stdArray | Sort | ソート |
| stdArray | Join | 文字列結合 |

## 各APIエンドポイント定義

### stdHTTP

#### 1. Create - HTTPリクエスト作成

HTTPリクエストを作成し、同期または非同期で実行します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | `stdHTTP.Create(...)` |
| 種別 | コンストラクタ（静的メソッド） |
| 非同期対応 | 対応 |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| sURL | String | ○ | - | リクエスト先URL |
| RequestMethod | String | - | "GET" | HTTPメソッド（GET, POST, PUT, DELETE等） |
| ThreadingStyle | EHTTPSynchronisity | - | HTTPSync | 同期/非同期スタイル |
| options | Object | - | Nothing | リクエストオプション（CreateOptionsで作成） |
| Authenticator | stdICallable | - | Nothing | 認証処理コールバック |

**使用例（同期リクエスト）**

```vb
Dim req As stdHTTP
Set req = stdHTTP.Create("https://api.example.com/data")
Debug.Print req.ResponseText
Debug.Print req.ResponseStatus  ' 200
```

**使用例（非同期リクエスト）**

```vb
Dim req As stdHTTP
Set req = stdHTTP.Create("https://api.example.com/data", ThreadingStyle:=HTTPAsync)
req.Await
Debug.Print req.ResponseText
```

**使用例（POSTリクエスト）**

```vb
Dim req As stdHTTP
Set req = stdHTTP.Create( _
    "https://api.example.com/data", _
    RequestMethod:="POST", _
    options:=stdHTTP.CreateOptions( _
        Body:="{""name"":""value""}", _
        Headers:=stdHTTP.CreateHeaders("Content-Type", "application/json") _
    ) _
)
Debug.Print req.ResponseText
```

**レスポンスプロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| ResponseStatus | Long | HTTPステータスコード |
| ResponseStatusText | String | ステータステキスト |
| ResponseText | String | レスポンスボディ（テキスト） |
| ResponseBody | Byte() | レスポンスボディ（バイナリ） |
| ResponseHeader(name) | Variant | 指定ヘッダーの値 |
| ResponseHeaders | Object | 全レスポンスヘッダー |
| isFinished | Boolean | リクエスト完了状態 |

---

#### 2. CreateOptions - リクエストオプション作成

HTTPリクエストのオプションを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | `stdHTTP.CreateOptions(...)` |
| 種別 | 静的メソッド |
| 戻り値 | Object (Dictionary) |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| Body | String | - | "" | リクエストボディ |
| Headers | Object | - | Nothing | リクエストヘッダー |
| Cookies | Object | - | Nothing | クッキー |
| ContentTypeDefault | EHTTPContentType | - | - | デフォルトContent-Type |
| Insecure | Boolean | - | False | SSL検証を無視 |
| EnableRedirects | Boolean | - | True | リダイレクト有効化 |
| TimeoutMS | Long | - | 5000 | タイムアウト（ミリ秒） |
| Proxy | Object | - | Nothing | プロキシ設定 |

**使用例**

```vb
Dim opts As Object
Set opts = stdHTTP.CreateOptions( _
    Body:="{""key"":""value""}", _
    Headers:=stdHTTP.CreateHeaders("Authorization", "Bearer token123"), _
    TimeoutMS:=10000, _
    Insecure:=False _
)
```

**Content-Type列挙値（EHTTPContentType）**

| 値 | 説明 |
| --- | --- |
| ContentType_HTML | text/html |
| ContentType_Plain | text/plain |
| ContentType_JSON | application/json |
| ContentType_XML | application/xml |
| ContentType_CSV | text/csv |
| ContentType_ZIP | application/zip |
| ContentType_Binary | application/octet-stream |

---

#### 3. CreateHeaders - ヘッダー作成

リクエストヘッダーのDictionaryを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | `stdHTTP.CreateHeaders(...)` |
| 種別 | 静的メソッド |
| 戻り値 | Object (Dictionary) |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| v | ParamArray | - | key1, value1, key2, value2, ... 形式 |

**使用例**

```vb
Dim headers As Object
Set headers = stdHTTP.CreateHeaders( _
    "Content-Type", "application/json", _
    "Authorization", "Bearer token123", _
    "X-Custom-Header", "custom-value" _
)
```

---

#### 4. CreateProxy - プロキシ設定作成

プロキシ設定オブジェクトを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | `stdHTTP.CreateProxy(...)` |
| 種別 | 静的メソッド |
| 戻り値 | Object (Dictionary) |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| proxyType | EHTTPProxyType | ○ | - | プロキシタイプ |
| Server | String | - | - | プロキシサーバーアドレス |
| ByPass | String | - | - | バイパスリスト |
| User | String | - | - | 認証ユーザー名 |
| Pass | String | - | - | 認証パスワード |

**プロキシタイプ列挙値（EHTTPProxyType）**

| 値 | 説明 |
| --- | --- |
| ProxyTypeAuto | システムプロキシ設定を自動検出 |
| ProxyTypeManual | 手動でプロキシを指定 |
| ProxyTypeNone | プロキシを使用しない |

**使用例**

```vb
Dim proxy As Object
Set proxy = stdHTTP.CreateProxy( _
    ProxyTypeManual, _
    Server:="192.168.1.100:8080", _
    User:="proxyuser", _
    Pass:="proxypass" _
)
```

---

### 認証機能（stdHTTPAuthenticators）

#### WindowsAuthenticator

Windows認証を使用します。

```vb
Set req = stdHTTP.Create( _
    "https://internal.example.com/api", _
    Authenticator:=stdCallback.CreateFromModule("stdHTTPAuthenticators", "WindowsAuthenticator") _
)
```

#### HttpBasicAuthenticator

HTTP Basic認証を使用します。

```vb
Set req = stdHTTP.Create( _
    "https://api.example.com/protected", _
    Authenticator:=stdCallback.CreateFromModule("stdHTTPAuthenticators", "HttpBasicAuthenticator") _
        .Bind("username", "password") _
)
```

#### TokenAuthenticator

トークン認証（APIキー等）を使用します。

```vb
Set req = stdHTTP.Create( _
    "https://api.example.com/v1/data", _
    Authenticator:=stdCallback.CreateFromModule("stdHTTPAuthenticators", "TokenAuthenticator") _
        .Bind("X-API-Key", "your-api-key-here") _
)
```

---

### stdJSON

#### 1. Create - JSONオブジェクト作成

空のJSONオブジェクトまたは配列を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | `stdJSON.Create(...)` |
| 種別 | コンストラクタ |
| 戻り値 | stdJSON |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| iType | EStdJsonType | - | eJSONObject | JSONタイプ |

**JSONタイプ列挙値（EStdJsonType）**

| 値 | 説明 |
| --- | --- |
| eJSONObject | オブジェクト {} |
| eJSONArray | 配列 [] |

**使用例**

```vb
' オブジェクト作成
Dim obj As stdJSON
Set obj = stdJSON.Create()
obj.Add "name", "John"
obj.Add "age", 30
Debug.Print obj.ToString()  ' {"name": "John", "age": 30}

' 配列作成
Dim arr As stdJSON
Set arr = stdJSON.Create(eJSONArray)
arr.Add 1
arr.Add 2
arr.Add 3
Debug.Print arr.ToString()  ' [1, 2, 3]
```

---

#### 2. CreateFromString - 文字列からJSON作成

JSON文字列をパースしてstdJSONオブジェクトを作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | `stdJSON.CreateFromString(...)` |
| 種別 | コンストラクタ |
| 戻り値 | stdJSON |

**パラメータ**

| パラメータ名 | 型 | 必須 | デフォルト | 説明 |
| --- | --- | --- | --- | --- |
| sJSON | String | ○ | - | JSON文字列 |
| AllowUnquotedKeys | Boolean | - | False | 引用符なしキーを許可 |
| AllowSingleQuoteStrings | Boolean | - | False | シングルクォート文字列を許可 |
| StoreLargeNumsAsString | Boolean | - | False | 大きな数値を文字列として保持 |

**使用例**

```vb
Dim json As stdJSON
Set json = stdJSON.CreateFromString("{""name"":""John"",""items"":[1,2,3]}")
Debug.Print json("name")        ' John
Debug.Print json("items")(1)    ' 1
```

---

#### 3. プロパティとメソッド

| メンバー | 種別 | 説明 |
| --- | --- | --- |
| Item(keyOrIndex) | プロパティ | 要素の取得/設定 |
| Length | プロパティ | 要素数 |
| JsonType | プロパティ | JSONタイプ |
| Exists(keyOrIndex) | プロパティ | 存在確認 |
| Add(...) | メソッド | 要素追加 |
| AddArray(...) | メソッド | 配列追加 |
| AddObject(...) | メソッド | オブジェクト追加 |
| ToString(pretty) | メソッド | 文字列変換 |
| ToFile(path) | メソッド | ファイル出力 |
| ToVBObject() | メソッド | VBオブジェクトに変換 |
| ChildrenInfo() | メソッド | 子要素情報取得 |

---

### stdArray

#### 1. Create - 配列作成

要素を指定して配列を作成します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| メソッド | `stdArray.Create(...)` |
| 種別 | コンストラクタ |
| 戻り値 | stdArray |

**パラメータ**

| パラメータ名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| params | ParamArray | - | 配列要素 |

**使用例**

```vb
Dim arr As stdArray
Set arr = stdArray.Create(1, 2, 3, 4, 5)
Debug.Print arr.Join(",")  ' 1,2,3,4,5
```

---

#### 2. 配列操作メソッド

| メソッド | 説明 | 使用例 |
| --- | --- | --- |
| Push(el) | 末尾に追加 | `arr.Push 6` |
| Pop() | 末尾から削除して返す | `val = arr.Pop()` |
| Shift() | 先頭から削除して返す | `val = arr.Shift()` |
| Unshift(el) | 先頭に追加 | `arr.Unshift 0` |
| Remove(index) | 指定位置から削除 | `arr.Remove 3` |
| Slice(start, end) | 部分配列取得 | `Set sub = arr.Slice(1, 3)` |
| Concat(arr2) | 配列結合 | `Set merged = arr.Concat(arr2)` |
| Clone() | 複製 | `Set copy = arr.Clone()` |
| Reverse() | 逆順 | `Set rev = arr.Reverse()` |
| Sort() | ソート | `Set sorted = arr.Sort()` |

---

#### 3. 関数型操作メソッド

| メソッド | 説明 | 使用例 |
| --- | --- | --- |
| Map(cb) | 変換 | `Set doubled = arr.Map(stdLambda.Create("$1*2"))` |
| Filter(cb) | フィルタ | `Set evens = arr.Filter(stdLambda.Create("$1 mod 2 = 0"))` |
| Reduce(cb, init) | 集約 | `sum = arr.Reduce(stdLambda.Create("$1+$2"), 0)` |
| ForEach(cb) | 反復 | `arr.ForEach(stdLambda.Create("Debug.Print $1"))` |
| IsEvery(cb) | 全一致判定 | `allPositive = arr.IsEvery(stdLambda.Create("$1>0"))` |
| IsSome(cb) | 一部一致判定 | `hasNeg = arr.IsSome(stdLambda.Create("$1<0"))` |
| Unique(cb) | 重複除去 | `Set uniq = arr.Unique()` |
| GroupBy(cb) | グループ化 | `Set groups = arr.GroupBy(stdLambda.Create("$1 mod 2"))` |
| Max(cb) | 最大値 | `maxVal = arr.Max()` |
| Min(cb) | 最小値 | `minVal = arr.Min()` |

---

#### 4. プロパティ

| プロパティ | 型 | 説明 |
| --- | --- | --- |
| Length | Long | 要素数 |
| item(i) | Variant | i番目の要素 |
| arr | Variant | VBA配列として取得 |

---

## エラーハンドリング

### stdHTTPエラー

| エラーコード | 説明 |
| --- | --- |
| 10000 | タイムアウト |
| 12002 | WINHTTP_ERROR_TIMEOUT |
| 12007 | WINHTTP_ERROR_NAME_NOT_RESOLVED |
| 12029 | WINHTTP_ERROR_CANNOT_CONNECT |
| 12030 | WINHTTP_ERROR_CONNECTION_ERROR |
| 12037 | WINHTTP_ERROR_SECURE_CERT_DATE_INVALID |
| 12038 | WINHTTP_ERROR_SECURE_CERT_CN_INVALID |
| 12045 | WINHTTP_ERROR_SECURE_INVALID_CA |

### stdJSONエラー

| エラーコード | 説明 |
| --- | --- |
| 1 | JSONParseError - 不正なJSON形式 |

---

## 備考

### 依存関係

- stdHTTPは`WinHttp.WinHttpRequest.5.1` COMオブジェクトを使用
- stdJSONは`Scripting.Dictionary` COMオブジェクトを使用

### パフォーマンス考慮事項

- 大量のHTTPリクエストを行う場合は非同期モード（HTTPAsync）を推奨
- JSON処理で大きな数値（16桁以上）を扱う場合は`StoreLargeNumsAsString`を有効化
- stdArrayのSort()メソッドは`bSortInPlace:=True`でパフォーマンス向上

### セキュリティ考慮事項

- `Insecure:=True`オプションはSSL証明書検証を無効化するため、開発環境のみで使用
- HTTP Basic認証は平文でクレデンシャルを送信するため、HTTPS経由でのみ使用
- APIキーやパスワードはコード内にハードコーディングせず、環境変数や設定ファイルから読み込むことを推奨
