# stdVBA コードリーディングガイドライン

## はじめに

このガイドラインは、stdVBAのコードベースを効率的に理解するための手引きです。
VBA（Visual Basic for Applications）に精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

**stdVBAとは:**
stdVBAは、VBAアプリケーション開発のための標準ライブラリコレクションです。JavaScript風のモダンなAPIを提供し、配列操作、ラムダ式、正規表現、ウィンドウ管理、プロセス制御などの機能を実現します。

---

## 1. 言語基礎

> このセクションでは、VBAの基本構文と概念を解説します。

### 1.1 プログラム構造

VBAのクラスファイル（`.cls`）は、以下のヘッダー構造で始まります。

```vb
' ファイル: src/stdArray.cls:1-9
VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "stdArray"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = False
Attribute VB_PredeclaredId = True
Attribute VB_Exposed = False
```

**主要なAttribute:**
- `VB_Name`: クラス名を定義
- `VB_PredeclaredId = True`: 静的クラスとして動作（インスタンス化せずに使用可能）
- `VB_Exposed`: 外部公開設定

### 1.2 データ型と変数

VBAでは、`Dim`キーワードで変数を宣言し、`As`で型を指定します。

```vb
' ファイル: src/stdArray.cls:90-97
Private Type TThis
  BaseArray() As Variant
  Length As Long
  ProxyLength As Long
  Chunking As Long
  Initialised As Boolean
End Type
Private This As TThis
```

**主要なデータ型:**
| 型 | 説明 | 例 |
|---|---|---|
| `Long` | 32ビット整数 | `Dim i As Long` |
| `String` | 文字列 | `Dim s As String` |
| `Boolean` | 真偽値 | `Dim b As Boolean` |
| `Variant` | 任意の型 | `Dim v As Variant` |
| `Object` | オブジェクト参照 | `Dim obj As Object` |
| `LongPtr` | ポインタ（32/64ビット互換） | `Dim ptr As LongPtr` |

**Type構造体:**
stdVBAでは、クラスの内部状態を`Private Type TThis`で定義し、`Private This As TThis`でインスタンス変数として保持するパターンを採用しています。

### 1.3 制御構造

```vb
' ファイル: src/stdArray.cls:148-163（条件分岐とループ）
Public Function Create(ParamArray params() As Variant) As stdArray
    Set Create = New stdArray

    Dim i As Long
    Dim lb As Long: lb = LBound(params)
    Dim ub As Long: ub = UBound(params)

    Call Create.protInit(ub - lb + 1, 10)

    For i = lb To ub
        Call Create.Push(params(i))
    Next

    RaiseEvent AfterCreate(Create)
End Function
```

**主要な制御構造:**
- `If...Then...Else...End If`: 条件分岐
- `For...To...Next`: カウンタ付きループ
- `For Each...In...Next`: コレクション反復
- `Do While...Loop`: 条件付きループ
- `Select Case...End Select`: 多分岐

### 1.4 関数/メソッド定義

VBAでは、戻り値のある処理は`Function`、戻り値のない処理は`Sub`で定義します。

```vb
' ファイル: src/stdCallback.cls:185-198
Public Function Create(mType As String, Parent As Variant, name As String, _
    Optional cType As Integer = VbCallType.VbMethod) As stdCallback
  Dim iType As ParentType
  If mType = "Module" Then
    iType = ParentType.Module
  ElseIf mType = "Object" Then
    iType = ParentType.object
  Else
    CriticalRaise "Incorrect module type supplied"
  End If

  Set Create = New stdCallback
  Call Create.protInit(iType, Parent, name, cType)
End Function
```

**アクセス修飾子:**
- `Public`: 外部からアクセス可能
- `Private`: クラス内部のみ
- `Friend`: 同一プロジェクト内からアクセス可能

### 1.5 モジュール/インポート

VBAには明示的なインポート構文はありませんが、同一プロジェクト内のクラス・モジュールは自動的に参照可能です。

```vb
' ファイル: src/stdLambda.cls:69-70
'Implement stdICallable interface
Implements stdICallable
```

`Implements`キーワードでインターフェースを実装します。stdVBAでは`stdICallable`が共通インターフェースとして使用されています。

---

## 2. プロジェクト固有の概念

> このセクションでは、stdVBA特有の概念を解説します。

### 2.1 ファクトリークラスパターン

stdVBAの全クラスは、ファクトリーパターンを採用しています。

```vb
' ファイル: Templates/ClassTemplate.cls:11-17
Public Function Create() as ClassTemplate
  set Create = new ClassTemplate
  Call Create.protInit()
End Function

Friend Sub protInit()

End Sub
```

**パターンの特徴:**
1. `Create`メソッドでインスタンスを生成
2. `protInit`（protected init）で内部初期化
3. `VB_PredeclaredId = True`により、`stdArray.Create(...)`のように静的呼び出しが可能

### 2.2 stdICallableインターフェース

stdVBAのコア概念である呼び出し可能オブジェクトのインターフェースです。

```vb
' ファイル: src/stdICallable.cls:50-72
'Run the function with supplied parameters
Public Function Run(ParamArray params() as variant) as variant: End Function

'Run the function with a array of parameters
Public Function RunEx(ByVal params as variant) as variant: End Function

'Bind a set of parameters to the function call
Public Function Bind(ParamArray params() as variant) as stdICallable: End Function

'Making late-bound calls to stdICallable members
Public Function SendMessage(ByVal sMessage as string, ByRef success as boolean, _
    ByVal params as variant) as Variant: End Function
```

このインターフェースを実装するクラス:
- `stdLambda`: ラムダ式
- `stdCallback`: コールバック関数

### 2.3 ラムダ式（stdLambda）

JavaScript風のラムダ式をVBAで実現します。

```vb
' README.md より
Debug.Print arr.Map(stdLambda.Create("$1+1")).join          '2,3,4,5,6,7,8,9,10,11
Debug.Print arr.Filter(stdLambda.Create("$1>=5")).join      '5,6,7,8,9,10
Debug.Print arr.Reduce(stdLambda.Create("$1+$2"))           '55
```

**パラメータ記法:**
- `$1`, `$2`, `$3`...: 引数参照
- `$1.property`: オブジェクトプロパティアクセス
- `$1.method()`: メソッド呼び出し

### 2.4 Win32 API統合

多くのクラスがWin32 APIを使用してWindows機能にアクセスします。

```vb
' ファイル: src/stdArray.cls:73-79
#If Mac Then
  Private Declare PtrSafe Function CopyMemory Lib "/usr/lib/libc.dylib" ...
#ElseIf VBA7 Then
  Private Declare PtrSafe Sub CopyMemory Lib "kernel32" Alias "RtlMoveMemory" ...
#Else
  Private Declare Sub CopyMemory Lib "kernel32" Alias "RtlMoveMemory" ...
#End If
```

**条件付きコンパイル:**
- `#If Mac Then`: Mac対応
- `#If VBA7 Then`: 64ビットVBA対応
- `#If Win64 Then`: 64ビットWindows対応

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `std{Name}.cls` | 汎用ライブラリクラス | `stdArray.cls`, `stdLambda.cls` |
| `stdI{Name}.cls` | インターフェース定義 | `stdICallable.cls` |
| `std{Name}.bas` | 補助モジュール | `stdHTTPAuthenticators.bas` |
| `xl{Name}.cls` | Excel固有クラス | `xlTable.cls` |
| `{Name}Tests.bas` | テストモジュール | `stdArrayTests.bas` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `Create` | コンストラクタメソッド | `Create()`, `CreateFromArray()` |
| `CreateFrom{Source}` | 特定ソースからの生成 | `CreateFromHwnd()`, `CreateFromPoint()` |
| `prot` | Protected（内部用） | `protInit()` |
| `s{Name}` | String変数 | `sMethod`, `sName` |
| `i{Name}` | Integer/Long変数 | `iIndex`, `iType` |
| `b{Name}` | Boolean変数 | `bInitialised` |
| `o{Name}` | Object変数 | `oRegex`, `oObject` |
| `v{Name}` | Variant変数 | `vResult`, `vBound` |
| `T{Name}` | Type定義 | `TThis`, `TBoundLambda` |

### 3.3 プログラム分類一覧

| 分類 | プレフィックス | 説明 |
|-----|--------------|------|
| 型・コレクション | `std` | Array, Enumerator, Dictionary等 |
| データ処理 | `std` | JSON, XML, Regex等 |
| オートメーション | `std` | Window, Process, Clipboard等 |
| Excel固有 | `xl` | Table, Timer, Events等 |
| Word固有 | `wd` | Document等 |
| PowerPoint固有 | `pp` | （未実装） |
| Access固有 | `ax` | （未実装） |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
stdVBA-master/
├── src/                    # メインソースコード
│   ├── stdArray.cls        # 配列操作クラス
│   ├── stdLambda.cls       # ラムダ式クラス
│   ├── stdCallback.cls     # コールバッククラス
│   ├── stdICallable.cls    # 呼び出し可能インターフェース
│   ├── stdEnumerator.cls   # 列挙子クラス
│   ├── stdRegex.cls        # 正規表現クラス
│   ├── stdWindow.cls       # ウィンドウ管理クラス
│   ├── stdProcess.cls      # プロセス管理クラス
│   ├── stdClipboard.cls    # クリップボード操作クラス
│   ├── stdCOM.cls          # COMオブジェクトラッパー
│   └── WIP/                # 開発中のクラス
├── tests/                  # テストコード
│   ├── Main.bas            # テストエントリーポイント
│   ├── lib/                # テストフレームワーク
│   │   └── Test.cls        # テストユーティリティ
│   └── std*Tests.bas       # 各クラスのテスト
├── docs/                   # ドキュメント
│   ├── stdLambda.md        # stdLambdaの詳細ドキュメント
│   ├── stdEnumerator.md    # stdEnumeratorの詳細ドキュメント
│   └── assets/             # 画像等のアセット
├── Templates/              # テンプレート
│   └── ClassTemplate.cls   # クラステンプレート
├── Tools/                  # 開発ツール
├── README.md               # プロジェクト概要
├── contribution.md         # 貢献ガイドライン
├── changelog.md            # 変更履歴
└── fullBuild.xlsm          # ビルド用Excelファイル
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | 本番用ライブラリコード | `std*.cls`, `std*.bas` |
| `src/WIP/` | 開発中・実験的コード | 未リリースのクラス群 |
| `tests/` | ユニットテスト | `*Tests.bas` |
| `tests/lib/` | テストフレームワーク | `Test.cls` |
| `docs/` | ドキュメント | `std*.md` |
| `Templates/` | 開発用テンプレート | `ClassTemplate.cls` |
| `Tools/` | 補助ツール | ビルド・開発支援 |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

stdVBAは「ライブラリコレクション」アーキテクチャを採用しています。各クラスは独立して動作可能であり、依存関係を最小限に抑えています。

```
┌─────────────────────────────────────────────────────────────┐
│                      Application Layer                       │
│              (Excel, Word, PowerPoint, Access)               │
└─────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────┐
│                    stdVBA Library Layer                      │
│  ┌─────────────┐ ┌─────────────┐ ┌─────────────────────────┐│
│  │    Types    │ │    Data     │ │      Automation         ││
│  │ stdArray    │ │ stdJSON     │ │ stdWindow               ││
│  │ stdEnumerator│ │ stdRegex    │ │ stdProcess              ││
│  │ stdLambda   │ │ stdXML      │ │ stdClipboard            ││
│  │ stdCallback │ │             │ │ stdAcc                  ││
│  └─────────────┘ └─────────────┘ └─────────────────────────┘│
│                         │                                    │
│                         ▼                                    │
│  ┌──────────────────────────────────────────────────────────┐│
│  │              Common Interface: stdICallable              ││
│  └──────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────┐
│                    Win32 API / COM Layer                     │
│            (kernel32, oleAut32, VBE7, msvbvm60)              │
└─────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| Types | データ構造・関数型操作 | `stdArray.cls`, `stdLambda.cls`, `stdEnumerator.cls` |
| Data | データ形式処理 | `stdJSON.cls`, `stdRegex.cls`, `stdXML.cls` |
| Automation | システム操作 | `stdWindow.cls`, `stdProcess.cls`, `stdClipboard.cls` |
| Interface | 共通契約 | `stdICallable.cls` |
| Platform | Win32/COM統合 | 各クラス内のDeclare文 |

### 5.3 データフロー

stdVBAでは、関数型プログラミングスタイルのデータフローを推奨しています。

```
入力データ
    │
    ▼
┌────────────────┐     ┌────────────────┐
│ stdArray.Create │ ──▶ │ .Map(lambda)   │
└────────────────┘     └────────────────┘
                              │
                              ▼
                       ┌────────────────┐
                       │ .Filter(lambda)│
                       └────────────────┘
                              │
                              ▼
                       ┌────────────────┐
                       │ .Reduce(lambda)│
                       └────────────────┘
                              │
                              ▼
                         最終結果
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

stdVBAはライブラリであり、単一のエントリーポイントはありません。各クラスの`Create`メソッドがエントリーポイントとして機能します。

```vb
' 使用例: README.md より
Sub Main()
  Dim arr as stdArray
  set arr = stdArray.Create(1,2,3,4,5,6,7,8,9,10)

  Debug.Print arr.Map(stdLambda.Create("$1+1")).join  '2,3,4,5,6,7,8,9,10,11
End Sub
```

### 6.2 ビジネスロジック

各stdクラスは特定のドメインのビジネスロジックをカプセル化しています。

**stdArray - 配列操作:**
```vb
' ファイル: src/stdArray.cls:144-163
Public Function Create(ParamArray params() As Variant) As stdArray
    Set Create = New stdArray
    Dim i As Long
    Dim lb As Long: lb = LBound(params)
    Dim ub As Long: ub = UBound(params)

    Call Create.protInit(ub - lb + 1, 10)

    For i = lb To ub
        Call Create.Push(params(i))
    Next
    RaiseEvent AfterCreate(Create)
End Function
```

### 6.3 データアクセス

stdVBAは外部データソースへの直接アクセスよりも、データ操作・変換に焦点を当てています。

**stdRegex - 正規表現処理:**
```vb
' ファイル: src/stdRegex.cls:109-130
Public Function Create(ByVal pattern As String, Optional ByVal flags As String = "") As stdRegex
  If Not Me Is stdRegex Then
    Call CriticalRaise("Constructor called on object not class")
    Exit Function
  End If

  'Cache regex objects for faster speed
  Static cache As Object: If cache Is Nothing Then Set cache = CreateObject("Scripting.Dictionary")
  Dim sKey As String: sKey = pattern & "&" & flags
  If Not cache.exists(sKey) Then
    Dim r As stdRegex
    Set r = New stdRegex
    Call r.protInit(pattern, flags)
    Set cache(sKey) = r
  End If
  Set Create = cache(sKey)
End Function
```

### 6.4 ユーティリティ/共通機能

**テストフレームワーク:**
```vb
' ファイル: tests/lib/Test.cls:57-66
Public Sub Assert(ByVal sName As String, ByVal result As Boolean)
    If loTests Is Nothing Then Me.protInit

    Dim lr As ListRow
    Set lr = loTests.ListRows.Add()
    lr.Range.value = Array("Test", sName, result, GetTickCount() - dtLastCall)
    If loTests.ListRows.count = 1 Then Call RefreshGlobalFormats
    If shTests Is ActiveSheet Then lr.Range.Select
    dtLastCall = GetTickCount()
End Sub
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| Factory Pattern | 静的Createメソッドでインスタンス生成 | 高 | 全stdクラス |
| TThis Pattern | Private Typeで状態管理 | 高 | 全stdクラス |
| Callback Pattern | stdICallableによる関数渡し | 高 | stdLambda, stdCallback |
| Cache Pattern | 静的変数でキャッシュ | 中 | stdRegex, stdLambda |
| Event Pattern | イベント駆動の拡張 | 中 | stdArray |
| Platform Abstraction | 条件付きコンパイル | 中 | Win32 API使用クラス |

### 各パターンの詳細

#### パターン1: Factory Pattern（ファクトリーパターン）

**目的:** インスタンス生成を制御し、静的クラスのような使用感を提供する。

**実装例:**
```vb
' ファイル: src/stdRegex.cls:109-130
Public Function Create(ByVal pattern As String, Optional ByVal flags As String = "") As stdRegex
  If Not Me Is stdRegex Then
    Call CriticalRaise("Constructor called on object not class")
    Exit Function
  End If

  Set Create = New stdRegex
  Call Create.protInit(pattern, flags)
End Function
```

**解説:** `VB_PredeclaredId = True`と組み合わせることで、`stdRegex.Create(...)`のように、newなしでインスタンスを生成できます。

#### パターン2: TThis Pattern（状態管理パターン）

**目的:** クラスの内部状態を構造化して管理する。

**実装例:**
```vb
' ファイル: src/stdArray.cls:90-97
Private Type TThis
  BaseArray() As Variant
  Length As Long
  ProxyLength As Long
  Chunking As Long
  Initialised As Boolean
End Type
Private This As TThis
```

**解説:** 複数のインスタンス変数をTypeでグループ化し、コードの可読性と保守性を向上させます。

#### パターン3: Callback Pattern（コールバックパターン）

**目的:** 関数を引数として渡し、処理をカスタマイズ可能にする。

**実装例:**
```vb
' ファイル: tests/stdArrayTests.bas:95-100
Test.Assert "indexOf 1 If can find ret index", _
    stdArray.Create(1,2,3,2,3,4).indexOf(3) = 3

' Map操作でのラムダ使用
set arr = arr.Map(stdLambda.Create("$1*2"))
```

**解説:** `stdICallable`インターフェースにより、`stdLambda`と`stdCallback`の両方を同じ方法で使用できます。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定（`Create`メソッド）
2. 処理の流れを追跡（メソッドチェーン、イベント）
3. データの変換を確認（Map, Filter, Reduce）
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: 配列のフィルタリングとマッピング

**概要:** 数値配列から偶数のみを抽出し、2倍にして合計を計算する

**処理フロー:**
```
stdArray.Create → Filter(偶数判定) → Map(2倍) → Reduce(合計)
```

**詳細な追跡:**

1. **配列の作成** (`src/stdArray.cls:148-163`)
   ```vb
   set arr = stdArray.Create(1,2,3,4,5,6,7,8,9,10)
   ```

2. **フィルタリング**
   ```vb
   set arr = arr.Filter(stdLambda.Create("$1 mod 2 = 0"))
   ' 結果: [2,4,6,8,10]
   ```

3. **マッピング**
   ```vb
   set arr = arr.Map(stdLambda.Create("$1 * 2"))
   ' 結果: [4,8,12,16,20]
   ```

4. **集約**
   ```vb
   Debug.Print arr.Reduce(stdLambda.Create("$1 + $2"))
   ' 結果: 60
   ```

#### 例2: 正規表現によるデータ抽出

**概要:** 文字列から特定パターンにマッチするデータを抽出

**処理フロー:**
```
stdRegex.Create → Match/MatchAll → 結果の加工
```

**詳細な追跡:**

1. **正規表現オブジェクトの作成** (`src/stdRegex.cls:109-130`)
   ```vb
   set rx = stdRegex.Create("(?<Site>\d{5})-(\w{2}\d)")
   ```

2. **マッチング実行**
   ```vb
   set match = rx.Match("12345-AA1")
   Debug.Print match("Site")  ' => "12345"
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイント（Createメソッド）を特定したか
- [ ] 呼び出し関係（メソッドチェーン）を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリング（CriticalRaise）を確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要と使用例
2. `contribution.md` - 設計思想と規約
3. `docs/stdLambda.md` - ラムダ式の詳細解説

#### 特定機能を理解したい場合
1. `docs/std{クラス名}.md` - 該当クラスのドキュメント
2. `src/std{クラス名}.cls` - ソースコード
3. `tests/std{クラス名}Tests.bas` - テストケース

#### 改修作業を行う場合
1. `Templates/ClassTemplate.cls` - クラステンプレート
2. `changelog.md` - 変更履歴の確認
3. `tests/lib/Test.cls` - テストフレームワークの理解

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要、使用例 | 最初に参照 |
| contribution.md | 貢献ガイドライン、設計思想 | 開発前に参照 |
| changelog.md | 変更履歴 | 更新前に参照 |
| docs/stdLambda.md | ラムダ式の詳細仕様 | ラムダ使用時 |
| docs/stdEnumerator.md | 列挙子の詳細仕様 | コレクション操作時 |
| docs/stdWindow.md | ウィンドウ操作の詳細仕様 | ウィンドウ自動化時 |
| docs/stdAcc.md | アクセシビリティAPIの詳細 | UI自動化時 |
| docs/stdProcess.md | プロセス管理の詳細仕様 | プロセス操作時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `VB_PredeclaredId = True`とは何ですか？
A: このアトリビュートを設定すると、クラスが自動的にインスタンス化され、`stdArray.Create(...)`のように静的メソッドとして呼び出せます。VBAの標準モジュールに近い動作ですが、クラスの機能（プロパティ、メソッド、イベント）を保持します。

#### Q: `protInit`とは何ですか？
A: "protected init"の略で、クラス初期化用の内部メソッドです。`Friend`修飾子により、同一プロジェクト内からのみ呼び出せます。通常は`Create`メソッドから呼び出されます。

#### Q: `#If VBA7 Then`などの条件付きコンパイルはどう読めばいいですか？
A: これはコンパイル時にコードを分岐するための構文です：
- `VBA7`: Office 2010以降（64ビットサポート）
- `Win64`: 64ビットWindows
- `Mac`: macOS環境

#### Q: `LongPtr`型とは何ですか？
A: 32ビット環境では`Long`、64ビット環境では`LongLong`として動作するポインタ型です。Win32 API呼び出しで使用します。

### デバッグのヒント

1. **テストの実行:**
   ```vb
   ' tests/Main.bas を開いてMainTestAllを実行
   Call MainTestAll
   ```

2. **ブレークポイントの設定:**
   - `Create`メソッドの先頭
   - `protInit`メソッドの先頭
   - イベントハンドラの先頭

3. **ローカルウィンドウの活用:**
   - `This`変数の内容を確認して状態を把握

4. **イミディエイトウィンドウの活用:**
   ```vb
   ? stdArray.Create(1,2,3).join()
   ```

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Factory Pattern | インスタンス生成を専用メソッドに集約するパターン |
| stdICallable | stdVBAの共通インターフェース、関数として呼び出し可能なオブジェクトを定義 |
| Lambda | 無名関数、stdVBAでは文字列から動的に生成 |
| Callback | 後から呼び出すための関数参照 |
| PredeclaredId | クラスの自動インスタンス化設定 |
| TThis | stdVBAの状態管理用Type構造体の命名規約 |
| protInit | protected initialization、内部初期化メソッド |
| Win32 API | Windows OSのネイティブAPI |
| COM | Component Object Model、Windowsのオブジェクト技術 |
| LongPtr | ポインタサイズに応じて変化する整数型 |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| src/stdArray.cls | 配列操作 | Create, Map, Filter, Reduce, Push, Pop |
| src/stdLambda.cls | ラムダ式 | Create, Run, Bind |
| src/stdCallback.cls | コールバック | CreateFromModule, CreateFromObjectMethod |
| src/stdEnumerator.cls | 列挙子 | CreateFromIEnumVariant, ForEach, Map |
| src/stdRegex.cls | 正規表現 | Create, Test, Match, MatchAll, Replace |
| src/stdWindow.cls | ウィンドウ管理 | CreateFromDesktop, CreateFromHwnd |
| src/stdProcess.cls | プロセス管理 | Create, CreateFromProcessId, ForceQuit |
| src/stdClipboard.cls | クリップボード | Text, Files, Picture |
| src/stdICallable.cls | インターフェース | Run, RunEx, Bind, SendMessage |
| tests/Main.bas | テストエントリー | MainTestAll |
| tests/lib/Test.cls | テストフレームワーク | Assert, Topic, Refresh |

### C. 参考資料

- [stdVBA GitHub リポジトリ](https://github.com/sancarn/stdVBA)
- [stdVBA Examples リポジトリ](https://github.com/sancarn/stdVBA-examples)
- [YouTube: stdVBA入門シリーズ](https://www.youtube.com/watch?v=IJ8PNiqnuMU&list=PLfpJqtusLMmotZqzFtZ6FCPRin5N_BjPA)
- [VBA言語リファレンス（Microsoft Docs）](https://docs.microsoft.com/ja-jp/office/vba/api/overview/)
- [Win32 API リファレンス（Microsoft Docs）](https://docs.microsoft.com/ja-jp/windows/win32/api/)
