# バッチ設計書 6-VBA-stdError-Wrapper/main.js

## 概要

本ドキュメントは、VBAソースコードにstdErrorラッパー処理を適用するNode.jsスクリプト「VBA-stdError-Wrapper/main.js」の設計仕様を記載するものである。

### 本バッチの処理概要

このバッチは、VBAソースコード内のすべての関数、サブルーチン、プロパティに対してエラーハンドリングコードを自動注入する。具体的には、Err.RaiseをErr_Raiseに置換し、On Error GoTo 0をstdErrorWrapper_ErrorOccurredラベルへのジャンプに置換し、各プロシージャにstdError.getSentry呼び出しとエラーハンドラを追加する。

**業務上の目的・背景**：VBAにおけるエラーハンドリングは手動で記述する必要があり、記述漏れが発生しやすい。本スクリプトにより、すべてのプロシージャに一貫したエラーハンドリング機構を自動的に注入することで、エラー発生時のスタックトレース取得やデバッグを容易にする。stdErrorライブラリと連携し、構造化されたエラー情報を提供する。

**バッチの実行タイミング**：stdError_wrap.ymlワークフローから自動実行される。masterブランチへのpush時に実行され、変換後のコードはstdError-Wrappedブランチに公開される。

**主要な処理内容**：
1. src/ディレクトリ内のファイル一覧取得
2. 各VBAファイルの読み込み
3. Err.RaiseをErr_Raiseに置換
4. On Error GoTo 0をstdErrorWrapper_ErrorOccurredに置換
5. 各プロシージャにstdError.getSentryラッパーを注入
6. Err_Raise関数をファイル末尾に追加
7. 変換後のファイルを上書き保存

**前後の処理との関連**：本スクリプトはstdError_wrap.yml（No.3）から呼び出される。VBA-stdError-Wrapper/build.bat（No.9）でTypeScriptからビルドされる必要がある。stdErrorライブラリへの依存があり、変換後のコードを実行するにはstdErrorモジュールが必要。

**影響範囲**：src/ディレクトリ配下のすべてのVBAソースファイルを変換対象とし、ファイルを直接上書きする。

## バッチ種別

コード変換 / エラーハンドリング注入

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（ワークフロー/手動） |
| 実行時刻 | - |
| 実行曜日 | - |
| 実行日 | - |
| トリガー | stdError_wrap.ymlワークフロー / 手動実行 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js | Node.js環境が利用可能であること |
| ビルド済み | build.batでmain.jsがビルドされていること |
| ソースファイル | src/ディレクトリにVBAファイルが存在すること |
| stdError | 変換後のコード実行にstdErrorライブラリが必要 |

### 実行可否判定

- src/ディレクトリが存在し、ファイルが含まれている場合に処理を実行
- ディレクトリが空の場合、処理はスキップされる

## 入力仕様

### 入力パラメータ

本スクリプトはコマンドライン引数を受け取らない。

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| src/*.bas | VBAソースファイル | 標準モジュールファイル |
| src/*.cls | VBAソースファイル | クラスモジュールファイル |
| src/*.frm | VBAソースファイル | ユーザーフォームファイル |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| src/*.bas | VBAソースファイル | stdErrorラッパー適用済み標準モジュール |
| src/*.cls | VBAソースファイル | stdErrorラッパー適用済みクラスモジュール |
| src/*.frm | VBAソースファイル | stdErrorラッパー適用済みユーザーフォーム |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | 入力ファイルと同名（上書き） |
| 出力先 | src/ディレクトリ |
| 文字コード | UTF-8 |
| 改行コード | CRLF（\r\n） |

### 変換内容

#### 置換処理

| 変換前 | 変換後 |
|-------|-------|
| Err.Raise | Err_Raise |
| On Error GoTo 0 | On Error GoTo stdErrorWrapper_ErrorOccurred |

#### プロシージャラッパー注入

変換前:
```vb
Public Function MyMethod(ByVal param1 As Type1) As ReturnType
  ' 処理内容
End Function
```

変換後:
```vb
Public Function MyMethod(ByVal param1 As Type1) As ReturnType
  With stdError.getSentry("ModuleName#MyMethod", "param1", param1)
    On Error GoTo stdErrorWrapper_ErrorOccurred
    ' 処理内容（インデント追加）
    Exit Function
    stdErrorWrapper_ErrorOccurred:
      Call Err_Raise(Err.Number, Err.Source, Err.Description)
  End With
End Function
```

#### 追加関数

```vb
Private Sub Err_Raise(ByVal number As Long, Optional ByVal source As String = "", Optional ByVal description As String = "")
  Call stdError.Raise(description)
End Sub
```

## 処理フロー

### 処理シーケンス

```
1. src/ディレクトリのファイル一覧取得
   └─ fs.readdirSync使用、ファイルのみフィルタリング
2. 各ファイルの読み込み
   └─ fs.readFileSync(utf8)
3. モジュール名取得
   └─ VB_Name属性から取得、なければファイル名
4. Err.Raise置換
   └─ 正規表現でErr_Raiseに置換
5. On Error GoTo 0置換
   └─ 正規表現でstdErrorWrapper_ErrorOccurredに置換
6. UDT情報抽出
   └─ Type宣言を検出してパラメータ除外リストに追加
7. プロシージャ検出・変換
   └─ Public/Private/Friend関数/サブ/プロパティを正規表現で検出
8. 条件付きコンパイル対応
   └─ #If...#End Ifブロックを検出してヘッダーに含める
9. パラメータ解析
   └─ parseParameters関数でパラメータ情報を抽出
10. ラッパーコード注入
    └─ stdError.getSentryとエラーハンドラを注入
11. Err_Raise関数追加
    └─ ファイル末尾に追加
12. ファイル上書き保存
    └─ fs.writeFileSync(utf8)
```

### フローチャート

```mermaid
flowchart TD
    A[スクリプト開始] --> B[src/ディレクトリ読み込み]
    B --> C[ファイル一覧フィルタリング]
    C --> D{ファイルあり?}
    D -->|なし| E[スクリプト終了]
    D -->|あり| F[ファイル読み込み]
    F --> G[モジュール名取得]
    G --> H[Err.Raise置換]
    H --> I[On Error GoTo 0置換]
    I --> J[UDT情報抽出]
    J --> K[プロシージャ検出]
    K --> L[条件付きコンパイル対応]
    L --> M[パラメータ解析]
    M --> N[ラッパーコード注入]
    N --> O[Err_Raise関数追加]
    O --> P[ファイル上書き保存]
    P --> Q{次のファイル?}
    Q -->|あり| F
    Q -->|なし| E
```

## データベース操作仕様

### 操作別データベース影響一覧

本スクリプトはデータベースを使用しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ファイル読み込みエラー | ファイルが存在しない、権限不足 | ファイルパス・権限確認 |
| - | パースエラー | 不正なVBA構文 | 対象ファイルの構文確認 |
| - | 書き込みエラー | 出力先への書き込み権限不足 | 権限確認 |
| - | 正規表現エラー | 特殊なVBA構文 | 対象ファイルの構文確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（同期処理） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

### 障害時対応

1. コンソールログでエラー内容を確認
2. 対象VBAファイルの構文を確認
3. 条件付きコンパイルブロックの構造を確認
4. UDT、ParamArray、配列パラメータは除外対象であることを確認

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 該当なし（ファイルI/O） |
| コミットタイミング | 該当なし |
| ロールバック条件 | 該当なし（上書き保存） |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 50-100ファイル程度 |
| 目標処理時間 | 30秒以内 |
| メモリ使用量上限 | Node.js標準制限内 |

## 排他制御

- 同時実行は想定しない
- ファイル書き込みは上書きモード

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 情報ログ | ファイル処理時 | 処理中のファイル名 |
| エラーログ | エラー発生時 | エラーメッセージ |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 実行時間 | 60秒 | コンソール出力 |

## 備考

- TypeScript（main.ts）からesbuildでビルド
- ソースマップ（main.js.map）も生成
- 条件付きコンパイル（#If...#End If）に対応
- 以下のパラメータ型はラッパーのパラメータログから除外:
  - UDT（ユーザー定義型）
  - ParamArray
  - 配列（()付きパラメータ）
- コールスタック名形式: "ModuleName#MethodName" または "ModuleName#MethodName[Get|Let|Set]"
- stdError.getSentryの引数: コールスタック名, パラメータ名1, 値1, パラメータ名2, 値2, ...
