# 機能設計書 1-stdArray

## 概要

本ドキュメントは、stdVBAライブラリにおける動的配列クラス `stdArray` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdArrayは、VBAの標準配列の制限を克服し、JavaScriptやPythonのような現代的なプログラミング言語と同様の配列操作機能を提供する動的配列クラスである。

**業務上の目的・背景**：VBAの標準配列は固定長であり、要素の追加・削除のたびにReDimを手動で管理する必要がある。また、Map、Filter、Reduceといった関数型プログラミングのパラダイムに基づく配列操作メソッドが存在しない。stdArrayはこれらの課題を解決し、より直感的で保守性の高いコードを記述可能にする。

**機能の利用シーン**：
- データの動的な収集と蓄積（ログ記録、データ集計）
- 配列データのフィルタリング、変換、ソート処理
- コレクションデータの関数型処理チェーン
- Excelシートからのデータ読み込みと加工

**主要な処理内容**：
1. 動的な要素追加（Push, Unshift）と削除（Pop, Shift, Remove）
2. 配列の変換処理（Map, Filter, Reduce）
3. 配列のソート処理（QuickSortアルゴリズム）
4. 配列の結合・分割（Concat, Slice, Splice, SplitToChunks）
5. 要素検索（indexOf, lastIndexOf, includes）
6. 集約処理（Max, Min, Count, GroupBy）
7. イベント駆動型の配列操作（BeforeAdd, AfterAdd等のイベント）

**関連システム・外部連携**：stdICallableインターフェースを実装したコールバックオブジェクト（stdLambda, stdCallback）と連携し、高階関数を実現する。

**権限による制御**：特になし。すべてのVBA実行環境で利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないライブラリクラス |

## 機能種別

データ構造（動的配列） / データ操作 / 関数型プログラミング支援

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| params | ParamArray(Variant) | No | Create時の初期要素 | なし |
| iInitialLength | Long | No | 初期配列長 | 0以上 |
| iChunking | Long | No | チャンクサイズ | 1以上 |
| arr | Variant(Array) | Yes | CreateFromArray時の配列 | IsArray = True |
| col | Collection | Yes | CreateFromCollection時のコレクション | Nothing以外 |
| cb | stdICallable | Yes(Map等) | コールバックオブジェクト | Nothing以外 |

### 入力データソース

- パラメータとして渡される値
- VBA配列（Variant配列）
- VBA Collection
- 文字列（CreateFromStringで分割）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Length | Long | 配列の要素数 |
| item(i) | Variant | i番目の要素 |
| arr | Variant() | VBA配列形式のデータ |

### 出力先

- stdArrayオブジェクト自身（メソッドチェーン用）
- VBA配列（arr プロパティ）
- Collection（AsCollection メソッド）
- 文字列（Join メソッド）

## 処理フロー

### 処理シーケンス

```
1. 初期化
   └─ protInit: 内部配列のサイズとチャンキング設定を初期化

2. 要素追加（Push）
   └─ BeforeAddイベント発火 → キャンセル判定
   └─ 配列サイズチェック → 必要に応じてReDim
   └─ 要素追加 → Length増加
   └─ AfterAddイベント発火

3. 要素削除（Pop）
   └─ BeforeRemoveイベント発火 → キャンセル判定
   └─ 最後の要素を取得
   └─ Length減少
   └─ AfterRemoveイベント発火

4. Map処理
   └─ Clone()で配列複製
   └─ 各要素にコールバック適用
   └─ 結果を新配列として返却

5. Sort処理
   └─ SortStruct配列へコピー
   └─ QuickSortアルゴリズム実行
   └─ 結果を元配列または新配列へコピー
```

### フローチャート

```mermaid
flowchart TD
    A[Create/CreateFromArray] --> B[protInit]
    B --> C{要素操作}
    C -->|Push| D[BeforeAddイベント]
    D -->|キャンセル?| E{bCancel}
    E -->|Yes| F[終了]
    E -->|No| G[サイズチェック]
    G -->|拡張必要| H[ReDim]
    G -->|十分| I[要素追加]
    H --> I
    I --> J[AfterAddイベント]
    J --> K[戻り値返却]

    C -->|Map| L[Clone]
    L --> M[ループ開始]
    M --> N[コールバック実行]
    N --> O{次要素?}
    O -->|Yes| M
    O -->|No| P[新配列返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | インデックス基準 | 配列のインデックスは1から始まる（1-based） | 全操作 |
| BR-002 | チャンキング | 配列拡張時は指定サイズ単位で拡張 | Push, Unshift |
| BR-003 | イベントキャンセル | BeforeAdd/BeforeRemoveでキャンセル可能 | Push, Pop, Remove, Unshift |
| BR-004 | 不変性 | Map, Filter等は新しい配列を返却（元配列は変更しない） | Map, Filter, Reduce, Unique |
| BR-005 | ソートインプレース | bSortInPlace=Trueで元配列を直接ソート可能 | Sort |

### 計算ロジック

**チャンキング計算**:
```
必要チャンク数 = Ceil(Length / Chunking)
新配列サイズ = 必要チャンク数 * Chunking
```

**QuickSort**:
- 中央値をピボットとして選択
- 左右に分割してパーティション
- 再帰的にソート実行

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本クラスはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 未初期化エラー | Initialised=Falseで操作 | エラー処理（コメントアウト状態） |
| - | インデックスエラー | 範囲外インデックス指定 | エラー処理（コメントアウト状態） |
| - | ソートアルゴリズムエラー | 無効なアルゴリズム指定 | stdError.Raise |

### リトライ仕様

リトライ仕様なし（同期処理のみ）

## トランザクション仕様

トランザクション管理なし（メモリ内操作のみ）

## パフォーマンス要件

- Push操作: O(1) 平均、O(n) 最悪（配列拡張時）
- Pop操作: O(1)
- Sort操作: O(n log n) 平均、O(n²) 最悪
- Map/Filter操作: O(n)
- indexOf操作: O(n)

## セキュリティ考慮事項

- 外部入力の検証は呼び出し側の責任
- コールバック実行時の例外は呼び出し側で処理

## 備考

- VB_PredeclaredId = True により、stdArrayをインスタンス化せずにファクトリメソッドを呼び出し可能
- For Each構文に対応（NewEnumプロパティ）
- イベント（WithEvents）による拡張が可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、stdArrayの内部データ構造（TThis型）を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdArray.cls | `src/stdArray.cls` | 81-97行目: SortDirection, SortStruct, TThis型の定義 |

**読解のコツ**:
- `TThis`型がクラスの内部状態を保持
- `BaseArray`が実際のデータ格納配列
- `Length`が論理的な長さ、`ProxyLength`が物理的な配列サイズ
- `Chunking`が拡張単位

#### Step 2: コンストラクタを理解する

ファクトリメソッドパターンによるオブジェクト生成を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdArray.cls | `src/stdArray.cls` | 148-163行目: Createメソッド |
| 2-2 | stdArray.cls | `src/stdArray.cls` | 234-242行目: protInitメソッド（初期化処理） |

**主要処理フロー**:
1. **148行目**: `Create`関数 - ParamArrayで可変長引数を受け取る
2. **155行目**: `protInit`呼び出し - 内部配列の初期化
3. **157-159行目**: 各要素をPushで追加
4. **162行目**: AfterCreateイベント発火

#### Step 3: 基本操作を理解する

Push/Popを中心とした基本的な配列操作を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdArray.cls | `src/stdArray.cls` | 428-450行目: Pushメソッド |
| 3-2 | stdArray.cls | `src/stdArray.cls` | 454-473行目: Popメソッド |
| 3-3 | stdArray.cls | `src/stdArray.cls` | 518-527行目: Shiftメソッド |
| 3-4 | stdArray.cls | `src/stdArray.cls` | 532-558行目: Unshiftメソッド |

**主要処理フロー**:
- **432行目**: BeforeAddイベント発火とキャンセル判定
- **435-438行目**: 配列サイズ拡張判定とReDim
- **440-441行目**: 要素追加とLength更新
- **444行目**: AfterAddイベント発火

#### Step 4: 高階関数を理解する

Map, Filter, Reduceなどの関数型プログラミングメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdArray.cls | `src/stdArray.cls` | 904-923行目: Mapメソッド |
| 4-2 | stdArray.cls | `src/stdArray.cls` | 1003-1021行目: Filterメソッド |
| 4-3 | stdArray.cls | `src/stdArray.cls` | 966-998行目: Reduceメソッド |

**主要処理フロー**:
- **907行目**: Clone()で配列を複製
- **910-917行目**: 各要素にコールバック適用、PutItemで結果設定

#### Step 5: ソートアルゴリズムを理解する

QuickSortによるソート処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | stdArray.cls | `src/stdArray.cls` | 304-344行目: Sortメソッド |
| 5-2 | stdArray.cls | `src/stdArray.cls` | 354-394行目: Sort_QuickSortメソッド |

**主要処理フロー**:
- **313-326行目**: SortStruct配列へのコピーとソート値計算
- **366-391行目**: QuickSortのパーティション処理
- **392-393行目**: 再帰呼び出し

### プログラム呼び出し階層図

```
stdArray.Create(params)
    │
    ├─ stdArray.protInit(iInitialLength, iChunking)
    │      └─ ReDim This.BaseArray
    │
    └─ stdArray.Push(el) [各要素に対して]
           ├─ RaiseEvent BeforeAdd
           ├─ ReDim Preserve (必要時)
           ├─ CopyVariant
           └─ RaiseEvent AfterAdd

stdArray.Map(cb)
    │
    ├─ stdArray.Clone()
    │      └─ stdArray.Push (各要素)
    │
    └─ stdICallable.Run(element) [各要素に対して]
           └─ PutItem(i, result)

stdArray.Sort(cb)
    │
    ├─ stdArray.Clone() (bSortInPlace=Falseの場合)
    │
    ├─ SortStruct配列作成
    │
    └─ Sort_QuickSort(arr, cb)
           └─ 再帰呼び出し
```

### データフロー図

```
[入力]                     [処理]                         [出力]

Create(1,2,3,4,5) ─────▶ protInit() ─────────────────▶ stdArray
                              │                           (Length=5)
                              └─▶ Push() x 5
                                      │
                                      └─▶ BaseArray()

stdArray ─────────────▶ Map(lambda) ──────────────────▶ 新stdArray
(各要素)                      │                           (変換後)
                              └─▶ cb.Run() x n

stdArray ─────────────▶ Filter(lambda) ───────────────▶ 新stdArray
(各要素)                      │                           (条件合致のみ)
                              └─▶ cb.Run() x n
                                      │
                                      └─▶ Push() (条件True時)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdArray.cls | `src/stdArray.cls` | ソース | 動的配列クラスの実装 |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース |
| stdLambda.cls | `src/stdLambda.cls` | ソース | ラムダ式生成クラス（Map等で使用） |
| stdCallback.cls | `src/stdCallback.cls` | ソース | コールバックオブジェクト生成クラス |
