# 機能設計書 2-stdEnumerator

## 概要

本ドキュメントは、stdVBAライブラリにおけるイテレータ・列挙子クラス `stdEnumerator` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdEnumeratorは、VBAにおける様々なコレクション型データに対して統一的なイテレーション（反復処理）機能を提供するクラスである。配列、Collection、ListObject、IEnumVARIANTを実装するオブジェクトなど、異なるデータソースを共通のインターフェースで操作可能にする。

**業務上の目的・背景**：VBAでは、配列とCollectionで異なる反復処理の記述が必要であり、また関数型プログラミングで一般的なMap、Filter、Reduceなどの高階関数が標準で提供されていない。stdEnumeratorはこれらの課題を解決し、Rubyの`Enumerable`モジュールに触発された豊富なメソッドを提供することで、データ処理コードの可読性と保守性を向上させる。

**機能の利用シーン**：
- ExcelのListObject（テーブル）データの行単位処理
- Workbook/Worksheetコレクションの一括操作
- ツリー構造データの平坦化処理
- コールバック関数による動的データ生成（スパイラル点列、ランダムウォーク等）
- データのグループ化、集計、フィルタリング

**主要な処理内容**：
1. 多様なデータソースからの列挙子作成（CreateFromArray, CreateFromIEnumVariant, CreateFromListObject, CreateFromCallable等）
2. 関数型処理メソッド（Map, Filter, Reduce, ForEach）
3. ソート・ユニーク化・逆順処理（Sort, Unique, Reverse）
4. 集約処理（Max, Min, Sum, CountBy, GroupBy）
5. 検索・条件チェック（indexOf, includes, checkAll, checkAny, FindFirst）
6. データ変換（AsArray, AsCollection, AsDictionary, AsArray2D, Join）
7. 遅延評価によるメモリ効率化（Callableベースの列挙）

**関連システム・外部連携**：stdICallableインターフェースを実装したコールバックオブジェクト（stdLambda, stdCallback）と連携し、高階関数を実現する。Excel ListObjectとの統合機能を提供。

**権限による制御**：特になし。すべてのVBA実行環境で利用可能。Mac環境では一部機能（空配列生成）に制限あり。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないライブラリクラス |

## 機能種別

データ構造（イテレータ） / データ操作 / 関数型プログラミング支援

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| v | Variant(Array) | Yes | CreateFromArray時の配列 | IsArray = True |
| o | Object | Yes | CreateFromIEnumVariant時のオブジェクト | IEnumVARIANT実装 |
| lo | ListObject | Yes | CreateFromListObject時のテーブル | Nothing以外 |
| cb | stdICallable | Yes | CreateFromCallable時のコールバック | Nothing以外 |
| iMaxLength | Long | No | 最大長（デフォルト1000000） | 正の整数 |
| dictionary | Object | Yes | CreateFromDictionaryEx時の辞書 | Scripting.Dictionary |
| oRoot | Object | Yes | CreateFromTree時のルートノード | Nothing以外 |
| getChildren | stdICallable | Yes | CreateFromTree時の子取得コールバック | Nothing以外 |

### 入力データソース

- VBA配列（1次元・多次元）
- VBA Collection
- Excel ListObject（テーブル）
- IEnumVARIANTを実装するオブジェクト（Workbooks, Worksheets等）
- コールバック関数（動的データ生成用）
- Scripting.Dictionary
- ツリー構造オブジェクト

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Length | Long | 列挙子の要素数 |
| Item(i) | Variant | i番目の要素 |
| AsArray | Variant() | VBA配列形式のデータ |
| AsCollection | Collection | Collectionオブジェクト |
| AsDictionary | Object | Scripting.Dictionary |
| AsArray2D | Variant() | 2次元配列（ヘッダー付き） |

### 出力先

- stdEnumeratorオブジェクト自身（メソッドチェーン用）
- VBA配列（AsArray）
- Collection（AsCollection）
- Dictionary（AsDictionary）
- 文字列（Join）
- 2次元配列（AsArray2D）

## 処理フロー

### 処理シーケンス

```
1. 初期化（ファクトリメソッド）
   └─ CreateFromArray / CreateFromIEnumVariant / CreateFromCallable等
   └─ protInit: モード設定と内部配列への変換

2. イテレーション（NextItem）
   └─ 現在のインデックスをインクリメント
   └─ モードに応じたデータ取得
      ├─ FromArray/FromIEnumVariant: 内部配列から取得
      └─ FromCallable: コールバック実行で次要素取得

3. Map処理
   └─ 新規Collectionを作成
   └─ NextItemでループ
   └─ 各要素にコールバック適用 → Collection.Add
   └─ CreateFromIEnumVariantで新stdEnumerator返却

4. Filter処理
   └─ 新規Collectionを作成
   └─ NextItemでループ
   └─ コールバックがTrueの要素のみCollection.Add
   └─ CreateFromIEnumVariantで新stdEnumerator返却

5. Sort処理
   └─ SortStruct配列を構築
   └─ 挿入ソートで連結リスト構造を構築
   └─ 連結リストを辿ってCollectionに収集
   └─ CreateFromIEnumVariantで新stdEnumerator返却
```

### フローチャート

```mermaid
flowchart TD
    A[ファクトリメソッド呼び出し] --> B{データソース種別}
    B -->|Array| C[CreateFromArray]
    B -->|Object| D[CreateFromIEnumVariant]
    B -->|ListObject| E[CreateFromListObject]
    B -->|Callable| F[CreateFromCallable]

    C --> G[protInit]
    D --> G
    E --> H[行をDictionaryに変換]
    H --> I[Collectionに追加]
    I --> G
    F --> G

    G --> J[内部状態設定]
    J --> K[stdEnumerator返却]

    K --> L{操作メソッド}
    L -->|Map| M[新Collection作成]
    M --> N[NextItemループ]
    N --> O[cb.Run実行]
    O --> P[Collection.Add]
    P --> Q{次要素?}
    Q -->|Yes| N
    Q -->|No| R[新stdEnumerator返却]

    L -->|Filter| S[新Collection作成]
    S --> T[NextItemループ]
    T --> U[cb.Run実行]
    U --> V{結果=True?}
    V -->|Yes| W[Collection.Add]
    V -->|No| X[スキップ]
    W --> Y{次要素?}
    X --> Y
    Y -->|Yes| T
    Y -->|No| R
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | インデックス基準 | 配列のインデックスは1から始まる（1-based） | AsArray, Item |
| BR-002 | 最大長制限 | iMaxLengthを超える要素数はエラー | Length計算時 |
| BR-003 | 不変性 | Map, Filter等は新しいstdEnumeratorを返却 | Map, Filter, Sort, Unique, Reverse |
| BR-004 | 遅延評価 | FromCallableモードでは要素取得時にコールバック実行 | CreateFromCallable |
| BR-005 | Nullによる終端 | コールバックがNullを返すと列挙終了 | FromCallableモード |
| BR-006 | ListObject変換 | ListObjectの各行はDictionaryに変換される | CreateFromListObject |

### 計算ロジック

**Length計算**:
```
FromArray/FromIEnumVariant: UBound(BaseArray) - LBound(BaseArray) + 1
FromCallable: NextItemがFalseを返すまでカウント
```

**ソート（挿入ソート+連結リスト）**:
```
各要素について:
  1. SortStructに値とソート値を格納
  2. 既存の連結リストを走査
  3. ソート値の比較により挿入位置を決定
  4. iPrev/iNextで連結リストを更新
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本クラスはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | 最大長超過 | Length > iMaxLength | エラー発生、フリーズ防止 |
| 1 | インデックスエラー | 存在しないインデックス指定 | "No such item" エラー |
| 1 | 型変換エラー | AsArray2DでDictionary以外 | "Type conversion not defined" |
| 1 | Callableエラー | Verbose戻り値不正 | 配列形式エラー |
| 1 | Mac非対応 | 非対応の空配列型 | "unsupported on Mac" |

### リトライ仕様

リトライ仕様なし（同期処理のみ）

## トランザクション仕様

トランザクション管理なし（メモリ内操作のみ）

## パフォーマンス要件

- Map/Filter操作: O(n)
- Sort操作: O(n²)（挿入ソート）
- indexOf/includes操作: O(n)
- Flatten操作: O(n*m) （ネストの深さに依存）
- Length計算: O(1) FromArray、O(n) FromCallable

## セキュリティ考慮事項

- 外部入力の検証は呼び出し側の責任
- コールバック実行時の例外は呼び出し側で処理
- iMaxLengthによる無限ループ防止機構

## 備考

- VB_PredeclaredId = True により、stdEnumeratorをインスタンス化せずにファクトリメソッドを呼び出し可能
- For Each構文に対応（NewEnumプロパティ）
- Ruby EnumerableモジュールにインスパイアされたAPI設計
- Mac環境では空配列生成に制限あり（SafeArrayCreateVector API非対応）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、stdEnumeratorの内部データ構造（TThis型、EnumeratorType）を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdEnumerator.cls | `src/stdEnumerator.cls` | 219-224行目: EnumeratorType enum定義 |
| 1-2 | stdEnumerator.cls | `src/stdEnumerator.cls` | 226-232行目: SortStruct型定義 |
| 1-3 | stdEnumerator.cls | `src/stdEnumerator.cls` | 234-249行目: TCurrentIteration, TThis型定義 |

**読解のコツ**:
- `EnumeratorType`がデータソースの種別を示す（FromCallable, FromIEnumVariant, FromArray）
- `TThis`型がクラスの内部状態を保持
- `BaseArray`がFromArray/FromIEnumVariantモードでの実データ格納先
- `Callback`がFromCallableモードでのデータ生成関数

#### Step 2: ファクトリメソッドを理解する

様々なデータソースからstdEnumeratorを生成するファクトリメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdEnumerator.cls | `src/stdEnumerator.cls` | 263-266行目: CreateFromCallable |
| 2-2 | stdEnumerator.cls | `src/stdEnumerator.cls` | 284-287行目: CreateFromArray |
| 2-3 | stdEnumerator.cls | `src/stdEnumerator.cls` | 294-297行目: CreateFromIEnumVariant |
| 2-4 | stdEnumerator.cls | `src/stdEnumerator.cls` | 314-350行目: CreateFromListObject |
| 2-5 | stdEnumerator.cls | `src/stdEnumerator.cls` | 415-456行目: protInit初期化処理 |

**主要処理フロー**:
1. **263行目**: `CreateFromCallable` - コールバック関数による動的生成
2. **284行目**: `CreateFromArray` - VBA配列からの生成
3. **294行目**: `CreateFromIEnumVariant` - For Each対応オブジェクトからの生成
4. **314-350行目**: `CreateFromListObject` - Excel テーブルからの生成（各行をDictionaryに変換）
5. **415-456行目**: `protInit` - モードに応じた内部配列への変換

#### Step 3: イテレーション機構を理解する

NextItem関数による統一的なイテレーション機構を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdEnumerator.cls | `src/stdEnumerator.cls` | 1396-1448行目: NextItem関数 |
| 3-2 | stdEnumerator.cls | `src/stdEnumerator.cls` | 487-500行目: enumNext関数 |
| 3-3 | stdEnumerator.cls | `src/stdEnumerator.cls` | 503-509行目: enumRefresh関数 |

**主要処理フロー**:
- **1396行目**: NextItem - モードに応じた次要素取得
- **1399-1416行目**: FromArray/FromIEnumVariantモードでの配列アクセス
- **1417-1421行目**: FromCallableモードでのコールバック実行
- **1422-1446行目**: FromCallableVerboseモードでの詳細戻り値処理

#### Step 4: 高階関数を理解する

Map, Filter, Reduceなどの関数型プログラミングメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdEnumerator.cls | `src/stdEnumerator.cls` | 682-692行目: Mapメソッド |
| 4-2 | stdEnumerator.cls | `src/stdEnumerator.cls` | 697-708行目: Filterメソッド |
| 4-3 | stdEnumerator.cls | `src/stdEnumerator.cls` | 1034-1053行目: Reduceメソッド |
| 4-4 | stdEnumerator.cls | `src/stdEnumerator.cls` | 669-677行目: ForEachメソッド |

**主要処理フロー**:
- **682-692行目**: Map - 新Collectionを作成し、各要素にコールバック適用
- **697-708行目**: Filter - 条件に合致する要素のみCollectionに追加
- **1034-1053行目**: Reduce - アキュムレータを使った集約処理

#### Step 5: ソートアルゴリズムを理解する

挿入ソート+連結リストによるソート処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | stdEnumerator.cls | `src/stdEnumerator.cls` | 713-807行目: Sortメソッド |

**主要処理フロー**:
- **715-790行目**: SortStruct配列構築と連結リスト形成
- **736-778行目**: 挿入位置の探索とリンク更新
- **793-800行目**: 連結リストを辿ってCollection収集

### プログラム呼び出し階層図

```
stdEnumerator.CreateFromArray(arr)
    │
    └─ stdEnumerator.protInit(FromArray, iMaxLength, arr)
           └─ This.BaseArray = arr

stdEnumerator.CreateFromIEnumVariant(obj)
    │
    └─ stdEnumerator.protInit(FromIEnumVariant, iMaxLength, obj)
           └─ For Each → CopyVariant → BaseArray構築

stdEnumerator.CreateFromListObject(lo)
    │
    ├─ HeaderRowRange.Value取得
    ├─ DataBodyRange.Value取得
    └─ For Each Row
           ├─ CreateObject("Scripting.Dictionary")
           ├─ 各カラム値を設定
           └─ Collection.Add(row)

stdEnumerator.Map(cb)
    │
    ├─ New Collection
    │
    └─ While NextItem(kv, v, i)
           ├─ cb.Run(v, kv)
           └─ Collection.Add(result)

stdEnumerator.Filter(cb)
    │
    ├─ New Collection
    │
    └─ While NextItem(kv, v, i)
           ├─ cb.Run(v, kv)
           └─ If True: Collection.Add(v)

stdEnumerator.Sort(cb)
    │
    ├─ SortStruct配列作成
    │
    ├─ While NextItem
    │      ├─ sortValue計算
    │      └─ 挿入位置探索 & リンク更新
    │
    └─ 連結リスト収集 → CreateFromIEnumVariant
```

### データフロー図

```
[入力]                           [処理]                           [出力]

Array ─────────────────▶ CreateFromArray ──────────────────▶ stdEnumerator
                                │                                (FromArray)
                                └─▶ BaseArray = arr

IEnumVariant ──────────▶ CreateFromIEnumVariant ──────────▶ stdEnumerator
(Collection等)                  │                                (FromIEnumVariant)
                                └─▶ For Each → BaseArray構築

ListObject ────────────▶ CreateFromListObject ────────────▶ stdEnumerator
                                │                                (FromIEnumVariant)
                                ├─▶ Header取得
                                ├─▶ DataBody取得
                                └─▶ Dictionary化 → Collection

stdEnumerator ─────────▶ Map(lambda) ─────────────────────▶ 新stdEnumerator
                                │
                                └─▶ NextItem → cb.Run → Collection.Add

stdEnumerator ─────────▶ Filter(lambda) ──────────────────▶ 新stdEnumerator
                                │
                                └─▶ NextItem → cb.Run → 条件True時Add

stdEnumerator ─────────▶ AsArray() ───────────────────────▶ Variant()
                                │
                                └─▶ NextItem → vRet(i) = v
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdEnumerator.cls | `src/stdEnumerator.cls` | ソース | イテレータ・列挙子クラスの実装 |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース |
| stdLambda.cls | `src/stdLambda.cls` | ソース | ラムダ式生成クラス（Map等で使用） |
| stdCallback.cls | `src/stdCallback.cls` | ソース | コールバックオブジェクト生成クラス |
