# 機能設計書 20-stdSentry

## 概要

本ドキュメントは、stdVBAライブラリのセンチネルパターン実装クラス「stdSentry」の機能設計を記載する。stdSentryは、RAII（Resource Acquisition Is Initialization）パターンをVBAで実現するためのクラスであり、リソースの確実な解放や設定の確実な復元を保証する。

### 本機能の処理概要

stdSentryは、stdICallableインターフェースを活用してセンチネル（見張り）パターンを汎用的に実装するVBAクラスモジュールである。

**業務上の目的・背景**：VBAにおけるリソース管理やエラーハンドリングを確実にする必要があるケースに対応する。With...End Withブロックのスコープを利用して、ブロック終了時に自動的にクリーンアップ処理を実行することで、リソースリークや設定の戻し忘れを防止する。

**機能の利用シーン**：
- アプリケーション設定の一時的な変更と確実な復元
- エラースタックのプッシュ/ポップ管理
- ファイルハンドルやロックの確実な解放
- 任意の「前処理→後処理」パターンの実装

**主要な処理内容**：
1. 汎用センチネルファクトリの作成（Create）
2. パフォーマンス最適化センチネル（CreateOptimiser）
3. オブジェクトプロパティセンチネル（CreateFromObjectProperty）
4. オブジェクトメソッドセンチネル（CreateFromObjectMethod）
5. エラースタックセンチネル（CreateErrorStack）
6. AutomationSecurityセンチネル（CreateFromAutomationSecurity）

**関連システム・外部連携**：stdICallableインターフェース、stdLambda、stdCallback、stdError

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | なし | - | VBAライブラリクラスのため画面連携なし |

## 機能種別

ユーティリティ機能 / デザインパターン / センチネルパターン

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| OnInit | stdICallable | Yes | 初期化時に実行するコールバック | stdICallable実装 |
| OnDestroy | stdICallable | Yes | 終了時に実行するコールバック | stdICallable実装 |
| passVars | Boolean | No | OnInitの結果をOnDestroyに渡すか（既定: True） | Boolean |
| obj | Object | Yes | 対象オブジェクト（CreateFromObject*） | 任意のオブジェクト |
| sPropertyName | String | Yes | プロパティ名（CreateFromObjectProperty） | 有効なプロパティ名 |

### CreateOptimiserパラメータ

| パラメータ名 | 型 | 既定値 | 説明 |
|-------------|-----|-------|------|
| EnableEvents | vbTriState | vbUseDefault | イベント発火設定 |
| ScreenUpdating | vbTriState | vbUseDefault | 画面更新設定 |
| Calculation | Long | -1 | 計算モード（-1は変更なし） |

### 入力データソース

- stdICallable実装オブジェクト（stdLambda、stdCallback等）
- 対象オブジェクトとプロパティ/メソッド名
- 設定値

## 出力仕様

### 出力データ

センチネルオブジェクト自体が出力。With...End Withブロックで使用し、ブロック終了時に自動的にOnDestroyが実行される。

### 出力先

- stdICallableを通じて任意の処理を実行

## 処理フロー

### 処理シーケンス

```
1. ファクトリ作成（Create）
   └─ OnInit/OnDestroy/passVarsを保持

2. センチネル実行（Run/RunEx）
   ├─ OnInit実行 → 結果（destroyArgs）取得
   └─ 新しいstdSentryインスタンス作成
      └─ protInitInstance(OnDestroy, destroyArgs, passVars)

3. ブロック終了（Class_Terminate）
   └─ passVars=Trueなら OnDestroy.RunEx(destroyArgs)
      passVars=Falseなら OnDestroy.Run()
```

### フローチャート

```mermaid
flowchart TD
    A[Create] --> B[protInitFactory]
    B --> C[OnInit/OnDestroy保存]
    C --> D[ファクトリオブジェクト]

    E[Run/RunEx] --> F[OnInit.RunEx]
    F --> G[destroyArgs取得]
    G --> H[new stdSentry]
    H --> I[protInitInstance]
    I --> J[センチネルオブジェクト]

    K[Withブロック終了] --> L[Class_Terminate]
    L --> M{passVars?}
    M -->|True| N[OnDestroy.RunEx destroyArgs]
    M -->|False| O[OnDestroy.Run]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ファクトリパターン | Createでファクトリ作成、Run/RunExでインスタンス作成 | 汎用センチネル |
| BR-02 | RAII保証 | Class_Terminateでクリーンアップ実行 | 全センチネル |
| BR-03 | 変数引き渡し | OnInit結果をOnDestroyに渡す（passVars=True時） | 汎用センチネル |
| BR-04 | stdICallable実装 | stdSentry自体もstdICallableを実装 | インターフェース |

### 便利メソッドの用途

| メソッド | 用途 | 必要ライブラリ |
|---------|------|---------------|
| CreateOptimiser | アプリケーション設定の最適化と復元 | stdLambda |
| CreateFromObjectProperty | オブジェクトプロパティの一時変更と復元 | stdLambda |
| CreateFromObjectMethod | オブジェクトメソッドの開始/終了ペア呼び出し | stdCallback |
| CreateErrorStack | エラースタックの管理 | stdError, stdCallback |
| CreateFromAutomationSecurity | AutomationSecurity設定の変更と復元 | stdLambda |

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | なし | - | データベース操作なし |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 5 | stdLambda未参照 | CreateOptimiser等でstdLambdaが未定義 | stdLambdaをインポート |
| 5 | stdCallback未参照 | CreateFromObjectMethodでstdCallbackが未定義 | stdCallbackをインポート |
| 5 | stdError未参照 | CreateErrorStackでstdErrorが未定義 | stdErrorをインポート |
| 5 | 不正なCalculation | xlCalculation以外の値指定 | 正しい値を使用 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし（メモリ内操作のみ）

## パフォーマンス要件

- stdICallable呼び出しのオーバーヘッドは最小限
- stdLambda/stdCallbackの生成コストに注意

## セキュリティ考慮事項

- OnInit/OnDestroyで任意のコードが実行されるため、信頼できるコールバックのみを使用
- AutomationSecurity変更は一時的でも慎重に

## 備考

- stdICallableインターフェースを実装しており、Run/Bindが利用可能
- PredeclaredId属性により、stdSentry.CreateOptimiserのように直接アクセス可能
- SendMessageメソッドでメタ情報（OnInit、OnDestroy等）にアクセス可能
- VB_UserMemId = 0属性によりRun()が既定メンバー

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

内部で使用される型定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdSentry.cls | `src/stdSentry.cls` 86-90行目 | TFactory型（ファクトリ用） |
| 1-2 | stdSentry.cls | `src/stdSentry.cls` 91-95行目 | TInstance型（インスタンス用） |
| 1-3 | stdSentry.cls | `src/stdSentry.cls` 96-100行目 | TThis型（全状態管理） |

**読解のコツ**: ファクトリとインスタンスで役割が異なる。ファクトリはOnInit/OnDestroyを保持、インスタンスはOnDestroyとdestroyArgsを保持。

#### Step 2: ファクトリ作成を理解する

Create関数とprotInitFactoryを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdSentry.cls | `src/stdSentry.cls` 118-121行目 | Create関数 |
| 2-2 | stdSentry.cls | `src/stdSentry.cls` 245-251行目 | protInitFactory関数 |

**主要処理フロー**:
- **118-121行目**: Create - new stdSentry → protInitFactory
- **245-251行目**: protInitFactory - This.Factory.OnInit/OnDestroy/passVars設定

#### Step 3: 便利メソッドを理解する

各種CreateFrom*メソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdSentry.cls | `src/stdSentry.cls` 135-163行目 | CreateOptimiser関数 |
| 3-2 | stdSentry.cls | `src/stdSentry.cls` 177-189行目 | CreateFromObjectProperty関数 |
| 3-3 | stdSentry.cls | `src/stdSentry.cls` 203-208行目 | CreateFromObjectMethod関数 |
| 3-4 | stdSentry.cls | `src/stdSentry.cls` 222-225行目 | CreateErrorStack関数 |
| 3-5 | stdSentry.cls | `src/stdSentry.cls` 237-239行目 | CreateFromAutomationSecurity関数 |

**主要処理フロー**:
- **135-163行目**: CreateOptimiser - stdLambdaでOnInit/OnDestroy作成 → Create → Run
- **177-189行目**: CreateFromObjectProperty - プロパティGet/Set用Lambda作成
- **203-208行目**: CreateFromObjectMethod - メソッド呼び出しCallback作成

#### Step 4: センチネル実行を理解する

Run/RunExメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdSentry.cls | `src/stdSentry.cls` 269-273行目 | Run関数 |
| 4-2 | stdSentry.cls | `src/stdSentry.cls` 278-286行目 | RunEx関数 |

**主要処理フロー**:
- **269-273行目**: Run - ParamArrayをコピーしてRunExに委譲
- **278-286行目**: RunEx - OnInit実行 → destroyArgs取得 → new stdSentry → protInitInstance

#### Step 5: クリーンアップを理解する

Class_Terminateでのクリーンアップ処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | stdSentry.cls | `src/stdSentry.cls` 253-264行目 | protInitInstance関数 |
| 5-2 | stdSentry.cls | `src/stdSentry.cls` 331-341行目 | Class_Terminate |

**主要処理フロー**:
- **253-264行目**: protInitInstance - OnDestroy/DestroyArgs/passVarsをインスタンスに設定
- **331-341行目**: Class_Terminate - passVarsに応じてOnDestroy.RunEx/Runを呼び出し

### プログラム呼び出し階層図

```
stdSentry.Create(OnInit, OnDestroy, passVars)
    │
    └─ protInitFactory(OnInit, OnDestroy, passVars)
           │
           └─ This.Factory設定
                  ├─ .OnInit = OnInit
                  ├─ .OnDestroy = OnDestroy
                  └─ .passVars = passVars

stdSentry.Run(args...)
    │
    └─ RunEx(args)
           │
           ├─ This.Factory.OnInit.RunEx(args)    [stdICallable]
           │      └─ destroyArgs取得
           │
           └─ new stdSentry
                  │
                  └─ protInitInstance(OnDestroy, destroyArgs, passVars)
                         │
                         └─ This.Instance設定

[Withブロック終了]
    │
    └─ Class_Terminate
           │
           ├─ passVars=True の場合
           │      └─ OnDestroy.RunEx(destroyArgs)
           │
           └─ passVars=False の場合
                  └─ OnDestroy.Run

stdSentry.CreateOptimiser(EnableEvents, ScreenUpdating, Calculation)
    │
    ├─ stdLambda.CreateMultiLine で OnInit作成
    │      └─ 現在設定保存 → 新設定適用 → 元設定を返却
    │
    ├─ stdLambda.CreateMultiLine で OnDestroy作成
    │      └─ 元設定を復元
    │
    └─ Create(OnInit, OnDestroy).Run(EnableEvents, ScreenUpdating, Calculation)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

OnInit/OnDestroy ────────▶ Create()
     │                         │
     │                         ▼
     │                    protInitFactory
     │                         │
     └─────────────────────────┼───────────────────────▶ ファクトリオブジェクト
                               │
                               ▼
                            Run(args)
                               │
                               ├─ OnInit.RunEx(args)
                               │      │
                               │      └─ destroyArgs取得
                               │
                               ├─ new stdSentry
                               │
                               └─ protInitInstance
                                      │
                                      └─────────────────▶ センチネルオブジェクト
                                                               │
                                                               ▼
                                                      [Withブロック内のコード]
                                                               │
                                                               ▼
                                                      Class_Terminate
                                                               │
                                                               └─ OnDestroy実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdSentry.cls | `src/stdSentry.cls` | ソース | センチネルパターン実装クラス本体 |
| stdICallable.cls | `src/stdICallable.cls` | インターフェース | コールバックインターフェース |
| stdLambda.cls | `src/stdLambda.cls` | ソース | CreateOptimiser等で使用 |
| stdCallback.cls | `src/stdCallback.cls` | ソース | CreateFromObjectMethod等で使用 |
| stdError.cls | `src/stdError.cls` | ソース | CreateErrorStackで使用 |
