# 機能設計書 23-stdUIElement

## 概要

本ドキュメントは、stdVBAライブラリにおける`stdUIElement`クラスの機能設計を記述する。stdUIElementは、MSForms（UserForm）コントロールのラッパークラスであり、イベントルーティングと統一的なコントロール操作インターフェースを提供する。

### 本機能の処理概要

stdUIElementは、VBAのUserFormで使用される各種コントロール（TextBox、ComboBox、CommandButton等）を統一的に扱うためのラッパークラスである。WithEventsを活用して各コントロールのイベントを受信し、stdICallableコールバックへルーティングすることで、動的なイベント処理とコード再利用を実現する。

**業務上の目的・背景**：VBAのUserFormでは、各コントロールのイベント処理を個別にコーディングする必要があり、コードの重複や保守性の低下を招きやすい。stdUIElementは、イベント処理の一元化、コントロールの動的生成と操作、統一的なプロパティアクセスを提供し、UserFormアプリケーションの開発効率と保守性を向上させる。

**機能の利用シーン**：カスタムUserFormの動的構築、イベント駆動型のUI操作、フォームのコントロール一括管理、MVVMやMVCパターンの実装補助、テスト可能なUIロジックの分離などの場面で利用される。

**主要な処理内容**：
1. 既存コントロールからのstdUIElementラッパー作成
2. コントロールタイプを指定した新規コントロール作成
3. 各種イベント（Click、Change、KeyDown等）のコールバックへのルーティング
4. Value、Caption、位置・サイズプロパティの統一的なアクセス
5. コントロールタイプに応じた適切なプロパティ操作
6. GUIDを用いた一意なコントロール識別

**関連システム・外部連携**：stdICallableインターフェースを実装したコールバック（stdLambda、stdCallback）と連携。MSForms（Microsoft Forms 2.0 Object Library）のコントロール群をラップ。

**権限による制御**：特別な権限制御は不要。UserFormの標準的な使用権限のみ必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | （画面なし） | 該当なし | 本機能はVBAライブラリであり、UIを持つ画面は存在しない。ただし、UserFormの構築に使用される |

## 機能種別

UI操作 / イベントルーティング / コントロールラッパー

## 入力仕様

### 入力パラメータ

#### CreateFromControl関数
| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ctrl | Control | Yes | ラップ対象のMSFormsコントロール | - |
| callback | stdICallable | Yes | イベント受信コールバック | (el, event, params)=>void |

#### CreateFromType関数
| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| oControls | MSForms.Controls | Yes | コントロールを追加するコレクション | - |
| iCtrlType | EUIElementType | Yes | 作成するコントロールの種別 | Enum値 |
| sName | String | No | コントロール名 | - |
| Caption | String | No | 表示テキスト | - |
| Value | Variant | No | 初期値 | - |
| callback | stdICallable | No | イベント受信コールバック | - |
| fLeft | Double | No | 左位置（デフォルト: 0） | - |
| fTop | Double | No | 上位置（デフォルト: 0） | - |
| fWidth | Double | No | 幅 | - |
| fHeight | Double | No | 高さ | - |

### 入力データソース

- プログラムからのパラメータ指定
- 既存のMSFormsコントロール

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| uiType | EUIElementType | コントロールの種別 |
| Control | Control | ラップされた元のコントロール |
| uiObject | Object | イベントシンク（WithEvents変数） |
| Value | Variant | コントロールの値 |
| Caption | String | 表示テキスト |
| Left | Double | 左位置 |
| Top | Double | 上位置 |
| Width | Double | 幅 |
| Height | Double | 高さ |
| id | String | 一意のGUID |

### 出力先

- コールバック関数への各種イベント通知
- プロパティ経由での値取得

## 処理フロー

### 処理シーケンス

```
1. コントロールラッパーの作成
   ├─ CreateFromControl: 既存コントロールをラップ
   └─ CreateFromType: 新規コントロールを作成してラップ

2. protInit（初期化処理）
   ├─ messageRouter（コールバック）の保存
   ├─ コントロール参照の保存
   ├─ GUIDの生成
   └─ コントロールタイプの判定とWithEvents変数への設定

3. イベント発生時
   ├─ 対応するWithEvents変数のイベントハンドラが呼び出される
   └─ callEventCBを通じてmessageRouterのRunを実行

4. プロパティアクセス時
   ├─ コントロールタイプに応じた適切なプロパティを操作
   └─ 必要に応じてChangeイベントを発火
```

### フローチャート

```mermaid
flowchart TD
    A[CreateFromControl/CreateFromType] --> B[protInit]
    B --> C[コールバック保存]
    C --> D[GUIDを生成]
    D --> E{コントロールタイプ判定}
    E -->|UserForm| F1[pUserFormに設定]
    E -->|Frame| F2[pFrameに設定]
    E -->|TextBox| F3[pTextBoxに設定]
    E -->|...| F4[対応する変数に設定]
    F1 --> G[イベント待機]
    F2 --> G
    F3 --> G
    F4 --> G
    G --> H{イベント発生?}
    H -->|Yes| I[イベントハンドラ呼び出し]
    I --> J[callEventCB]
    J --> K{コールバック登録済?}
    K -->|Yes| L[messageRouter.Run実行]
    K -->|No| M[何もしない]
    L --> G
    M --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-23-01 | コントロールタイプ自動判定 | TypeOf演算子で実行時にコントロールタイプを判定 | protInit時 |
| BR-23-02 | イベント統一化 | 全イベントをEUIElementEvent列挙体で統一 | イベント発生時 |
| BR-23-03 | コールバック呼び出し形式 | (el, event, params)の3引数形式で呼び出し | callEventCB |
| BR-23-04 | GUID自動生成 | 各インスタンスに一意のUUID v4を生成 | インスタンス作成時 |
| BR-23-05 | Captionサポート | UserForm, Frame, Label, CheckBox, OptionButton, ToggleButton, CommandButtonのみ | Caption Get/Let |
| BR-23-06 | Valueサポート | TextBox, ComboBox, ListBox, CheckBox, OptionButton, ToggleButton, ScrollBar, SpinButton, ImageのみValue対応 | Value Get/Let/Set |

### 計算ロジック

CheckBoxのLeft位置補正：
```vba
uiObject.Left = this.Left + IIf(this.Width <= 12, 0, this.Width / 2 - 6)
```

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 型不一致 | サポート外のプロパティアクセス | Select Caseで対応コントロールのみ処理 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし（本機能はトランザクション制御を行わない）

## パフォーマンス要件

- イベント発生ごとにコールバック呼び出しが行われるため、コールバック処理は軽量にすること
- MouseMoveイベントは高頻度で発生するため、必要に応じてフィルタリングを検討

## セキュリティ考慮事項

- stdUIElementはUserForm内のコントロールをラップするため、通常のUserFormセキュリティ制約に準拠
- コールバックにbindGlobalでグローバル変数を注入する機能があるため、適切なスコープ管理が必要

## 備考

- サポートされるコントロールタイプ（EUIElementType）：
  - グループコントロール: UserForm, Frame, MultiPage
  - 表示コントロール: Label, Image, TabStrip
  - 編集コントロール: TextBox, ComboBox, ListBox, CheckBox, OptionButton, ToggleButton, ScrollBar, SpinButton
  - 実行コントロール: CommandButton
- 一部コントロール（ScrollBar, SpinButton）ではClickイベントがサポートされていない
- ImageとTextBoxではMouseDownイベントでMouseClickを補完

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

列挙体と内部状態の型定義を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdUIElement.cls | `src/stdUIElement.cls` | EUIElementEvent（12-27行目）でサポートされるイベント一覧を確認 |
| 1-2 | stdUIElement.cls | `src/stdUIElement.cls` | EUIElementType（28-51行目）でサポートされるコントロール種別を確認 |
| 1-3 | stdUIElement.cls | `src/stdUIElement.cls` | TThis（52-59行目）で内部状態を確認 |

**読解のコツ**:
- `EUIElementEvent`は発火可能なイベントの列挙体
- `EUIElementType`はラップ可能なコントロールの列挙体
- `TThis`にはmessageRouter（コールバック）、Control、iType、id等が格納

#### Step 2: WithEvents変数を理解する

イベント受信用のWithEvents変数群を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdUIElement.cls | `src/stdUIElement.cls` | WithEvents変数宣言（62-76行目）でイベントシンクを理解 |

**読解のコツ**:
- 各コントロールタイプに対応するWithEvents変数が宣言されている
- VBAではWithEventsにObject型を使用できないため、具体的な型で宣言が必要

#### Step 3: エントリーポイントを理解する

コントロールラッパー作成のエントリーポイントを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdUIElement.cls | `src/stdUIElement.cls` | CreateFromControl（89-93行目）で既存コントロールをラップ |
| 3-2 | stdUIElement.cls | `src/stdUIElement.cls` | CreateFromType（107-142行目）で新規コントロールを作成してラップ |

**主要処理フロー**:
- **89-93行目**: 既存コントロールからラッパーを生成
- **107-142行目**: Controlsコレクションに新規コントロールを追加し、プロパティを設定

#### Step 4: 初期化処理を理解する

protInitによるコントロールタイプ判定とWithEvents変数への設定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdUIElement.cls | `src/stdUIElement.cls` | protInit（281-360行目）でコントロールタイプを判定 |

**主要処理フロー**:
- **281-284行目**: messageRouterとControlの保存、GUIDの生成
- **285-359行目**: TypeOf演算子でコントロールタイプを判定し、対応するWithEvents変数に設定

#### Step 5: イベントハンドラを理解する

各種イベントハンドラの実装パターンを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | stdUIElement.cls | `src/stdUIElement.cls` | Clickイベントハンドラ（423-439行目） |
| 5-2 | stdUIElement.cls | `src/stdUIElement.cls` | Changeイベントハンドラ（448-457行目） |
| 5-3 | stdUIElement.cls | `src/stdUIElement.cls` | KeyDownイベントハンドラ（463-475行目） |
| 5-4 | stdUIElement.cls | `src/stdUIElement.cls` | MouseDownイベントハンドラ（509-527行目） |

**読解のコツ**:
- 各イベントハンドラは`callEventCB`を呼び出してコールバックにルーティング
- 引数にはMeの参照、イベント種別、追加パラメータを渡す

#### Step 6: コールバック呼び出しを理解する

イベントルーティングの中心となる関数を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 6-1 | stdUIElement.cls | `src/stdUIElement.cls` | callEventCB（573-578行目）でコールバックを呼び出し |
| 6-2 | stdUIElement.cls | `src/stdUIElement.cls` | BindGlobals（375-378行目）でグローバル変数を注入 |

**主要処理フロー**:
- **573-578行目**: messageRouterが設定されていれば、(MeRef, iMsg, args)でRunを実行
- **375-378行目**: EUIElementEventの名前→値マッピングをコールバックに注入

### プログラム呼び出し階層図

```
CreateFromControl / CreateFromType
    │
    ├─ BindGlobals (コールバックへの定数注入)
    │      └─ query.SendMessage("bindGlobal", ...)
    │
    └─ protInit
           ├─ getGUID (UUID v4生成)
           │
           └─ TypeOf判定
                  └─ pTextBox / pComboBox / ... に設定

イベント発生時:
    │
    pTextBox_Change / pComboBox_Click / ...
           │
           └─ callEventCB
                  │
                  └─ this.messageRouter.Run(MeRef, iMsg, args)
```

### データフロー図

```
[入力]                         [処理]                          [出力]

Control ──────────────▶ protInit ──────────▶ WithEvents変数設定
        │
callback ─────────────▶ BindGlobals ───────▶ EUIElementEvent注入
        │
                        イベント発生
                            │
                        イベントハンドラ
                            │
                        callEventCB
                            │
                        messageRouter.Run ──▶ (el, event, params)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdUIElement.cls | `src/stdUIElement.cls` | ソース | UIElementクラス本体 |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース |
| stdCallback.cls | `src/stdCallback.cls` | ソース | コールバック実装（イベントハンドラに使用） |
| stdLambda.cls | `src/stdLambda.cls` | ソース | ラムダ関数実装（イベントハンドラに使用） |
