# 機能設計書 3-stdLambda

## 概要

本ドキュメントは、stdVBAライブラリにおけるラムダ関数生成クラス `stdLambda` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdLambdaは、VBAで文字列式からラムダ関数（無名関数）を動的に生成し、実行するクラスである。JavaScriptやPythonのようなモダンな言語における無名関数やアロー関数に相当する機能を、VBAの文字列として記述し、stdICallableインターフェースとして扱うことを可能にする。

**業務上の目的・背景**：VBAには本来、関数オブジェクトやラムダ式の概念が存在しない。そのため、コールバックを必要とする処理（Map、Filter、Sort等）では、都度個別の関数を定義する必要があり、コードの冗長性が高まる。stdLambdaはこの課題を解決し、インラインで関数ロジックを定義することで、より簡潔で読みやすいコードの記述を可能にする。

**機能の利用シーン**：
- stdArray/stdEnumeratorのMap、Filter、Reduce等での変換・抽出ロジック指定
- stdWindowでのウィンドウ検索条件指定
- 動的な計算式の評価
- データ変換パイプラインの構築
- 条件式の動的生成と評価

**主要な処理内容**：
1. 文字列式のトークン化（Tokenise）
2. トップダウン式パーサーによる構文解析（parseBlock, parseExpression等）
3. スタックベース仮想マシンによる評価（evaluate）
4. 引数バインディング（Bind, BindEx）
5. グローバル変数のバインディング（BindGlobal）
6. パフォーマンスキャッシュによる最適化
7. 制御フロー構文のサポート（if-then-else, for, do-loop）

**関連システム・外部連携**：stdICallableインターフェースを実装し、stdArray、stdEnumerator、stdWindow等の高階関数メソッドと連携する。

**権限による制御**：特になし。bSandboxExtrasパラメータにより、追加関数の呼び出しを制限可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないライブラリクラス |

## 機能種別

式評価エンジン / ラムダ関数生成 / 関数型プログラミング支援

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| sEquation | String | Yes | ラムダ式（文字列） | 構文的に正しい式 |
| bUsePerformanceCache | Boolean | No | パフォーマンスキャッシュ使用（デフォルトFalse） | - |
| bSandboxExtras | Boolean | No | 追加関数のサンドボックス化（デフォルトFalse） | - |
| params | ParamArray(Variant) | No | Run時の引数 | - |

### 入力データソース

- 文字列式（ラムダ式構文）
- Run時のパラメータ（$1, $2, ...で参照）
- BindGlobalで登録されたグローバル変数

### サポートされる構文

**演算子**:
- 算術: `+`, `-`, `*`, `/`, `^`, `mod`
- 比較: `=`, `<>`, `<`, `<=`, `>`, `>=`, `like`, `is`
- 論理: `and`, `or`, `not`, `xor`
- 文字列: `&`

**制御フロー**:
- `if ... then ... else ... end`
- `for ... to ... step ... next`
- `do ... until/while ... loop`

**その他**:
- `let/set variable = expression`
- `fun name(params) ... end`
- プロパティアクセス: `$1.Property`
- メソッド呼び出し: `$1.Method(args)`

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Run() | Variant | ラムダ式の評価結果 |
| Equation | String | 元の式文字列 |
| Bind() | stdLambda | 引数がバインドされた新しいラムダ |

### 出力先

- Runメソッドの戻り値（Variant型）
- stdICallableインターフェース経由での呼び出し結果

## 処理フロー

### 処理シーケンス

```
1. Create(式文字列)
   └─ キャッシュチェック → キャッシュヒット時は既存オブジェクト返却
   └─ キャッシュミス時は新規stdLambda生成 → protInit呼び出し

2. protInit
   └─ 式文字列をTokenise（字句解析）
   └─ parseBlock呼び出し（構文解析開始）
      └─ parseStatement → parseExpression
         └─ 優先度に従って再帰的にパース
         └─ 演算子・オペランドをOperation配列に追加
   └─ finishOperations（後処理）

3. Run(params)
   └─ isBoundLambda? → Yes: Bound.Lambda.RunExに委譲
   └─ No: evaluate(operations, params)呼び出し
      └─ スタックベース仮想マシンで評価
      └─ Jump Tableによる高速命令ディスパッチ
      └─ 結果をスタックトップから取得して返却

4. Bind(params)
   └─ 新規stdLambdaをBoundLambdaとして作成
   └─ 元のラムダと引数を保持
```

### フローチャート

```mermaid
flowchart TD
    A[Create呼び出し] --> B{キャッシュ存在?}
    B -->|Yes| C[キャッシュから返却]
    B -->|No| D[新規stdLambda生成]
    D --> E[protInit]
    E --> F[Tokenise]
    F --> G[parseBlock]
    G --> H[operations配列構築]
    H --> I[キャッシュに格納]
    I --> C

    J[Run呼び出し] --> K{isBoundLambda?}
    K -->|Yes| L[Bound.Lambda.RunEx]
    K -->|No| M[evaluate]
    M --> N[スタック初期化]
    N --> O[命令ループ]
    O --> P{命令種別}
    P -->|iPush| Q[スタックPush]
    P -->|iArithmetic_*| R[演算実行]
    P -->|iJump_*| S[ジャンプ処理]
    P -->|iFunc_Call| T[関数呼び出し]
    P -->|iReturn| U[結果返却]
    Q --> O
    R --> O
    S --> O
    T --> O
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 引数参照 | $1, $2, ...でRun時の引数を参照 | 式内 |
| BR-002 | キャッシング | 同一式はDictionaryでキャッシュ | Create時 |
| BR-003 | 演算子優先度 | VBAの標準優先度に従う | 式評価時 |
| BR-004 | 型推論 | 実行時の引数型に基づいて動的に型決定 | Run時 |
| BR-005 | サンドボックス | bSandboxExtras=Trueで追加関数を無効化 | Create時 |
| BR-006 | BoundLambda | Bind後のラムダは元ラムダへの参照を保持 | Bind時 |

### 演算子優先度（高→低）

```
1. ^ (べき乗)
2. 単項 +, - (符号)
3. *, / (乗除)
4. mod (剰余)
5. +, - (加減)
6. & (文字列結合)
7. =, <>, <, <=, >, >=, is, like (比較)
8. not (論理否定)
9. and (論理積)
10. or (論理和)
11. xor (排他的論理和)
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本クラスはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | 構文エラー | 不正な式構文 | Throwでエラー発生 |
| 1 | 引数不足 | $Nで存在しない引数参照 | "Argument N not supplied" |
| 1 | スコープエラー | 関数外からの変数アクセス | "Can't access variable" |
| 1 | 型エラー | 変数と関数の混同 | "Expected a function/variable" |
| 1 | 不明命令 | 未知の仮想マシン命令 | "Unknown instruction" |

### リトライ仕様

リトライ仕様なし（同期処理のみ）

## トランザクション仕様

トランザクション管理なし（メモリ内操作のみ）

## パフォーマンス要件

- Create: O(n) （式の長さに比例）
- Run: O(m) （操作数に比例）
- パフォーマンスキャッシュ有効時: O(1)での結果取得（キャッシュヒット時）
- Jump Tableによる命令ディスパッチで高速実行

## セキュリティ考慮事項

- bSandboxExtrasによる追加関数の制限
- 式文字列は信頼できるソースからのみ受け付けること
- rtcCallByNameによるオブジェクトメソッド呼び出しは慎重に使用

## 備考

- VB_PredeclaredId = True により、stdLambdaをインスタンス化せずにファクトリメソッドを呼び出し可能
- stdICallableインターフェースを実装し、他のstdVBAクラスと相互運用可能
- スタックベース仮想マシンとJump Tableによる効率的な評価
- Mac環境では一部API（rtcCallByName）に制限あり

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、stdLambdaの内部データ構造（TThis型、IInstruction enum）を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdLambda.cls | `src/stdLambda.cls` | 84-98行目: TokenDefinition, token, Operation型定義 |
| 1-2 | stdLambda.cls | `src/stdLambda.cls` | 100-159行目: IInstruction enum（仮想マシン命令セット） |
| 1-3 | stdLambda.cls | `src/stdLambda.cls` | 161-164行目: LambdaType enum |
| 1-4 | stdLambda.cls | `src/stdLambda.cls` | 171-200行目: TThis型定義 |

**読解のコツ**:
- `IInstruction`が仮想マシンの命令セットを定義
- `TThis`がクラスの内部状態を保持
- `operations()`が解析済みの命令配列
- `tokens()`が字句解析結果

#### Step 2: ファクトリメソッドを理解する

Createメソッドとキャッシング機構を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdLambda.cls | `src/stdLambda.cls` | 226-237行目: Createメソッド |
| 2-2 | stdLambda.cls | `src/stdLambda.cls` | 251-253行目: CreateMultilineメソッド |
| 2-3 | stdLambda.cls | `src/stdLambda.cls` | 259-296行目: protInit初期化処理 |

**主要処理フロー**:
1. **228行目**: キャッシュDictionary初期化
2. **230行目**: キャッシュキー生成とチェック
3. **231-232行目**: キャッシュミス時の新規オブジェクト生成
4. **275行目**: 式のトークン化（Tokenise呼び出し）
5. **281行目**: 構文解析開始（parseBlock呼び出し）

#### Step 3: 字句解析を理解する

Tokenise関数とトークン定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdLambda.cls | `src/stdLambda.cls` | 444-523行目: getTokenDefinitions |

**主要処理フロー**:
- **450-501行目**: 各トークンタイプの正規表現定義
- リテラル、演算子、キーワード、変数名等のパターン

#### Step 4: 構文解析を理解する

パーサーの再帰下降構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdLambda.cls | `src/stdLambda.cls` | 535-559行目: parseBlock |
| 4-2 | stdLambda.cls | `src/stdLambda.cls` | 574-582行目: parseStatement |
| 4-3 | stdLambda.cls | `src/stdLambda.cls` | 608-610行目: parseExpression |
| 4-4 | stdLambda.cls | `src/stdLambda.cls` | 673-705行目: parseComparisonPriority1 |
| 4-5 | stdLambda.cls | `src/stdLambda.cls` | 962-991行目: parseValuePriority1 |

**主要処理フロー**:
- **608-610行目**: parseExpression → parseLogicPriority1から開始
- **673-705行目**: 比較演算子のパース
- **962-991行目**: リテラル値、変数、括弧のパース

#### Step 5: 仮想マシンを理解する

evaluate関数によるスタックベース評価を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | stdLambda.cls | `src/stdLambda.cls` | 1290-1500行目: evaluate関数 |

**主要処理フロー**:
- **1291-1293行目**: スタック初期化
- **1320-1338行目**: メインループとJump Table
- **1342-1351行目**: iPush処理
- **1396-1401行目**: iArithmetic_Add処理

### プログラム呼び出し階層図

```
stdLambda.Create(sEquation)
    │
    ├─ キャッシュチェック (This.Singleton.Cache)
    │
    └─ stdLambda.protInit(iStandardLambda, sEquation, ...)
           │
           ├─ Tokenise(sEquation)
           │      └─ getTokenDefinitions()
           │      └─ 正規表現マッチングでトークン配列生成
           │
           ├─ parseBlock("eof")
           │      └─ parseStatement()
           │             └─ parseExpression()
           │                    └─ parseLogicPriority1() → ... → parseValuePriority1()
           │                           └─ addOperation(命令, 値, スタック変化)
           │
           └─ finishOperations()

stdLambda.Run(params)
    │
    └─ evaluate(This.operations, params)
           │
           ├─ スタック初期化
           │
           └─ While opIndex <= opCount
                  ├─ Jump Table による命令ディスパッチ
                  ├─ iPush → GoSub StackPush
                  ├─ iArithmetic_* → 演算実行 → StackPush
                  ├─ iJump_* → opIndex更新
                  └─ iReturn_WithValue → 結果返却

stdLambda.Bind(params)
    │
    └─ stdLambda.BindEx(params)
           └─ 新規stdLambda.protInit(iBoundLambda, Me, params)
```

### データフロー図

```
[入力]                      [処理]                          [出力]

式文字列 ─────────────▶ Tokenise ─────────────────────▶ tokens()
"$1 + $2"                   │
                            └─▶ getTokenDefinitions()
                                    │
                                    └─▶ 正規表現マッチング

tokens() ────────────▶ parseBlock ───────────────────▶ operations()
                            │
                            └─▶ parseExpression()
                                    │
                                    └─▶ addOperation()

operations() ────────▶ evaluate ─────────────────────▶ 結果(Variant)
+ params               │
                       ├─▶ スタック操作
                       └─▶ 命令実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdLambda.cls | `src/stdLambda.cls` | ソース | ラムダ関数生成クラスの実装 |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース |
| stdCallback.cls | `src/stdCallback.cls` | ソース | コールバックオブジェクト生成クラス |
| stdArray.cls | `src/stdArray.cls` | ソース | 動的配列クラス（stdLambdaを使用） |
| stdEnumerator.cls | `src/stdEnumerator.cls` | ソース | 列挙子クラス（stdLambdaを使用） |
