# 機能設計書 4-stdCallback

## 概要

本ドキュメントは、stdVBAライブラリにおけるコールバックオブジェクト生成クラス `stdCallback` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdCallbackは、VBAにおいて既存のモジュール関数、オブジェクトメソッド、または関数ポインタをファーストクラスオブジェクトとしてラップし、後から呼び出し可能なコールバックオブジェクトを生成するクラスである。stdLambdaが文字列から新しい関数を生成するのに対し、stdCallbackは既存の関数への参照を保持する。

**業務上の目的・背景**：VBAでは、関数を変数に格納したり、引数として渡したりすることができない。AddressOfは限定的な場面でしか使用できず、Application.Runは文字列ベースで型安全性がない。stdCallbackはこれらの課題を解決し、既存関数をオブジェクトとして扱うことで、コールバックパターンの実装を容易にする。

**機能の利用シーン**：
- クラスモジュール内のメソッドをコールバックとして渡す
- 標準モジュールの関数をコールバックとして渡す
- 外部ワークブックの関数を呼び出す
- プライベート関数をポインタ経由で呼び出す
- イベントハンドラの動的登録
- 部分適用（カリー化）による関数の事前設定

**主要な処理内容**：
1. モジュール関数からのコールバック生成（CreateFromModule）
2. オブジェクトメソッドからのコールバック生成（CreateFromObjectMethod）
3. オブジェクトプロパティからのコールバック生成（CreateFromObjectProperty）
4. 関数ポインタからのコールバック生成（CreateFromPointer）
5. 外部ワークブック関数からのコールバック生成（CreateFromWorkbookModule）
6. 引数のバインディング（Bind, BindEx）
7. コールバックの実行（Run, RunEx）

**関連システム・外部連携**：stdICallableインターフェースを実装し、stdArray、stdEnumerator等の高階関数メソッドと連携する。rtcCallByName API、DispCallFunc API、Application.Runを内部で使用。

**権限による制御**：特になし。ただし、CreateFromPointerはMac環境では非対応。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないライブラリクラス |

## 機能種別

コールバックオブジェクト / 関数ラッパー / 関数型プログラミング支援

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| sParentName | String | Yes | モジュール名（CreateFromModule時） | 空文字不可 |
| sMethodName | String | Yes | メソッド名 | 空文字不可 |
| object | Object | Yes | オブジェクト参照（CreateFromObjectMethod時） | Nothing不可 |
| cType | VbCallType | No | 呼び出し種別（デフォルトVbMethod） | - |
| MethodPointer | LongPtr | Yes | 関数ポインタ（CreateFromPointer時） | 0不可 |
| iRetType | VbVarType | No | 戻り値の型（CreateFromPointer時） | - |
| vParamTypes | Array(VbVarType) | No | 引数型配列（CreateFromPointer時） | - |
| sWorkbookPath | String | Yes | ワークブックパス（CreateFromWorkbookModule時） | 空文字不可 |

### 入力データソース

- 標準モジュールの関数
- クラスモジュールのメソッド/プロパティ
- 外部ワークブックの関数
- 関数ポインタ（AddressOf）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Run() | Variant | コールバック実行結果 |
| Bind() | stdCallback | 引数バインド済みの新しいコールバック |

### 出力先

- Runメソッドの戻り値（Variant型）
- stdICallableインターフェース経由での呼び出し結果

## 処理フロー

### 処理シーケンス

```
1. ファクトリメソッド呼び出し
   └─ CreateFromModule / CreateFromObjectMethod / CreateFromPointer等
   └─ protInit: 内部構造体に情報を格納

2. Bind（オプション）
   └─ 新規stdCallbackをBound型として作成
   └─ 元のコールバックと引数を保持

3. Run実行
   └─ BeforeRunイベント発火
   └─ ParentType判定
      ├─ Module: Application.Run呼び出し
      ├─ Object: rtcCallByName / CallByName呼び出し
      ├─ Bound: 元コールバックのRunEx呼び出し
      └─ Pointer: DispCallFunc呼び出し
   └─ AfterRunイベント発火
   └─ 結果返却
```

### フローチャート

```mermaid
flowchart TD
    A[ファクトリメソッド呼び出し] --> B{作成種別}
    B -->|Module| C[CreateFromModule]
    B -->|Object| D[CreateFromObjectMethod]
    B -->|Pointer| E[CreateFromPointer]
    B -->|Workbook| F[CreateFromWorkbookModule]

    C --> G[protInit]
    D --> G
    E --> G
    F --> G

    G --> H[CallbackStruct設定]
    H --> I[stdCallback返却]

    J[Run呼び出し] --> K{Bound?}
    K -->|Yes| L[引数結合]
    K -->|No| M[引数そのまま]
    L --> N[BeforeRunイベント]
    M --> N

    N --> O{ParentType}
    O -->|Module| P[Application.Run]
    O -->|Object| Q[rtcCallByName]
    O -->|Bound| R[元コールバック.RunEx]
    O -->|Pointer| S[DispCallFunc]

    P --> T[AfterRunイベント]
    Q --> T
    R --> T
    S --> T

    T --> U[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | Public関数のみ | CreateFromModule/CreateFromObjectMethodはPublic関数のみ対象 | Module/Object型 |
| BR-002 | Private関数対応 | CreateFromPointerはPrivate関数も呼び出し可能 | Pointer型 |
| BR-003 | 引数最大30個 | Application.Runの制限により引数は最大30個 | Module型 |
| BR-004 | Boundは左結合 | Bindされた引数は渡された引数の左側に配置 | Bound型 |
| BR-005 | Mac制限 | DispCallFuncはMac非対応 | Pointer型 |
| BR-006 | 初期化は1回のみ | protInitは1回のみ呼び出し可能 | 全型 |

### ParentType一覧

```
1: Module  - 標準モジュールの関数
2: Object  - オブジェクトのメソッド/プロパティ
3: Formula - 数式（未実装）
4: Bound   - バインド済みコールバック
5: Pointer - 関数ポインタ
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本クラスはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | 初期化エラー | 不正なモジュール種別 | CriticalRaise |
| 1 | 二重初期化 | 既に初期化済み | "Class is already initialised" |
| 1 | 未初期化 | 初期化前にRun呼び出し | "No callback specified" |
| 1 | 引数エラー | 引数が配列でない | "Arguments supplied needs to be an array" |
| 1 | ポインタ引数不一致 | 引数数と型定義数が不一致 | "Contradicting number of variables" |
| hResult | API呼び出しエラー | DispCallFunc失敗 | hResultをエラーコードとして発生 |
| Err.Number | rtcCallByName失敗 | オブジェクトメソッド呼び出し失敗 | Err.Descriptionを含めて発生 |

### リトライ仕様

リトライ仕様なし（同期処理のみ）

## トランザクション仕様

トランザクション管理なし（メモリ内操作のみ）

## パフォーマンス要件

- CreateFromModule/CreateFromObjectMethod: O(1)
- Run（Module型）: Application.Runのオーバーヘッド
- Run（Object型）: rtcCallByNameの直接呼び出しで最適化
- Run（Pointer型）: DispCallFuncによる直接呼び出し
- ConcatArrays: O(n)

## セキュリティ考慮事項

- 呼び出す関数はPublicである必要がある（Pointer型を除く）
- 外部ワークブック呼び出し時はパスの検証を推奨
- DispCallFuncによるポインタ呼び出しは慎重に使用

## 備考

- VB_PredeclaredId = True により、stdCallbackをインスタンス化せずにファクトリメソッドを呼び出し可能
- stdICallableインターフェースを実装し、他のstdVBAクラスと相互運用可能
- BeforeRun/AfterRunイベントにより、呼び出し前後の処理を挿入可能
- Run.VB_UserMemId = 0 により、デフォルトメンバーとして直接呼び出し可能（cb(args)形式）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、stdCallbackの内部データ構造（CallbackStruct、ParentType）を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdCallback.cls | `src/stdCallback.cls` | 132-138行目: ParentType enum定義 |
| 1-2 | stdCallback.cls | `src/stdCallback.cls` | 140-145行目: TCBByPointer型（ポインタ用） |
| 1-3 | stdCallback.cls | `src/stdCallback.cls` | 148-162行目: CallbackStruct, TThis型定義 |

**読解のコツ**:
- `ParentType`がコールバックの種別を示す
- `CallbackStruct`が実際のコールバック情報を保持
- `sModuleName`/`oObject`が呼び出し先
- `vBoundArgs`がバインド済み引数

#### Step 2: ファクトリメソッドを理解する

様々なソースからstdCallbackを生成するファクトリメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdCallback.cls | `src/stdCallback.cls` | 205-209行目: CreateFromModule |
| 2-2 | stdCallback.cls | `src/stdCallback.cls` | 217-221行目: CreateFromWorkbookModule |
| 2-3 | stdCallback.cls | `src/stdCallback.cls` | 228-232行目: CreateFromObjectMethod |
| 2-4 | stdCallback.cls | `src/stdCallback.cls` | 240-244行目: CreateFromObjectProperty |
| 2-5 | stdCallback.cls | `src/stdCallback.cls` | 252-259行目: CreateFromPointer |
| 2-6 | stdCallback.cls | `src/stdCallback.cls` | 265-307行目: protInit初期化処理 |

**主要処理フロー**:
1. **205行目**: `CreateFromModule` - モジュール名とメソッド名を指定
2. **217行目**: `CreateFromWorkbookModule` - 外部ワークブックパス付き
3. **228行目**: `CreateFromObjectMethod` - オブジェクト参照とメソッド名
4. **252行目**: `CreateFromPointer` - AddressOf結果を指定
5. **265-307行目**: `protInit` - ParentType別に内部構造体を設定

#### Step 3: 実行処理を理解する

Run/RunExメソッドによるコールバック実行を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdCallback.cls | `src/stdCallback.cls` | 367-371行目: Runメソッド |
| 3-2 | stdCallback.cls | `src/stdCallback.cls` | 376-522行目: RunExメソッド |

**主要処理フロー**:
- **377-380行目**: Bound型の場合は引数を結合
- **383行目**: BeforeRunイベント発火
- **388-425行目**: Module型 - Application.Run呼び出し
- **426-446行目**: Object型 - rtcCallByName呼び出し
- **447-450行目**: Bound型 - 元コールバックのRunEx呼び出し
- **451-505行目**: Pointer型 - DispCallFunc呼び出し
- **512行目**: AfterRunイベント発火

#### Step 4: バインディングを理解する

Bind/BindExによる部分適用を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdCallback.cls | `src/stdCallback.cls` | 333-335行目: Bindメソッド |
| 4-2 | stdCallback.cls | `src/stdCallback.cls` | 340-344行目: BindExメソッド |
| 4-3 | stdCallback.cls | `src/stdCallback.cls` | 616-640行目: ConcatArrays関数 |

**主要処理フロー**:
- **340-344行目**: 新規stdCallbackをBound型として生成
- **278-285行目**: protInitでBound型の初期化（oObjectに元コールバック、vBoundArgsにバインド引数）
- **616-640行目**: ConcatArraysでバインド引数と渡された引数を結合

### プログラム呼び出し階層図

```
stdCallback.CreateFromModule(sParentName, sMethodName)
    │
    └─ stdCallback.protInit(ParentType.Module, sParentName, sMethodName)
           └─ CallbackStruct.sModuleName = sParentName
           └─ CallbackStruct.sMethodName = sMethodName

stdCallback.CreateFromObjectMethod(object, sMethodName)
    │
    └─ stdCallback.protInit(ParentType.Object, object, sMethodName, VbMethod)
           └─ CallbackStruct.oObject = object
           └─ CallbackStruct.sMethodName = sMethodName
           └─ CallbackStruct.iCallType = VbMethod

stdCallback.Run(params)
    │
    └─ stdCallback.RunEx(params)
           │
           ├─ RaiseEvent BeforeRun(Me, vArr)
           │
           ├─ [ParentType.Module]
           │      └─ Application.Run(moduleName.methodName, args...)
           │
           ├─ [ParentType.Object]
           │      └─ rtcCallByName(vRet, oObject, sMethodName, iCallType, vArgs)
           │
           ├─ [ParentType.Bound]
           │      └─ oObject.RunEx(ConcatArrays(vBoundArgs, vArr))
           │
           ├─ [ParentType.Pointer]
           │      └─ DispCallFunc(0, iMethodAddress, CC_STDCALL, iReturnType, ...)
           │
           └─ RaiseEvent AfterRun(Me, vRet)

stdCallback.Bind(params)
    │
    └─ stdCallback.BindEx(params)
           └─ 新規stdCallback.protInit(ParentType.Bound, Me, params)
```

### データフロー図

```
[入力]                          [処理]                          [出力]

モジュール名 ─────────────▶ CreateFromModule ────────────▶ stdCallback
+ メソッド名                       │                          (Module型)
                                   └─▶ protInit設定

オブジェクト ─────────────▶ CreateFromObjectMethod ──────▶ stdCallback
+ メソッド名                       │                          (Object型)
                                   └─▶ protInit設定

関数ポインタ ─────────────▶ CreateFromPointer ───────────▶ stdCallback
+ 型情報                           │                          (Pointer型)
                                   └─▶ protInit設定

stdCallback ──────────────▶ Bind(args) ──────────────────▶ 新stdCallback
+ バインド引数                     │                          (Bound型)
                                   └─▶ 元CB + 引数を保持

stdCallback ──────────────▶ Run(args) ───────────────────▶ 実行結果
+ 引数                             │
                                   ├─▶ BeforeRunイベント
                                   ├─▶ 実際の関数呼び出し
                                   └─▶ AfterRunイベント
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdCallback.cls | `src/stdCallback.cls` | ソース | コールバックオブジェクト生成クラスの実装 |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース |
| stdLambda.cls | `src/stdLambda.cls` | ソース | ラムダ式生成クラス（代替手段） |
| stdError.cls | `src/stdError.cls` | ソース | エラーハンドリングクラス（存在時使用） |
