# 機能設計書 5-stdICallable

## 概要

本ドキュメントは、stdVBAライブラリにおける呼び出し可能インターフェース `stdICallable` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdICallableは、VBAにおいて「呼び出し可能なオブジェクト」の共通インターフェースを定義するクラスである。このインターフェースを実装することで、stdLambda、stdCallback、およびユーザー定義クラスを、stdArrayやstdEnumeratorの高階関数メソッド（Map、Filter、Reduce等）から統一的に呼び出すことが可能になる。

**業務上の目的・背景**：VBAには関数を引数として渡すための標準的な仕組みがない。stdICallableはこの課題を解決し、Javaの`Callable`インターフェースやC#の`Func`デリゲートに相当する抽象化を提供する。これにより、異なる実装（ラムダ式、コールバック、カスタムクラス）を同一のインターフェースで扱うことができる。

**機能の利用シーン**：
- stdArrayのMap、Filter、Reduce、Sort等でのコールバック指定
- stdEnumeratorの高階関数メソッドでのコールバック指定
- stdWindowのFindFirst等での検索条件指定
- ユーザー定義クラスでの呼び出し可能オブジェクト実装
- 複数の呼び出し可能オブジェクトを統一的に扱うコレクション

**主要な処理内容**：
1. Run - 可変引数での関数実行
2. RunEx - 配列形式の引数での関数実行
3. Bind - 引数の部分適用（カリー化）
4. SendMessage - 遅延バインド呼び出しによるメタ情報取得

**関連システム・外部連携**：stdLambda、stdCallbackが本インターフェースを実装。stdArray、stdEnumerator等が本インターフェースを引数として受け取る。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないインターフェースクラス |

## 機能種別

インターフェース / 抽象化 / 関数型プログラミング支援

## 入力仕様

### インターフェースメソッド

| メソッド名 | パラメータ | 戻り値 | 説明 |
|-----------|-----------|--------|------|
| Run | ParamArray params() | Variant | 可変引数で関数を実行 |
| RunEx | params As Variant | Variant | 配列形式の引数で関数を実行 |
| Bind | ParamArray params() | stdICallable | 引数をバインドした新しいCallableを返す |
| SendMessage | sMessage, success, params | Variant | メタ情報の取得や遅延バインド呼び出し |

### SendMessageの標準メッセージ

| メッセージ | 説明 | 戻り値 |
|-----------|------|--------|
| "obj" | 実装オブジェクト自身を返す | Object |
| "className" | クラス名を返す | String |
| "bindGlobal" | グローバル変数をバインド（stdLambda） | - |
| "equation" | 式文字列を返す（stdLambda） | String |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Run戻り値 | Variant | 関数実行結果 |
| RunEx戻り値 | Variant | 関数実行結果 |
| Bind戻り値 | stdICallable | バインド済みの新しいCallable |
| SendMessage戻り値 | Variant | メッセージに応じた値 |

### 出力先

- 各メソッドの戻り値

## 処理フロー

### インターフェース実装パターン

```
1. クラスでの実装宣言
   └─ Implements stdICallable

2. Run メソッドの実装
   └─ 通常はRunExに委譲
   └─ stdICallable_Run(ParamArray params()) As Variant
      └─ Call CopyVariant(stdICallable_Run, RunEx(params))

3. RunEx メソッドの実装
   └─ 配列形式の引数を処理
   └─ stdICallable_RunEx(ByVal params As Variant) As Variant
      └─ 実際の処理ロジック

4. Bind メソッドの実装
   └─ 新しいCallableオブジェクトを返す
   └─ stdICallable_Bind(ParamArray params()) As stdICallable
      └─ Set stdICallable_Bind = BindEx(params)

5. SendMessage メソッドの実装
   └─ メッセージに応じた処理
   └─ stdICallable_SendMessage(sMessage, success, params) As Variant
      └─ Select Case sMessage
         └─ "obj", "className" 等
```

### フローチャート

```mermaid
flowchart TD
    A[stdICallable使用側] --> B{メソッド呼び出し}
    B -->|Run| C[可変引数処理]
    B -->|RunEx| D[配列引数処理]
    B -->|Bind| E[部分適用]
    B -->|SendMessage| F[メタ情報取得]

    C --> G[実装クラスのRun]
    D --> H[実装クラスのRunEx]
    E --> I[新しいstdICallable返却]
    F --> J[メッセージ処理]

    G --> K[結果返却]
    H --> K
    I --> L[バインド済みCallable]
    J --> M[メタ情報返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | Run/RunEx互換性 | Runは内部でRunExに委譲することを推奨 | 全実装 |
| BR-002 | Bind戻り値 | BindはstdICallableを実装したオブジェクトを返す | 全実装 |
| BR-003 | SendMessage失敗 | 未対応メッセージではsuccess=Falseを返す | 全実装 |
| BR-004 | 標準メッセージ | "obj"と"className"は全実装でサポート推奨 | 全実装 |

### 実装クラス一覧

| クラス | 説明 |
|--------|------|
| stdLambda | 文字列式からのラムダ関数 |
| stdCallback | 既存関数への参照ラッパー |
| ユーザー定義クラス | Implements stdICallableで独自実装 |

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本インターフェースはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | 未実装エラー | Bindが未実装の場合 | Err.Raise |
| 1 | 未実装エラー | SendMessageが未実装の場合 | Err.Raise |
| - | 実装依存 | 各実装クラスで定義 | 実装に依存 |

### リトライ仕様

リトライ仕様なし

## トランザクション仕様

トランザクション管理なし

## パフォーマンス要件

- インターフェース呼び出しオーバーヘッド: 最小限（VBAのインターフェース機構による）
- 実際のパフォーマンスは実装クラスに依存

## セキュリティ考慮事項

- インターフェースのみでは特になし
- 実装クラスのセキュリティ考慮事項に従う

## 備考

- VB_PredeclaredId = False（インターフェースのため）
- VB_Exposed = False（ライブラリ内部で使用）
- ToPointerメソッドは将来の拡張として検討中（コメントアウト状態）
- 関数ポインタへの変換は技術的課題により未実装

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: インターフェース定義を理解する

まず、stdICallableのインターフェース定義を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdICallable.cls | `src/stdICallable.cls` | 53行目: Runメソッド宣言 |
| 1-2 | stdICallable.cls | `src/stdICallable.cls` | 58行目: RunExメソッド宣言 |
| 1-3 | stdICallable.cls | `src/stdICallable.cls` | 63行目: Bindメソッド宣言 |
| 1-4 | stdICallable.cls | `src/stdICallable.cls` | 72行目: SendMessageメソッド宣言 |

**読解のコツ**:
- `Run`は可変引数（ParamArray）で呼び出し
- `RunEx`は配列形式で呼び出し（内部処理に便利）
- `Bind`は部分適用のため新しいstdICallableを返す
- `SendMessage`は遅延バインドによるメタ情報取得

#### Step 2: 実装例を理解する

コメント内の実装例を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdICallable.cls | `src/stdICallable.cls` | 14-30行目: クラス実装例（コメント） |
| 2-2 | stdICallable.cls | `src/stdICallable.cls` | 34-37行目: 使用例（コメント） |
| 2-3 | stdICallable.cls | `src/stdICallable.cls` | 39-43行目: stdArrayとの連携例（コメント） |

**主要処理フロー**:
1. **17-20行目**: stdICallable_RunはRunExに委譲
2. **21-23行目**: stdICallable_RunExで実際の処理
3. **24-26行目**: stdICallable_Bindは未実装でエラー（必要に応じて実装）
4. **27-29行目**: stdICallable_SendMessageは未実装でエラー（必要に応じて実装）

#### Step 3: 実際の実装クラスを参照する

stdLambdaとstdCallbackでの実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdLambda.cls | `src/stdLambda.cls` | 70行目: Implements stdICallable |
| 3-2 | stdLambda.cls | `src/stdLambda.cls` | 376-383行目: stdICallable_Run実装 |
| 3-3 | stdCallback.cls | `src/stdCallback.cls` | 11行目: Implements stdICallable |
| 3-4 | stdCallback.cls | `src/stdCallback.cls` | 312-314行目: stdICallable_Run実装 |

### プログラム呼び出し階層図

```
stdArray.Map(cb As stdICallable)
    │
    └─ cb.Run(element) または cb.RunEx(Array(element))
           │
           ├─ [stdLambda]
           │      └─ evaluate(operations, params)
           │
           ├─ [stdCallback]
           │      └─ RunEx → Application.Run / rtcCallByName
           │
           └─ [ユーザー定義クラス]
                  └─ カスタム実装

stdICallable.Bind(args)
    │
    └─ 新規stdICallable返却
           │
           ├─ [stdLambda]
           │      └─ BindEx → BoundLambda生成
           │
           └─ [stdCallback]
                  └─ BindEx → Bound型コールバック生成

stdICallable.SendMessage("className", success, Null)
    │
    └─ Select Case sMessage
           ├─ "obj" → 自身を返却
           ├─ "className" → クラス名文字列
           └─ その他 → 実装依存
```

### データフロー図

```
[入力]                      [処理]                          [出力]

stdArray.Map ─────────▶ stdICallable.Run ────────────▶ 変換結果
+ element                    │
                             └─▶ 実装クラスの処理

引数配列 ────────────▶ stdICallable.RunEx ──────────▶ 実行結果
                             │
                             └─▶ 実装クラスの処理

バインド引数 ────────▶ stdICallable.Bind ───────────▶ 新stdICallable
                             │
                             └─▶ 部分適用されたCallable

メッセージ ──────────▶ stdICallable.SendMessage ───▶ メタ情報
+ params                     │
                             └─▶ "obj", "className"等
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdICallable.cls | `src/stdICallable.cls` | ソース | 呼び出し可能インターフェース定義 |
| stdLambda.cls | `src/stdLambda.cls` | ソース | インターフェース実装（ラムダ関数） |
| stdCallback.cls | `src/stdCallback.cls` | ソース | インターフェース実装（コールバック） |
| stdArray.cls | `src/stdArray.cls` | ソース | インターフェース使用側（Map等） |
| stdEnumerator.cls | `src/stdEnumerator.cls` | ソース | インターフェース使用側（Map等） |
