# 機能設計書 6-stdRefArray

## 概要

本ドキュメントは、stdVBAライブラリにおけるVBA配列の参照渡しラッパークラス `stdRefArray` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdRefArrayは、VBA配列を参照渡しでラップするクラスである。VBAでは配列を関数に渡す際にコピーが発生するため、大規模な配列を扱う場合にパフォーマンスが低下する。stdRefArrayはこの課題を解決し、配列のポインタを保持することで、コピーなしに配列データへのアクセスを可能にする。

**業務上の目的・背景**：VBAでは、Variant型に格納された配列を関数に渡すと、配列全体がコピーされる。これは小さな配列では問題にならないが、数万行のExcelデータを扱う場合など、大規模な配列ではメモリと処理時間の両面で大きなオーバーヘッドとなる。stdRefArrayはこの問題を解決し、大規模データの効率的な受け渡しを可能にする。

**機能の利用シーン**：
- 大規模なExcelデータの関数間受け渡し
- メモリ効率が重要な処理でのデータ共有
- 複数の関数で同一配列を参照する処理
- 配列データの変更を呼び出し元に反映させる処理

**主要な処理内容**：
1. VBA配列のポインタ取得（VarPtrArr）
2. Variant型へのポインタ格納とVT_BYREFフラグ設定（protInit）
3. 参照経由での配列データアクセス（Dataプロパティ）
4. CopyMemoryによる低レベルメモリ操作

**関連システム・外部連携**：kernel32.dll（Windows）またはlibc.dylib（Mac）のRtlMoveMemory/memmoveを使用。

**権限による制御**：特になし。ただし、低レベルAPI呼び出しのためVBAのセキュリティ設定に依存。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないライブラリクラス |

## 機能種別

データ構造（参照配列） / パフォーマンス最適化 / 低レベルメモリ操作

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| Data | Variant(Array) | Yes | ラップする配列 | IsArray = True |

### 入力データソース

- VBA配列（Variant配列、型付き配列）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Data | Variant | 参照経由でアクセスされる配列データ |
| GetData | Variant | デフォルトメンバーとしての配列データ |

### 出力先

- Dataプロパティ経由での配列アクセス
- GetDataプロパティ（デフォルトメンバー）

## 処理フロー

### 処理シーケンス

```
1. Create(arr)
   └─ 新規stdRefArrayインスタンス生成
   └─ protInit(arr)呼び出し

2. protInit(arr)
   └─ IsArray(arr)チェック
      └─ False: エラー発生（"Array required"）
      └─ True: 続行
   └─ VarPtrArr(arr)でポインタ取得
   └─ Dataにポインタを格納
   └─ VT_BYREFフラグを設定（VarType Or VT_BYREF）
   └─ CopyMemoryでVariant型ヘッダーを更新

3. VarPtrArr(arr)
   └─ 配列のVariant型チェック（vbArray + VT_BYREF）
   └─ VarPtr + オフセット(8)でポインタ取得
   └─ CopyMemoryでポインタ値をコピー

4. Data参照時
   └─ VBAがVT_BYREFフラグを解釈
   └─ 自動的にポインタを逆参照して配列にアクセス
```

### フローチャート

```mermaid
flowchart TD
    A[Create呼び出し] --> B[新規stdRefArray生成]
    B --> C[protInit呼び出し]
    C --> D{IsArray?}
    D -->|No| E[エラー発生]
    D -->|Yes| F[VarPtrArr呼び出し]
    F --> G[配列ポインタ取得]
    G --> H[Data = ポインタ]
    H --> I[VT_BYREFフラグ設定]
    I --> J[CopyMemoryでヘッダー更新]
    J --> K[stdRefArray返却]

    L[Data参照] --> M[VBAがVT_BYREF解釈]
    M --> N[ポインタ逆参照]
    N --> O[元配列データにアクセス]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 配列のみ | 配列以外の入力はエラー | protInit時 |
| BR-002 | 参照維持 | 元配列が存在する間のみ有効 | 使用中全期間 |
| BR-003 | 変更反映 | 参照経由の変更は元配列に反映 | Data経由アクセス時 |
| BR-004 | プラットフォーム対応 | Win32/Win64/Macで異なるポインタサイズ | 初期化時 |

### メモリレイアウト

**Variant型構造（Win64）**:
```
Offset 0-1:   VarType (2 bytes)
Offset 2-7:   Reserved (6 bytes)
Offset 8-15:  Data pointer (8 bytes)
Offset 16-23: Reserved (8 bytes)
Total: 24 bytes
```

**Variant型構造（Win32）**:
```
Offset 0-1:   VarType (2 bytes)
Offset 2-7:   Reserved (6 bytes)
Offset 8-11:  Data pointer (4 bytes)
Offset 12-15: Reserved (4 bytes)
Total: 16 bytes
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本クラスはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 5 | 引数エラー | 配列以外がprotInitに渡された | "Array required" |
| 5 | 引数エラー | VarPtrArrで配列以外が渡された | "Array required" |

### リトライ仕様

リトライ仕様なし

## トランザクション仕様

トランザクション管理なし（メモリ内操作のみ）

## パフォーマンス要件

- Create: O(1)（ポインタ取得のみ、コピーなし）
- Data参照: O(1)（ポインタ逆参照のみ）
- メモリ使用量: 元配列 + ポインタサイズ（8/4 bytes）のみ

**比較**:
| 操作 | 通常の配列渡し | stdRefArray |
|------|---------------|-------------|
| メモリ | 配列サイズ × 2 | 配列サイズ + 数bytes |
| 処理時間 | O(n)（コピー） | O(1) |

## セキュリティ考慮事項

- CopyMemoryは低レベルAPI呼び出しのため、ポインタ操作に注意
- 元配列が解放された後の参照は未定義動作
- 不正なポインタによるメモリ破壊のリスク

## 備考

- VB_PredeclaredId = True により、stdRefArrayをインスタンス化せずにファクトリメソッドを呼び出し可能
- GetData.VB_UserMemId = 0 により、デフォルトメンバーとして直接アクセス可能
- Win32/Win64/Macの各プラットフォームに対応
- VT_BYREF（&H4000）フラグによりVBAが自動的にポインタ逆参照を行う

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: API宣言とデータ構造を理解する

まず、CopyMemory APIとポインタサイズの定義を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdRefArray.cls | `src/stdRefArray.cls` | 24-38行目: CopyMemory API宣言（Mac/Windows） |
| 1-2 | stdRefArray.cls | `src/stdRefArray.cls` | 40-45行目: VT_BYREF定数とPTR_SIZE定義 |
| 1-3 | stdRefArray.cls | `src/stdRefArray.cls` | 49行目: Public Data As Variant |

**読解のコツ**:
- `CopyMemory`はWindows/Macで異なるライブラリを使用
- `VT_BYREF`（&H4000）は参照渡しを示すフラグ
- `PTR_SIZE`はWin64で8バイト、Win32で4バイト
- `Data`はPublicプロパティとして公開

#### Step 2: ファクトリメソッドを理解する

Create関数とprotInit関数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdRefArray.cls | `src/stdRefArray.cls` | 63-66行目: Createメソッド |
| 2-2 | stdRefArray.cls | `src/stdRefArray.cls` | 71-82行目: protInit初期化処理 |

**主要処理フロー**:
1. **63行目**: `Create(ByRef Data)` - 配列を参照で受け取る
2. **64行目**: 新規インスタンス生成
3. **65行目**: `protInit`呼び出し
4. **72行目**: `IsArray`チェック
5. **74行目**: `VarPtrArr`でポインタ取得
6. **77行目**: VT_BYREFフラグ計算
7. **78行目**: `CopyMemory`でVariantヘッダー更新

#### Step 3: ポインタ取得を理解する

VarPtrArr関数の内部処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdRefArray.cls | `src/stdRefArray.cls` | 87-101行目: VarPtrArr関数 |

**主要処理フロー**:
- **92行目**: `vtArrByRef = vbArray + VT_BYREF`定数
- **94行目**: `CopyMemory`でVarTypeを取得
- **95行目**: 配列かつByRefかチェック
- **97行目**: `VarPtr(arr) + 8`でデータポインタの位置を計算
- **97行目**: `CopyMemory`でポインタ値をコピー

#### Step 4: GetDataプロパティを理解する

デフォルトメンバーの実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdRefArray.cls | `src/stdRefArray.cls` | 51-58行目: GetDataプロパティ |

**主要処理フロー**:
- **52行目**: `VB_UserMemId = 0`でデフォルトメンバー指定
- **53-57行目**: Win64/Win32でサイズが異なる（24/16バイト）
- `CopyMemory`でVariant全体をコピー

### プログラム呼び出し階層図

```
stdRefArray.Create(arr)
    │
    └─ stdRefArray.protInit(arr)
           │
           ├─ IsArray(arr) チェック
           │
           ├─ VarPtrArr(arr)
           │      │
           │      ├─ CopyMemory(vt, arr, 2)  -- VarType取得
           │      │
           │      └─ CopyMemory(VarPtrArr, VarPtr(arr)+8, PTR_SIZE)  -- ポインタ取得
           │
           ├─ Data = ポインタ値
           │
           └─ CopyMemory(Data, vt Or VT_BYREF, 2)  -- ヘッダー更新

stdRefArray.Data 参照
    │
    └─ VBAランタイム
           │
           └─ VT_BYREFフラグ検出
                  │
                  └─ ポインタ逆参照 → 元配列データ
```

### データフロー図

```
[入力]                      [処理]                          [出力]

VBA配列 ─────────────▶ Create ───────────────────────▶ stdRefArray
(ByRef)                      │
                             └─▶ protInit
                                    │
                                    ├─▶ VarPtrArr → ポインタ取得
                                    │
                                    └─▶ Data = ポインタ + VT_BYREF

stdRefArray.Data ────▶ VBAランタイム ────────────────▶ 元配列データ
                             │                          (コピーなし)
                             └─▶ VT_BYREF解釈
                                    │
                                    └─▶ ポインタ逆参照
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdRefArray.cls | `src/stdRefArray.cls` | ソース | VBA配列の参照渡しラッパークラス |
